@extends('layouts.whatsapp-layout')

@section('content')
<div class="container-fluid">
    <div class="row mb-4">
        <div class="col-12">
            <h1 class="h3 mb-0 text-gray-800">WhatsApp Multi-Cliente Dashboard</h1>
            <p class="text-muted">Gestiona múltiples sesiones de WhatsApp para diferentes empresas</p>
        </div>
    </div>

    <!-- Estadísticas -->
    <div class="row mb-4">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Total de Sesiones
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800" id="total-sessions">0</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-comments fa-2x text-gray-300"></iconify-icon>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Sesiones Activas
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800" id="active-sessions">0</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-check-circle fa-2x text-gray-300"></iconify-icon>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                Esperando QR
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800" id="qr-sessions">0</div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-qrcode fa-2x text-gray-300"></iconify-icon>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-info shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                Estado del Servidor
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800" id="server-status">
                                <span class="badge badge-success">Conectado</span>
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-server fa-2x text-gray-300"></iconify-icon>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Formulario para crear nueva sesión -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Crear Nueva Sesión</h6>
                </div>
                <div class="card-body">
                    <form id="create-session-form" class="needs-validation" novalidate>
                        <div class="row">
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="session-id">ID de Sesión *</label>
                                    <input type="text" class="form-control" id="session-id" name="session_id" required
                                           placeholder="Ej: empresa-001">
                                    <small class="form-text text-muted">Identificador único para esta sesión</small>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="empresa-codigo">Código de Empresa</label>
                                    <input type="text" class="form-control" id="empresa-codigo" name="empresa_codigo"
                                           placeholder="Ej: EMP001">
                                    <small class="form-text text-muted">Código de la empresa asociada</small>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label for="empresa-nombre">Nombre de Empresa</label>
                                    <input type="text" class="form-control" id="empresa-nombre" name="empresa_nombre"
                                           placeholder="Ej: Mi Empresa S.A.">
                                    <small class="form-text text-muted">Nombre completo de la empresa</small>
                                </div>
                            </div>
                        </div>
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-plus"></iconify-icon> Crear Sesión
                        </button>
                    </form>
                </div>
            </div>
        </div>
    </div>

    <!-- Lista de sesiones -->
    <div class="row">
        <div class="col-12">
            <div class="card shadow">
                <div class="card-header py-3 d-flex justify-content-between align-items-center">
                    <h6 class="m-0 font-weight-bold text-primary">Sesiones Activas</h6>
                    <button class="btn btn-sm btn-outline-primary" onclick="loadDashboard()">
                        <i class="fas fa-sync-alt"></iconify-icon> Actualizar
                    </button>
                </div>
                <div class="card-body">
                    <div id="sessions-container" class="row">
                        <!-- Las sesiones se cargarán dinámicamente aquí -->
                    </div>
                    <div id="loading-message" class="text-center py-4">
                        <div class="spinner-border text-primary" role="status">
                            <span class="sr-only">Cargando...</span>
                        </div>
                        <p class="mt-2">Cargando sesiones...</p>
                    </div>
                    <div id="no-sessions-message" class="text-center py-4" style="display: none;">
                        <i class="fas fa-inbox fa-3x text-gray-300 mb-3"></iconify-icon>
                        <p class="text-muted">No hay sesiones activas</p>
                        <p class="text-muted">Crea una nueva sesión para comenzar</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

@push('scripts')
<script>
// Dashboard Multi-WhatsApp Functions
let sessions = [];
let refreshInterval;

// Cargar dashboard inicial
async function loadDashboard() {
    try {
        showLoading(true);

        const response = await fetch('{{ route("whatsapp.dashboard.data") }}');
        const data = await response.json();

        if (response.ok) {
            // Actualizar estadísticas
            document.getElementById('total-sessions').textContent = data.totalSessions || 0;
            document.getElementById('active-sessions').textContent = data.activeSessions || 0;

            // Contar sesiones esperando QR
            const qrSessions = data.sessions.filter(s => s.hasQR).length;
            document.getElementById('qr-sessions').textContent = qrSessions;

            // Actualizar sesiones
            sessions = data.sessions || [];
            renderSessions();
        } else {
            showError('Error: ' + (data.message || 'Error desconocido'));
        }
    } catch (error) {
        console.error('Error cargando dashboard:', error);
        showError('Error de conexión con el servidor WhatsApp');
        updateServerStatus(false);
    } finally {
        showLoading(false);
    }
}

// Mostrar/ocultar loading
function showLoading(show) {
    const loadingEl = document.getElementById('loading-message');
    const noSessionsEl = document.getElementById('no-sessions-message');
    const containerEl = document.getElementById('sessions-container');

    if (show) {
        loadingEl.style.display = 'block';
        noSessionsEl.style.display = 'none';
        containerEl.innerHTML = '';
    } else {
        loadingEl.style.display = 'none';
    }
}

// Mostrar error
function showError(message) {
    const container = document.getElementById('sessions-container');
    container.innerHTML = `
        <div class="col-12">
            <div class="alert alert-danger alert-dismissible fade show" role="alert">
                <i class="fas fa-exclamation-triangle"></iconify-icon> ${message}
                <button type="button" class="close" data-dismiss="alert" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
            </div>
        </div>
    `;
}

// Actualizar estado del servidor
function updateServerStatus(connected) {
    const statusEl = document.getElementById('server-status');
    if (connected) {
        statusEl.innerHTML = '<span class="badge badge-success">Conectado</span>';
    } else {
        statusEl.innerHTML = '<span class="badge badge-danger">Desconectado</span>';
    }
}

// Renderizar sesiones
function renderSessions() {
    const container = document.getElementById('sessions-container');
    const noSessionsEl = document.getElementById('no-sessions-message');

    container.innerHTML = '';

    if (sessions.length === 0) {
        noSessionsEl.style.display = 'block';
        return;
    }

    noSessionsEl.style.display = 'none';

    sessions.forEach(session => {
        const sessionCard = createSessionCard(session);
        container.appendChild(sessionCard);
    });
}

// Crear tarjeta de sesión
function createSessionCard(session) {
    const card = document.createElement('div');
    card.className = 'col-lg-4 col-md-6 mb-4';

    let statusClass = '';
    let statusText = '';
    let statusBadge = '';

    switch(session.status) {
        case 'connected':
        case 'ready':
            statusClass = 'border-left-success';
            statusText = 'Conectado';
            statusBadge = 'badge-success';
            break;
        case 'qr_required':
        case 'qr':
            statusClass = 'border-left-warning';
            statusText = 'QR Requerido';
            statusBadge = 'badge-warning';
            break;
        case 'authenticated':
            statusClass = 'border-left-info';
            statusText = 'Autenticado';
            statusBadge = 'badge-info';
            break;
        default:
            statusClass = 'border-left-secondary';
            statusText = 'Desconectado';
            statusBadge = 'badge-secondary';
    }

    let qrHtml = '';
    if (session.hasQR && session.qr) {
        qrHtml = `
            <div class="text-center mb-3">
                <img src="data:image/png;base64,${session.qr}" alt="QR Code" class="img-fluid" style="max-width: 200px;">
                <p class="text-muted small mt-2">Escanea este código QR con WhatsApp</p>
            </div>
        `;
    }

    card.innerHTML = `
        <div class="card ${statusClass} shadow h-100">
            <div class="card-header py-3">
                <div class="d-flex justify-content-between align-items-center">
                    <h6 class="m-0 font-weight-bold text-primary">${session.empresaNombre || session.sessionId}</h6>
                    <span class="badge ${statusBadge}">${statusText}</span>
                </div>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <strong>ID de Sesión:</strong> ${session.sessionId}<br>
                    <strong>Código Empresa:</strong> ${session.empresaCodigo || 'N/A'}<br>
                    <strong>Creado:</strong> ${new Date(session.createdAt).toLocaleString()}
                </div>
                ${qrHtml}
                <div class="d-flex justify-content-between">
                    <button class="btn btn-sm btn-primary" onclick="refreshSession('${session.sessionId}')">
                        <i class="fas fa-sync-alt"></iconify-icon> Actualizar
                    </button>
                    <button class="btn btn-sm btn-danger" onclick="closeSession('${session.sessionId}')">
                        <i class="fas fa-sign-out-alt"></iconify-icon> Cerrar
                    </button>
                </div>
            </div>
        </div>
    `;

    return card;
}

// Crear nueva sesión
document.getElementById('create-session-form').addEventListener('submit', async (e) => {
    e.preventDefault();

    const formData = new FormData(e.target);
    const data = {
        session_id: formData.get('session_id'),
        empresa_codigo: formData.get('empresa_codigo'),
        empresa_nombre: formData.get('empresa_nombre')
    };

    // Validar datos
    if (!data.session_id) {
        alert('El ID de sesión es requerido');
        return;
    }

    try {
        const submitBtn = e.target.querySelector('button[type="submit"]');
        submitBtn.disabled = true;
        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></iconify-icon> Creando...';

        const response = await fetch('{{ route("whatsapp.dashboard.create-session") }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            },
            body: JSON.stringify(data)
        });

        const result = await response.json();

        if (response.ok) {
            alert('Sesión creada exitosamente');
            e.target.reset();
            loadDashboard();
        } else {
            alert('Error: ' + (result.message || result.error || 'Error desconocido'));
        }
    } catch (error) {
        console.error('Error completo:', error);
        alert('Error creando sesión: ' + error.message);
    } finally {
        const submitBtn = e.target.querySelector('button[type="submit"]');
        submitBtn.disabled = false;
        submitBtn.innerHTML = '<i class="fas fa-plus"></iconify-icon> Crear Sesión';
    }
});

// Actualizar sesión específica
async function refreshSession(sessionId) {
    try {
        const response = await fetch(`/whatsapp/session/${sessionId}/status`);
        const status = await response.json();

        if (response.ok) {
            // Actualizar la sesión en el array
            const sessionIndex = sessions.findIndex(s => s.sessionId === sessionId);
            if (sessionIndex !== -1) {
                sessions[sessionIndex].status = status.status;
                sessions[sessionIndex].isReady = status.isReady;
                sessions[sessionIndex].hasQR = !!status.qr;
                sessions[sessionIndex].qr = status.qr;
            }

            renderSessions();
            alert('Sesión actualizada');
        } else {
            alert('Error: ' + (status.message || 'Error desconocido'));
        }
    } catch (error) {
        alert('Error actualizando sesión: ' + error.message);
    }
}

// Cerrar sesión
async function closeSession(sessionId) {
    if (!confirm('¿Estás seguro de que quieres cerrar esta sesión?')) {
        return;
    }

    try {
        const response = await fetch(`/whatsapp/session/${sessionId}/close`, {
            method: 'POST',
            headers: {
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        });

        const result = await response.json();

        if (response.ok) {
            alert('Sesión cerrada exitosamente');
            loadDashboard();
        } else {
            alert('Error: ' + (result.message || result.error || 'Error desconocido'));
        }
    } catch (error) {
        alert('Error cerrando sesión: ' + error.message);
    }
}

// Auto-refresh cada 10 segundos
function startAutoRefresh() {
    refreshInterval = setInterval(loadDashboard, 10000);
}

function stopAutoRefresh() {
    if (refreshInterval) {
        clearInterval(refreshInterval);
    }
}

// Inicializar
document.addEventListener('DOMContentLoaded', () => {
    console.log('Dashboard de WhatsApp inicializándose...');
    updateServerStatus(true);
    loadDashboard();
    startAutoRefresh();
});

// Detener auto-refresh cuando la página se cierra
window.addEventListener('beforeunload', stopAutoRefresh);
</script>
@endpush
@endsection
