{{-- Scripts para manejo de hardware POS --}}
<script>
// Configuración de hardware POS
const POSHardware = {
    barcodeBuffer: '',
    barcodeTimer: null,
    barcodeTimeout: 50, // ms entre caracteres

    // Inicializar escáner de código de barras
    initBarcodeScanner() {
        document.addEventListener('keydown', (e) => {
            // Si estamos en un input o textarea, no interceptar
            if (e.target.tagName === 'INPUT' || e.target.tagName === 'TEXTAREA') {
                return;
            }

            // Si es un carácter imprimible y no es una tecla especial
            if (e.key && e.key.length === 1 && !e.ctrlKey && !e.altKey && !e.metaKey) {
                e.preventDefault();
                this.barcodeBuffer += e.key;

                // Resetear timer
                if (this.barcodeTimer) {
                    clearTimeout(this.barcodeTimer);
                }

                // Si es Enter, procesar el código
                if (e.key === 'Enter') {
                    this.processBarcode();
                    return;
                }

                // Timer para detectar fin de escaneo
                this.barcodeTimer = setTimeout(() => {
                    this.processBarcode();
                }, this.barcodeTimeout);
            }
        });
    },

    processBarcode() {
        if (this.barcodeBuffer.trim()) {
            // Enviar el código a Livewire
            Livewire.dispatch('barcodeScanned', { barcode: this.barcodeBuffer.trim() });

            // Mostrar feedback visual
            this.showBarcodeFeedback();
        }

        // Limpiar buffer
        this.barcodeBuffer = '';
        if (this.barcodeTimer) {
            clearTimeout(this.barcodeTimer);
            this.barcodeTimer = null;
        }
    },

    showBarcodeFeedback() {
        // Crear un pequeño toast visual para feedback del escaneo
        const toast = document.createElement('div');
        toast.className = 'position-fixed top-0 start-50 translate-middle-x mt-3 alert alert-success alert-sm';
        toast.style.zIndex = '9999';
        toast.innerHTML = '<i class="ri-barcode-line me-2"></i>Código escaneado';
        document.body.appendChild(toast);

        setTimeout(() => {
            toast.remove();
        }, 1000);
    },

    // Inicializar impresora de tickets
    initPrinter() {
        Livewire.on('print-ticket', (event) => {
            const data = event[0];
            this.printTicket(data);
        });
    },

    printTicket(data) {
        // Verificar si hay impresora configurada
        if (!this.checkPrinterSupport()) {
            console.warn('Impresora no soportada o no configurada');
            return;
        }

        // Generar contenido del ticket
        const ticketContent = this.generateTicketContent(data);

        // Intentar imprimir usando diferentes métodos
        this.tryPrint(ticketContent);
    },

    checkPrinterSupport() {
        // Verificar soporte para impresoras térmicas o de tickets
        return window.electronAPI || window.print || window.navigator.usb;
    },

    generateTicketContent(data) {
        const fecha = new Date().toLocaleString();
        const empresa = data.empresa || 'Mi Empresa';
        const direccion = data.direccion || '';
        const telefono = data.telefono || '';
        const rif = data.rif || '';

        let content = '';
        content += '================================\n';
        content += `${empresa.toUpperCase()}\n`;
        if (rif) content += `RIF: ${rif}\n`;
        if (direccion) content += `${direccion}\n`;
        if (telefono) content += `Tel: ${telefono}\n`;
        content += '================================\n';
        content += `Fecha: ${fecha}\n`;
        content += `Ticket: ${data.ventaId || 'N/A'}\n`;
        content += '================================\n';
        content += 'CANT  DESCRIPCION        IMPORTE\n';
        content += '================================\n';

        // Aquí irían los productos del carrito
        // Por ahora dejamos espacio para ser llenado por Livewire

        content += '================================\n';
        content += 'GRACIAS POR SU COMPRA\n';
        content += '================================\n\n\n';

        return content;
    },

    tryPrint(content) {
        // Método 1: Usar API de Electron si está disponible
        if (window.electronAPI && window.electronAPI.print) {
            window.electronAPI.print(content);
            return;
        }

        // Método 2: Usar WebUSB para impresoras térmicas
        if (window.navigator.usb) {
            this.printViaUSB(content);
            return;
        }

        // Método 3: Ventana emergente de impresión (fallback)
        this.printViaWindow(content);
    },

    printViaWindow(content) {
        const printWindow = window.open('', '_blank', 'width=300,height=500');
        printWindow.document.write(`
            <html>
                <head>
                    <title>Ticket de Venta</title>
                    <style>
                        body { font-family: monospace; font-size: 12px; margin: 0; padding: 10px; }
                        pre { white-space: pre-wrap; word-wrap: break-word; }
                    </style>
                </head>
                <body>
                    <pre>${content}</pre>
                    <script>
                        window.onload = function() {
                            window.print();
                            setTimeout(() => window.close(), 1000);
                        };
                    </script>
                </body>
            </html>
        `);
        printWindow.document.close();
    },

    printViaUSB(content) {
        // Implementación básica para impresoras térmicas vía USB
        // Esto requeriría una implementación más completa según el modelo
        console.log('Intentando imprimir vía USB...');
        // Por ahora usamos el método de ventana como fallback
        this.printViaWindow(content);
    },

    // Inicializar cajón de dinero
    initCashDrawer() {
        Livewire.on('open-cash-drawer', () => {
            this.openCashDrawer();
        });
    },

    openCashDrawer() {
        // Comando ESC/POS para abrir cajón de dinero
        const drawerCommand = '\x1B\x70\x00\x19\xFA'; // Comando estándar para abrir cajón

        // Intentar enviar comando a la impresora
        if (window.electronAPI && window.electronAPI.sendToPrinter) {
            window.electronAPI.sendToPrinter(drawerCommand);
        } else {
            // Fallback: usar una señal acústica para indicar que se debería abrir el cajón
            this.playCashDrawerSound();
        }
    },

    playCashDrawerSound() {
        // Reproducir sonido de apertura de cajón
        const audio = new Audio('data:audio/wav;base64,UklGRnoGAABXQVZFZm10IBAAAAABAAEAQB8AAEAfAAABAAgAZGF0YQoGAACBhYqFbF1fdJivrJBhNjVgodDbq2EcBj+a2/LDciUFLIHO8tiJNwgZaLvt559NEAxQp+PwtmMcBjiR1/LMeSwFJHfH8N2QQAoUXrTp66hVFApGn+DyvmwhBSuBzvLZiTYIG2m98OScTgwOUarm7blmGgU7k9n1unEiBC13yO/eizEIHWq+8+OWT');
        audio.play().catch(() => {
            // Si no se puede reproducir audio, usar beep del sistema
            console.log('Abriendo cajón de dinero...');
        });
    },

    // Inicializar pantalla de cliente
    initCustomerDisplay() {
        Livewire.on('update-customer-display', (event) => {
            const data = event[0];
            this.updateCustomerDisplay(data);
        });
    },

    updateCustomerDisplay(data) {
        // Buscar pantalla de cliente (segunda ventana o monitor)
        const customerWindow = this.findCustomerDisplay();

        if (customerWindow) {
            this.renderCustomerDisplay(customerWindow, data);
        } else {
            // Si no hay pantalla secundaria, mostrar en una esquina de la pantalla principal
            // Verificar si está habilitada antes de mostrar overlay
            if (window.Livewire && window.Livewire.find && window.Livewire.find('{{ $this->getId() }}')) {
                const component = window.Livewire.find('{{ $this->getId() }}');
                if (component && component.customerDisplayEnabled) {
                    this.showCustomerDisplayOverlay(data);
                }
            } else {
                this.showCustomerDisplayOverlay(data);
            }
        }
    },

    findCustomerDisplay() {
        // Intentar encontrar una ventana de pantalla de cliente existente
        // Esto podría ser una ventana emergente o una conexión a un monitor secundario
        return null; // Por ahora retornamos null, se implementará según necesidades
    },

    renderCustomerDisplay(window, data) {
        // Renderizar información en la pantalla del cliente
        // Esto se personalizará según el diseño deseado
        console.log('Actualizando pantalla de cliente:', data);
    },

    showCustomerDisplayOverlay(data) {
        // Mostrar información del cliente como overlay en la pantalla principal
        const overlay = document.createElement('div');
        overlay.className = 'customer-display-overlay';
        overlay.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            background: #fff;
            border: 2px solid #007bff;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
            z-index: 9999;
            font-family: Arial, sans-serif;
            min-width: 300px;
        `;

        let content = '';
        if (data.action === 'sale_completed') {
            content = `
                <h4 style="margin: 0 0 10px 0; color: #28a745;">¡Gracias por su compra!</h4>
                <p style="margin: 5px 0;"><strong>Total:</strong> $${data.total.toFixed(2)}</p>
                <p style="margin: 5px 0;"><strong>Cambio:</strong> $${data.cambio.toFixed(2)}</p>
            `;
        }

        overlay.innerHTML = content;
        document.body.appendChild(overlay);

        // Remover después de 5 segundos
        setTimeout(() => {
            if (overlay.parentNode) {
                overlay.parentNode.removeChild(overlay);
            }
        }, 5000);
    }
};

// Inicializar cuando el DOM esté listo
document.addEventListener('DOMContentLoaded', function() {
    POSHardware.initBarcodeScanner();
    POSHardware.initPrinter();
    POSHardware.initCashDrawer();
    POSHardware.initCustomerDisplay();

    console.log('Hardware POS inicializado');
});

// Escuchar eventos de Livewire para enfocar el input de código de barras
Livewire.on('focus-barcode-input', () => {
    const barcodeInput = document.querySelector('[data-barcode-input]');
    if (barcodeInput) {
        barcodeInput.focus();
        barcodeInput.select();
    }
});
</script>

<style>
/* Estilos para el hardware POS */
.customer-display-overlay {
    animation: slideInRight 0.3s ease-out;
}

@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

/* Indicador visual para escaneo exitoso */
.barcode-scan-success {
    animation: pulse 0.5s ease-in-out;
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.05); }
    100% { transform: scale(1); }
}
</style>
