# 🔔 Sistema de Notificaciones Global

Sistema de notificaciones toast moderno y personalizable basado en el template de Materialize, implementado con JavaScript vanilla y compatible con Livewire.

## ✨ Características

- **🎨 Diseño moderno**: Basado en el template Materialize con estilos mejorados
- **📱 Responsive**: Adaptable a todos los tamaños de pantalla
- **🎯 Múltiples tipos**: Success, Error, Warning, Info
- **⚡ Animaciones suaves**: Transiciones fluidas y efectos visuales
- **🔧 Personalizable**: Posición, duración, colores, iconos
- **♿ Accesible**: Cumple con estándares de accesibilidad
- **📦 Sin dependencias**: JavaScript vanilla puro
- **⚡ Livewire**: Integración completa con eventos de Livewire
- **🌍 Global**: Función `notify()` disponible en todo el proyecto

## 📋 Instalación

El sistema ya está instalado en todos los layouts principales:

- `admin.blade.php`
- `horizontal.blade.php`
- `auth-basic.blade.php`
- `mensajeria.blade.php`

### Archivos incluidos:

```html
<!-- CSS -->
<link rel="stylesheet" href="{{ asset('css/notifications.css') }}">

<!-- JavaScript -->
<script src="{{ asset('js/notifications.js') }}"></script>
```

## 🚀 Uso Básico

### Método 1: Función global `notify()`

```javascript
// Notificaciones simples
notify('success', '✅ Operación completada exitosamente');
notify('error', '❌ Error al procesar la solicitud');
notify('warning', '⚠️ Advertencia: Verifica los datos ingresados');
notify('info', 'ℹ️ Información: El proceso puede tardar unos minutos');
```

### Método 2: Con opciones personalizadas

```javascript
notify('success', '✅ Guardado exitoso', {
    duration: 5000,        // Duración en milisegundos (0 = sin auto-cierre)
    dismissible: true,     // Mostrar botón de cierre
    ripple: true,          // Efecto ripple al hacer clic
    position: { x: 'right', y: 'top' } // Posición
});
```

### Método 3: Objeto de configuración

```javascript
notify({
    type: 'info',
    message: '📊 Estadísticas actualizadas',
    options: {
        duration: 3000,
        dismissible: true
    }
});
```

## 📍 Posiciones Disponibles

```javascript
// Cambiar posición global
setNotificationPosition('right', 'top');    // Por defecto
setNotificationPosition('left', 'top');     // Esquina superior izquierda
setNotificationPosition('right', 'bottom'); // Esquina inferior derecha
setNotificationPosition('left', 'bottom');  // Esquina inferior izquierda
```

## ⏱️ Duraciones

```javascript
// Duración estándar (3000ms)
notify('success', 'Mensaje estándar');

// Duración personalizada
notify('info', 'Mensaje largo', { duration: 7000 });

// Sin auto-cierre (persistent)
notify('warning', 'Mensaje persistente', { duration: 0 });

// Cambiar duración global
setNotificationDuration(5000);
```

## 🎨 Tipos de Notificaciones

| Tipo | Icono | Color | Uso |
|------|-------|--------|-----|
| `success` | ✅ | Verde (#28c76f) | Operaciones exitosas |
| `error` | ❌ | Rojo (#ea5455) | Errores y fallos |
| `warning` | ⚠️ | Naranja (#ff9f43) | Advertencias |
| `info` | ℹ️ | Azul (#00cfe8) | Información general |

## 🔌 Integración con Livewire

### Desde el componente Livewire:

```php
// En tu componente Livewire
public function save()
{
    // Tu lógica de guardado
    
    $this->dispatch('notify', [
        'type' => 'success',
        'message' => '✅ Datos guardados exitosamente'
    ]);
}

// Con opciones
public function delete()
{
    $this->dispatch('notify', [
        'type' => 'warning',
        'message' => '⚠️ ¿Estás seguro de eliminar este registro?',
        'options' => [
            'duration' => 0,
            'dismissible' => true
        ]
    ]);
}
```

### Desde JavaScript con Livewire:

```javascript
// Disparar evento manualmente
Livewire.dispatch('notify', {
    type: 'info',
    message: '🔄 Procesando solicitud...'
});
```

## 🎯 Funciones Avanzadas

### Contenido HTML

```javascript
notify('info', '<strong>📊 Reporte generado:</strong><br>Se han procesado <em>1,234</em> registros.');
```

### Múltiples notificaciones

```javascript
// Las notificaciones se apilan automáticamente
notify('success', '✅ Primer mensaje');
notify('error', '❌ Segundo mensaje');
notify('warning', '⚠️ Tercer mensaje');
```

### Limpiar todas las notificaciones

```javascript
// Limpiar todas las notificaciones activas
clearNotifications();
```

### Opciones personalizadas avanzadas

```javascript
notify('success', '🎉 ¡Felicitaciones!', {
    duration: 5000,
    dismissible: true,
    ripple: true,
    icon: true,              // Mostrar/ocultar icono
    className: 'custom-class' // Clase CSS adicional
});
```

## 🎨 Personalización CSS

### Clases CSS disponibles:

```css
/* Tamaños personalizados */
.notification-large { /* 400px ancho */ }
.notification-small { /* 250px ancho */ }
.notification-persistent { /* Bordes destacados */ }
.notification-urgent { /* Animación de pulso */ }

/* Colores personalizados */
.notyf__success .notyf__icon { background: linear-gradient(135deg, #28c76f, #1e9e5a); }
.notyf__error .notyf__icon { background: linear-gradient(135deg, #ea5455, #c93c3d); }
.notyf__warning .notyf__icon { background: linear-gradient(135deg, #ff9f43, #e67c1f); }
.notyf__info .notyf__icon { background: linear-gradient(135deg, #00cfe8, #0097b3); }
```

### Tema oscuro:

```css
@media (prefers-color-scheme: dark) {
    .notyf__toast {
        background: rgba(30, 30, 30, 0.95);
        border-color: rgba(255, 255, 255, 0.1);
    }
    .notyf__message {
        color: #fff;
    }
}
```

## 🧪 Demo y Pruebas

### Activar demo:

```html
<!-- Agregar en tu página -->
<script src="{{ asset('js/notifications-demo.js') }}"></script>
```

### Funciones de demo disponibles:

```javascript
// En la consola del navegador:
demoBasicNotifications();      // Notificaciones básicas
demoAdvancedNotifications(); // Notificaciones avanzadas
demoLivewireIntegration();     // Integración con Livewire
demoCustomOptions();         // Opciones personalizadas
demoDurationOptions();       // Diferentes duraciones
demoLargeNotifications();    // Notificaciones grandes
demoActionableNotifications(); // Con acciones
```

## 🛠️ Solución de Problemas

### Problemas comunes:

1. **Las notificaciones no aparecen**
   - Verificar que los archivos CSS y JS estén incluidos
   - Comprobar la consola del navegador por errores
   - Asegurar que el DOM esté completamente cargado

2. **Estilos no se aplican**
   - Verificar la ruta de los archivos CSS
   - Comprobar que no haya conflictos con otros estilos
   - Limpiar caché del navegador

3. **Livewire no dispara notificaciones**
   - Verificar que Livewire esté instalado y funcionando
   - Comprobar que los eventos se estén disparando correctamente
   - Verificar la consola del navegador

4. **Problemas de posición**
   - Verificar que no haya elementos con z-index alto
   - Comprobar que el body no tenga overflow hidden

### Debug:

```javascript
// Verificar si el sistema está cargado
console.log(typeof notify); // Debe mostrar: "function"

// Verificar configuración actual
console.log(notyf.options);

// Probar notificación simple
notify('info', 'Test de sistema');
```

## 📚 Ejemplos Completos

### Formulario con validación:

```javascript
// En tu JavaScript
document.getElementById('formulario').addEventListener('submit', function(e) {
    e.preventDefault();
    
    // Validar
    if (!validarFormulario()) {
        notify('error', '❌ Por favor, completa todos los campos requeridos');
        return;
    }
    
    // Procesar
    notify('info', '🔄 Procesando solicitud...', { duration: 0 });
    
    // Simular proceso
    setTimeout(() => {
        clearNotifications();
        notify('success', '✅ Formulario enviado exitosamente');
    }, 2000);
});
```

### Sistema de notificaciones progresivas:

```javascript
function procesoComplejo() {
    notify('info', '🚀 Iniciando proceso...');
    
    setTimeout(() => {
        notify('warning', '⚠️ Procesando datos (25%)', { duration: 3000 });
    }, 1000);
    
    setTimeout(() => {
        notify('warning', '⚠️ Validando información (50%)', { duration: 3000 });
    }, 3000);
    
    setTimeout(() => {
        notify('warning', '⚠️ Generando reporte (75%)', { duration: 3000 });
    }, 5000);
    
    setTimeout(() => {
        notify('success', '✅ Proceso completado exitosamente (100%)');
    }, 7000);
}
```

## 🔧 API Reference

### Funciones Globales:

| Función | Parámetros | Descripción |
|---------|------------|-------------|
| `notify(type, message, options)` | `type`: string, `message`: string, `options`: object | Muestra una notificación |
| `clearNotifications()` | Ninguno | Limpia todas las notificaciones |
| `setNotificationPosition(x, y)` | `x`: 'left'\|'right', `y`: 'top'\|'bottom' | Cambia la posición |
| `setNotificationDuration(ms)` | `ms`: number | Cambia la duración global |
| `setNotificationOptions(options)` | `options`: object | Cambia opciones globales |

### Opciones:

| Opción | Tipo | Default | Descripción |
|--------|------|---------|-------------|
| `duration` | number | 3000 | Duración en ms (0 = sin auto-cierre) |
| `dismissible` | boolean | false | Mostrar botón de cierre |
| `ripple` | boolean | true | Efecto ripple al hacer clic |
| `position` | object | {x: 'right', y: 'top'} | Posición de las notificaciones |
| `icon` | boolean | true | Mostrar/ocultar icono |
| `className` | string | '' | Clase CSS adicional |

---

## 🎉 ¡Listo!

Tu sistema de notificaciones está completamente configurado y listo para usar. 🚀

Para cualquier duda o mejora, consulta la documentación o prueba las funciones de demo.
