<?php

require_once __DIR__ . '/vendor/autoload.php';

$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Console\Kernel::class);
$kernel->bootstrap();

echo "=== Debug de Detalle Pedido ===\n\n";

try {
    // Buscar un pedido con detalles para analizar
    $pedido = \App\Models\Pedido::with(['detalles', 'detalles.producto'])->first();
    
    if (!$pedido) {
        echo "⚠️  No hay pedidos disponibles para analizar\n";
        exit(1);
    }
    
    echo "📋 Pedido ID: {$pedido->id}\n";
    echo "📋 Código: {$pedido->codigo}\n";
    echo "📋 Total de detalles: {$pedido->detalles->count()}\n\n";
    
    if ($pedido->detalles->isEmpty()) {
        echo "⚠️  El pedido no tiene detalles\n";
        exit(1);
    }
    
    echo "🔍 Analizando detalles del pedido:\n";
    foreach ($pedido->detalles as $index => $detalle) {
        echo "\n--- Detalle #" . ($index + 1) . " ---\n";
        echo "ID del detalle: {$detalle->id}\n";
        echo "Producto ID: {$detalle->productoId}\n";
        echo "Cantidad: ";
        
        // Verificar el tipo y valor de cantidad
        $cantidad = $detalle->cantidad;
        echo var_export($cantidad, true) . "\n";
        echo "Tipo de cantidad: " . gettype($cantidad) . "\n";
        
        if (is_null($cantidad)) {
            echo "❌ La cantidad es NULL\n";
        } elseif (!is_numeric($cantidad)) {
            echo "❌ La cantidad no es numérica\n";
        } elseif ($cantidad <= 0) {
            echo "⚠️  La cantidad es menor o igual a cero: {$cantidad}\n";
        } else {
            echo "✅ La cantidad es válida: {$cantidad}\n";
        }
        
        // Verificar el producto
        $producto = $detalle->producto;
        if ($producto) {
            echo "Producto: {$producto->name}\n";
            echo "Stock actual: {$producto->quantity}\n";
        } else {
            echo "❌ No se encontró el producto asociado\n";
        }
        
        // Verificar otros campos del detalle
        echo "Precio unitario: {$detalle->precio_unitario}\n";
        echo "Subtotal: {$detalle->subtotal}\n";
    }
    
    echo "\n=== Prueba de incremento ===\n";
    
    // Probar con el primer detalle que tenga cantidad válida
    $detalleValido = $pedido->detalles->first(function($detalle) {
        return is_numeric($detalle->cantidad) && $detalle->cantidad > 0 && $detalle->producto;
    });
    
    if ($detalleValido) {
        echo "Probando incremento con detalle ID: {$detalleValido->id}\n";
        echo "Cantidad: {$detalleValido->cantidad}\n";
        echo "Producto: {$detalleValido->producto->name}\n";
        echo "Stock antes: {$detalleValido->producto->quantity}\n";
        
        // Hacer una copia del stock original para comparar
        $stockOriginal = $detalleValido->producto->quantity;
        $cantidad = $detalleValido->cantidad;
        
        try {
            $detalleValido->producto->increment('quantity', $cantidad);
            echo "✅ Incremento exitoso\n";
            
            // Recargar el producto para ver el nuevo stock
            $detalleValido->producto->refresh();
            echo "Stock después: {$detalleValido->producto->quantity}\n";
            echo "Stock incrementado en: {$cantidad}\n";
            
            // Revertir el cambio para no afectar los datos
            $detalleValido->producto->decrement('quantity', $cantidad);
            echo "✅ Cambio revertido\n";
            
        } catch (\Exception $e) {
            echo "❌ Error al incrementar: " . $e->getMessage() . "\n";
        }
        
    } else {
        echo "⚠️  No se encontró un detalle válido para probar el incremento\n";
    }
    
    echo "\n=== Resumen ===\n";
    echo "El error 'Non-numeric value passed to increment method' ocurre cuando:\n";
    echo "1. La cantidad es NULL\n";
    echo "2. La cantidad no es numérica (string vacío, texto, etc.)\n";
    echo "3. La cantidad es <= 0 (aunque esto no causa el error, es un caso válido a manejar)\n";
    
} catch (\Exception $e) {
    echo "❌ Error durante el debug: " . $e->getMessage() . "\n";
    echo "Stack trace:\n" . $e->getTraceAsString() . "\n";
    exit(1);
}