<?php

namespace Database\Seeders;

use App\Models\Categoria;
use App\Models\Marca;
use App\Models\Producto;
use App\Models\ProductoVariant;
use App\Models\Empresa;
use App\Models\Sucursal;
use Illuminate\Database\Seeder;
use Illuminate\Support\Str;

class ProductosForumSupermayoristaSeeder extends Seeder
{
    // Almacenar marcas como propiedades de la clase para acceder desde el método auxiliar
    private $marcaLuckyStrike;
    private $marcaBelmont;
    private $marcaChimo;
    private $marcaPallMall;
    private $marcaViceroy;
    private $marcaNova;
    private $marcaEclipse;
    private $marcaUniversal;
    private $marcaAlive;
    private $marcaOsoBlanco;

    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Obtener empresa y sucursal (o crear si no existen)
        $empresa = Empresa::first() ?? Empresa::create([
            'nombre' => 'Forum Supermayorista',
            'ruc' => '1234567890123',
            'direccion' => 'Av. Venezuela, Plaza Venezuela',
            'telefono' => '0212-1234567',
            'email' => 'contacto@forum.com.ve',
            'moneda' => 'USD',
            'pais_id' => 1, // Asumimos que 1 es Venezuela
        ]);

        $sucursal = Sucursal::where('empresa_id', $empresa->id)->first() ?? Sucursal::create([
            'nombre' => 'Sucursal Plaza Venezuela',
            'direccion' => 'Av. Venezuela, Plaza Venezuela',
            'telefono' => '0212-1234567',
            'empresa_id' => $empresa->id,
        ]);

        // Crear categorías
        $categoriaViveres = $this->crearCategoria('Víveres', 'Productos de despensa básica', $empresa, $sucursal);
        $categoriaCigarros = $this->crearCategoria('Cigarrilos', 'Productos de tabaco y cigarros', $empresa, $sucursal);
        $categoriaConfiteria = $this->crearCategoria('Confitería', 'Productos de confitería', $empresa, $sucursal);
        $categoriaLavanderia = $this->crearCategoria('Lavandería', 'Productos para lavandería', $empresa, $sucursal);
        $categoriaHigienePersonal = $this->crearCategoria('Higiene Personal', 'Productos de cuidado e higiene personal', $empresa, $sucursal);
        $categoriaBebidas = $this->crearCategoria('Bebidas', 'Bebidas refrescantes y gaseosas', $empresa, $sucursal);
        $categoriaSnacks = $this->crearCategoria('Snacks', 'Productos de snacks y botanas', $empresa, $sucursal);
        $categoriaDesayuno = $this->crearCategoria('Desayuno', 'Productos para el desayuno', $empresa, $sucursal);
        $categoriaCuidadoBebe = $this->crearCategoria('Cuidado del Bebé', 'Productos para el cuidado del bebé', $empresa, $sucursal);

        // Crear marcas
        $this->crearMarcas($empresa, $sucursal);

        // Definir productos
        $productos = $this->definirProductos();

        // Crear productos usando el método optimizado
        foreach ($productos as $categoria => $marcas) {
            $categoriaObj = $this->obtenerCategoria($categoria, $categoriaViveres, $categoriaCigarros, $categoriaConfiteria, $categoriaLavanderia, $categoriaHigienePersonal, $categoriaBebidas, $categoriaSnacks, $categoriaDesayuno, $categoriaCuidadoBebe);
            
            foreach ($marcas as $marca => $productosArray) {
                $marcaObj = $this->obtenerMarca($marca);
                $this->crearProductos($productosArray, $categoriaObj, $marcaObj);
            }
        }

        // Mostrar mensajes de éxito
        $this->mostrarMensajesExito();
    }

    /**
     * Crear categoría
     */
    private function crearCategoria(string $nombre, string $descripcion, $empresa, $sucursal)
    {
        return Categoria::firstOrCreate(
            ['nombre' => $nombre, 'empresa_id' => $empresa->id],
            [
                'slug' => Str::slug($nombre),
                'descripcion' => $descripcion,
                'activo' => true,
                'sucursal_id' => $sucursal->id,
            ]
        );
    }

    /**
     * Crear todas las marcas
     */
    private function crearMarcas($empresa, $sucursal): void
    {
        $marcas = [
            'Polar' => 'Productos Polar',
            'Mary' => 'Productos Mary',
            'Santoni' => 'Productos Santoni',
            'Santa Monica' => 'Productos Santa Monica',
            'Amanecer' => 'Productos Amanecer',
            'Capri' => 'Productos Capri',
            'Allegri' => 'Productos Allegri',
            'Dobón' => 'Productos Dobón',
            'Pampero' => 'Productos Pampero',
            'Iberia' => 'Productos Iberia',
            'Boka' => 'Productos Boka',
            'Coamo' => 'Productos Coamo',
            'Pringles' => 'Productos Pringles',
            'Tom' => 'Productos Tom',
            'Lucky Strike' => 'Productos Lucky Strike',
            'Belmont' => 'Productos Belmont',
            'Chimo' => 'Productos Chimo',
            'Pall Mall' => 'Productos Pall Mall',
            'Viceroy' => 'Productos Viceroy',
            'Nova' => 'Productos Nova',
            'Eclipse' => 'Productos Eclipse',
            'Universal' => 'Productos Universal',
            'Alive' => 'Productos Alive',
            'Oso Blanco' => 'Productos Oso Blanco',
            'Nutribela' => 'Productos Nutribela',
            'Marlboro' => 'Productos Marlboro',
            'Camel' => 'Productos Camel',
            'Doritos' => 'Productos Doritos',
            'Lays' => 'Productos Lays',
            'Ruffles' => 'Productos Ruffles',
            'Cheetos' => 'Productos Cheetos',
            'Oreo' => 'Productos Oreo',
            'Ritz' => 'Productos Ritz',
            'Coca Cola' => 'Productos Coca Cola',
            'Pepsi' => 'Productos Pepsi',
            'Sprite' => 'Productos Sprite',
            'Fanta' => 'Productos Fanta',
            'Nestlé' => 'Productos Nestlé',
            'Maggi' => 'Productos Maggi',
            'Knorr' => 'Productos Knorr',
            'Hellmanns' => 'Productos Hellmanns',
            'Heinz' => 'Productos Heinz',
            'Kelloggs' => 'Productos Kelloggs',
            'Quaker' => 'Productos Quaker',
            'Tapa Amarilla' => 'Productos Tapa Amarilla',
            'Rexona' => 'Productos Rexona',
            'Carabobo' => 'Productos Carabobo',
            'Frut Cola' => 'Productos Frut Cola',
            'Minalba' => 'Productos Minalba',
            'Cool' => 'Productos Cool',
            'Vista' => 'Productos Vista',
            'Aldor' => 'Productos Aldor',
            'Alident' => 'Productos Alident',
            'Montalbán' => 'Productos Montalbán',
            'Cerro del Santo' => 'Productos Cerro del Santo',
            'Cheese Tris' => 'Productos Cheese Tris',
            'Carolina' => 'Productos Carolina',
            'Colgate' => 'Productos Colgate',
            'Cremor' => 'Productos Cremor',
            'Diablito Under Wood' => 'Productos Diablito Under Wood',
            'Don Kat' => 'Productos Don Kat',
            'Frubys' => 'Productos Frubys',
            'Frutea' => 'Productos Frutea',
            'Fruttsy' => 'Productos Fruttsy',
            'Frutys' => 'Productos Frutys',
            'Santa María' => 'Productos Santa María',
            'Anita' => 'Productos Anita',
            'Frita' => 'Productos Frita',
            'Jucosa' => 'Productos Jucosa',
            'El Valle' => 'Productos El Valle',
            'Lady Speed Stick' => 'Productos Lady Speed Stick',
            'Aromax' => 'Productos Aromax',
            'Lipton' => 'Productos Lipton',
            'Deline' => 'Productos Deline',
            'Mirringo' => 'Productos Mirringo',
            'Natulac' => 'Productos Natulac',
            'Concordia' => 'Productos Concordia',
            'Baby Finger' => 'Productos Baby Finger',
            'Hugme' => 'Productos Hugme',
            'Pin Pop' => 'Productos Pin Pop',
            'Power Max' => 'Productos Power Max',
            'Ideal' => 'Productos Ideal',
            'La Lucha' => 'Productos La Lucha',
            'Blue Softy' => 'Productos Blue Softy',
            'Speed Stick' => 'Productos Speed Stick',
            'Suavitel' => 'Productos Suavitel',
            'Super Pega' => 'Productos Super Pega',
            'Tam Cikolatam' => 'Productos Tam Cikolatam',
            'Puffy' => 'Productos Puffy',
            'Saltitacos' => 'Productos Saltitacos',
            'Fruton' => 'Productos Fruton',
            'Glup' => 'Productos Glup',
            'Cristal' => 'Productos Cristal',
            'Valencia' => 'Productos Valencia',
            'Justy' => 'Productos Justy',
            'Speed' => 'Productos Speed',
            'Gatorade' => 'Productos Gatorade',
            'Mi Brisa' => 'Productos Mi Brisa',
            'Heal' => 'Productos Heal',
            'Golden' => 'Productos Golden',
            'Frescolita' => 'Productos Frescolita',
            'Chinotto' => 'Productos Chinotto',
            'Hit' => 'Productos Hit',
            'Encendedor' => 'Productos Encendedor',
            'Panelada' => 'Productos Panelada',

        ];

        foreach ($marcas as $nombre => $descripcion) {
            $marca = Marca::firstOrCreate(
                ['nombre' => $nombre, 'empresa_id' => $empresa->id],
                [
                    'slug' => Str::slug($nombre),
                    'descripcion' => $descripcion,
                    'activo' => true,
                    'sucursal_id' => $sucursal->id,
                ]
            );

            // Almacenar marcas especiales como propiedades
            $this->almacenarMarcaEspecial($nombre, $marca);
        }
    }

    /**
     * Almacenar marcas especiales como propiedades
     */
    private function almacenarMarcaEspecial(string $nombre, $marca): void
    {
        switch ($nombre) {
            case 'Lucky Strike':
                $this->marcaLuckyStrike = $marca;
                break;
            case 'Belmont':
                $this->marcaBelmont = $marca;
                break;
            case 'Chimo':
                $this->marcaChimo = $marca;
                break;
            case 'Pall Mall':
                $this->marcaPallMall = $marca;
                break;
            case 'Viceroy':
                $this->marcaViceroy = $marca;
                break;
            case 'Nova':
                $this->marcaNova = $marca;
                break;
            case 'Eclipse':
                $this->marcaEclipse = $marca;
                break;
            case 'Universal':
                $this->marcaUniversal = $marca;
                break;
            case 'Alive':
                $this->marcaAlive = $marca;
                break;
            case 'Oso Blanco':
                $this->marcaOsoBlanco = $marca;
                break;
        }
    }

    /**
     * Definir todos los productos organizados por categoría y marca
     */
    private function definirProductos(): array
    {
        return [
            'Víveres' => [
                'Polar' => [
                    ['nombre' => 'Arroz Blanco Polar', 'descripcion' => 'Arroz blanco de alta calidad', 'precio_base' => 1.50, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-POLAR-1KG', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-POLAR-5KG', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Harina de Maíz Precocida Polar', 'descripcion' => 'Harina de maíz precocida para arepas', 'precio_base' => 1.20, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-POL-1KG', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Peso: 2kg', 'values' => ['Peso' => '2kg'], 'sku' => 'HAR-POL-2KG', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Harina de Trigo Pol', 'descripcion' => 'Harina de trigo 1kg', 'precio_base' => 1.20, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-POL-1KG', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Peso: 2kg', 'values' => ['Peso' => '2kg'], 'sku' => 'HAR-POL-2KG', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Aceite de Soja Soya', 'descripcion' => 'Aceite de soja 900ml', 'precio_base' => 2.00, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'ACE-SOY-900', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Pasta Larga tipo Spaghetti Santa Rita', 'descripcion' => 'Pasta seca tipo spaghetti', 'precio_base' => 0.90, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-POL-500', 'precio_ajuste' => 0, 'cantidad' => 110]]],
                    ['nombre' => 'Azúcar Morena', 'descripcion' => 'Azúcar morena Polar', 'precio_base' => 1.10, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'AZU-POL-1KG', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Sal de Mesa', 'descripcion' => 'Sal de mesa refinada', 'precio_base' => 0.50, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'SAL-POL-1KG', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Café Instantáneo', 'descripcion' => 'Café instantáneo soluble', 'precio_base' => 2.80, 'variantes' => [['nombre' => 'Peso: 200gr', 'values' => ['Peso' => '200gr'], 'sku' => 'CAF-POL-200', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Leche en Polvo', 'descripcion' => 'Leche en polvo entera', 'precio_base' => 3.20, 'variantes' => [['nombre' => 'Peso: 400gr', 'values' => ['Peso' => '400gr'], 'sku' => 'LEC-POL-400', 'precio_ajuste' => 0, 'cantidad' => 75], ['nombre' => 'Peso: 800gr', 'values' => ['Peso' => '800gr'], 'sku' => 'LEC-POL-800', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Galletas Saladas', 'descripcion' => 'Galletas saladas clásicas', 'precio_base' => 0.80, 'variantes' => [['nombre' => 'Peso: 250gr', 'values' => ['Peso' => '250gr'], 'sku' => 'GAL-POL-250', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                    ['nombre' => 'Salsa de Tomate', 'descripcion' => 'Salsa de tomate tradicional', 'precio_base' => 1.50, 'variantes' => [['nombre' => 'Peso: 340gr', 'values' => ['Peso' => '340gr'], 'sku' => 'SAL-POL-340', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Mayonesa', 'descripcion' => 'Mayonesa suave', 'precio_base' => 1.80, 'variantes' => [['nombre' => 'Peso: 350gr', 'values' => ['Peso' => '350gr'], 'sku' => 'MAY-POL-350', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Mermelada Fresa', 'descripcion' => 'Mermelada de fresa', 'precio_base' => 2.20, 'variantes' => [['nombre' => 'Peso: 454gr', 'values' => ['Peso' => '454gr'], 'sku' => 'MER-POL-454', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Atún en Agua', 'descripcion' => 'Atún en agua', 'precio_base' => 1.90, 'variantes' => [['nombre' => 'Peso: 160gr', 'values' => ['Peso' => '160gr'], 'sku' => 'ATU-POL-160', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Sardinas en Aceite', 'descripcion' => 'Sardinas en aceite vegetal', 'precio_base' => 1.60, 'variantes' => [['nombre' => 'Peso: 120gr', 'values' => ['Peso' => '120gr'], 'sku' => 'SAR-POL-120', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Mary' => [
                    [
                        'nombre' => 'Arroz Blanco Mary',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 1.40,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-MARY-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-MARY-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Mary',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 1.10,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-MARY-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-MARY-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Santoni' => [
                    [
                        'nombre' => 'Arroz Blanco Santoni',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 1.60,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-SANTONI-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-SANTONI-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Santoni',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 1.30,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-SANTONI-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-SANTONI-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Santa Monica' => [
                    [
                        'nombre' => 'Arroz Blanco Santa Monica',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 1.70,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-SANTAMONICA-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-SANTAMONICA-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Santa Monica',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 1.40,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-SANTAMONICA-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-SANTAMONICA-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Amanecer' => [
                    [
                        'nombre' => 'Arroz Blanco Amanecer',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 1.80,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-AMANECER-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-AMANECER-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Amanecer',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 1.50,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-AMANECER-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-AMANECER-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Capri' => [
                    [
                        'nombre' => 'Arroz Blanco Capri',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 1.90,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-CAPRI-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-CAPRI-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Capri',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 1.60,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-CAPRI-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-CAPRI-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Allegri' => [
                    [
                        'nombre' => 'Arroz Blanco Allegri',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 2.00,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-ALLEGRI-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-ALLEGRI-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Allegri',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 1.70,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-ALLEGRI-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-ALLEGRI-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Dobón' => [
                    [
                        'nombre' => 'Arroz Blanco Dobón',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 2.10,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-DOBON-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-DOBON-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Dobón',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 1.80,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-DOBON-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-DOBON-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Pampero' => [
                    [
                        'nombre' => 'Arroz Blanco Pampero',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 2.20,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-PAMPERO-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-PAMPERO-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Pampero',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 1.90,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-PAMPERO-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-PAMPERO-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Iberia' => [
                    [
                        'nombre' => 'Arroz Blanco Iberia',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 2.30,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-IBERIA-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-IBERIA-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Iberia',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 2.00,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-IBERIA-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-IBERIA-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Boka' => [
                    [
                        'nombre' => 'Arroz Blanco Boka',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 2.40,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-BOKA-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-BOKA-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Boka',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 2.10,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-BOKA-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-BOKA-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Coamo' => [
                    [
                        'nombre' => 'Arroz Blanco Coamo',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 2.50,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-COAMO-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-COAMO-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Coamo',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 2.20,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-COAMO-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-COAMO-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Pringles' => [
                    [
                        'nombre' => 'Arroz Blanco Pringles',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 2.60,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-PRINGLES-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-PRINGLES-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Pringles',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 2.30,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-PRINGLES-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-PRINGLES-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
                'Tom' => [
                    [
                        'nombre' => 'Arroz Blanco Tom',
                        'descripcion' => 'Arroz blanco de alta calidad',
                        'precio_base' => 2.70,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-TOM-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-TOM-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                    [
                        'nombre' => 'Harina de Maíz Precocida Tom',
                        'descripcion' => 'Harina de maíz precocida para arepas',
                        'precio_base' => 2.40,
                        'variantes' => [
                            ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-TOM-1KG', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'HAR-TOM-5KG', 'precio_ajuste' => 0, 'cantidad' => 50],
                        ]
                    ],
                ],
            ],
            'Cigarrilos' => [
                'Lucky Strike' => [
                    [
                        'nombre' => 'Cigarrillo Lucky Strike',
                        'descripcion' => 'Cigarrillo Lucky Strike',
                        'precio_base' => 1.00,
                        'variantes' => [
                            ['nombre' => 'Cantidad: 20', 'values' => ['Cantidad' => '20'], 'sku' => 'LS-20', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Cantidad: 10', 'values' => ['Cantidad' => '10'], 'sku' => 'LS-10', 'precio_ajuste' => 0, 'cantidad' => 100],
                        ]
                    ],
                ],
                'Belmont' => [
                    [
                        'nombre' => 'Cigarrillo Belmont',
                        'descripcion' => 'Cigarrillo Belmont',
                        'precio_base' => 1.10,
                        'variantes' => [
                            ['nombre' => 'Cantidad: 20', 'values' => ['Cantidad' => '20'], 'sku' => 'BM-20', 'precio_ajuste' => 0, 'cantidad' => 100],
                            ['nombre' => 'Cantidad: 10', 'values' => ['Cantidad' => '10'], 'sku' => 'BM-10', 'precio_ajuste' => 0, 'cantidad' => 100],
                        ]
                    ],
                ],
                'Chimo' => [
                    ['nombre' => 'Chimo el apureño', 'descripcion' => 'Chimo con extracto de tabaco y sabores naturales.', 'precio_base' => 1.43, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CHI-APU-1', 'precio_ajuste' => 0, 'cantidad' => 1000]]],
                    ['nombre' => 'Chimo El Tigrito Amarillo', 'descripcion' => 'Chimo El Tigrito Amarillo', 'precio_base' => 0.51, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CHI-TIG-AMA', 'precio_ajuste' => 0, 'cantidad' => 200]]],
                ],
                'Coamo' => [
                    ['nombre' => 'Aceite de Soya Coamo 900ml', 'descripcion' => 'Aceite de soya 900ml', 'precio_base' => 2.40, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'ACE-COA-900', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Ideal' => [
                    ['nombre' => 'Aceite Ideal 100ml', 'descripcion' => 'Aceite vegetal 100ml', 'precio_base' => 0.85, 'variantes' => [['nombre' => 'Volumen: 100ml', 'values' => ['Volumen' => '100ml'], 'sku' => 'ACE-IDE-100', 'precio_ajuste' => 0, 'cantidad' => 150]]],
                    ['nombre' => 'Aceite Ideal 250ml', 'descripcion' => 'Aceite vegetal 250ml', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Volumen: 250ml', 'values' => ['Volumen' => '250ml'], 'sku' => 'ACE-IDE-250', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                    ['nombre' => 'Aceite Ideal 430ml', 'descripcion' => 'Aceite vegetal 430ml', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 430ml', 'values' => ['Volumen' => '430ml'], 'sku' => 'ACE-IDE-430', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Aceite Ideal 900ml', 'descripcion' => 'Aceite vegetal 900ml', 'precio_base' => 3.25, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'ACE-IDE-900', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'La Lucha' => [
                    ['nombre' => 'Aceite La Lucha de Soja 900ml', 'descripcion' => 'Aceite de soja 900ml', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'ACE-LUC-900', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                ],
                'Concordia' => [
                    ['nombre' => 'Oleo de Soja Concordia', 'descripcion' => 'Aceite de soja Concordia', 'precio_base' => 2.95, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'ACE-CON-900', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
                'Montalbán' => [
                    ['nombre' => 'Azúcar Montalbán', 'descripcion' => 'Azúcar refinada', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'AZU-MON-1KG', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                ],
                'Cerro del Santo' => [
                    ['nombre' => 'Café Cerro del Santo 100g', 'descripcion' => 'Café molido 100g', 'precio_base' => 2.45, 'variantes' => [['nombre' => 'Peso: 100g', 'values' => ['Peso' => '100g'], 'sku' => 'CAF-CER-100', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Deline' => [
                    ['nombre' => 'Mantequilla Deline 250g', 'descripcion' => 'Mantequilla con sal', 'precio_base' => 3.75, 'variantes' => [['nombre' => 'Peso: 250g', 'values' => ['Peso' => '250g'], 'sku' => 'MAN-DEL-250', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                ],

                'Santoni' => [
                    ['nombre' => 'Arroz Zafiro Santoni', 'descripcion' => 'Arroz blanco de alta calidad', 'precio_base' => 1.35, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-SAN-ZAF-1KG', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-SAN-ZAF-5KG', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Arroz Premium Santoni', 'descripcion' => 'Arroz blanco de calidad premium', 'precio_base' => 1.60, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-SAN-PREM-1KG', 'precio_ajuste' => 0, 'cantidad' => 80], ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-SAN-PREM-5KG', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                    ['nombre' => 'Arroz Excelente Santoni', 'descripcion' => 'Arroz blanco de calidad excelente', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-SAN-EXC-1KG', 'precio_ajuste' => 0, 'cantidad' => 90], ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-SAN-EXC-5KG', 'precio_ajuste' => 0, 'cantidad' => 45]]],
                    ['nombre' => 'Harina PAN Santoni', 'descripcion' => 'Harina de maíz precocida para arepas', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'HAR-SAN-1KG', 'precio_ajuste' => 0, 'cantidad' => 115], ['nombre' => 'Peso: 2kg', 'values' => ['Peso' => '2kg'], 'sku' => 'HAR-SAN-2KG', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Galletas Santoni', 'descripcion' => 'Galletas saladas clásicas', 'precio_base' => 0.75, 'variantes' => [['nombre' => 'Peso: 200gr', 'values' => ['Peso' => '200gr'], 'sku' => 'GAL-SAN-200', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Sopa Santoni', 'descripcion' => 'Sopa instantánea sabor gallina', 'precio_base' => 1.10, 'variantes' => [['nombre' => 'Peso: 100gr', 'values' => ['Peso' => '100gr'], 'sku' => 'SOP-SAN-100', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                ],
                'Boka' => [
                    ['nombre' => 'Refresco Boka Sabor Fresa', 'descripcion' => 'Refresco en polvo sabor fresa 18gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 18gr', 'values' => ['Peso' => '18gr'], 'sku' => 'REF-BOK-FRE-18', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Refresco Boka Sabor Mango', 'descripcion' => 'Refresco en polvo sabor mango 18gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 18gr', 'values' => ['Peso' => '18gr'], 'sku' => 'REF-BOK-MAN-18', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Refresco Boka Sabor Sandía', 'descripcion' => 'Refresco en polvo sabor sandía 18gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 18gr', 'values' => ['Peso' => '18gr'], 'sku' => 'REF-BOK-SAN-18', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Refresco Boka Sabor Maracuyá', 'descripcion' => 'Refresco en polvo sabor maracuyá 18gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 18gr', 'values' => ['Peso' => '18gr'], 'sku' => 'REF-BOK-MAR-18', 'precio_ajuste' => 0, 'cantidad' => 110]]],
                    ['nombre' => 'Refresco Boka Sabor Guanábana', 'descripcion' => 'Refresco en polvo sabor guanábana 18gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 18gr', 'values' => ['Peso' => '18gr'], 'sku' => 'REF-BOK-GUA-18', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                    ['nombre' => 'Refresco Boka Sabor Corozo', 'descripcion' => 'Refresco en polvo sabor corozo 10gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'REF-BOK-COR-10', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Refresco Boka Sabor Lichi', 'descripcion' => 'Refresco en polvo sabor lichi 10gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'REF-BOK-LIC-10', 'precio_ajuste' => 0, 'cantidad' => 95]]],
                    ['nombre' => 'Refresco Boka Sabor Tamarindo', 'descripcion' => 'Refresco en polvo sabor tamarindo 10gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'REF-BOK-TAM-10', 'precio_ajuste' => 0, 'cantidad' => 130]]],
                    ['nombre' => 'Refresco Boka Sabor Lima Limón', 'descripcion' => 'Refresco en polvo sabor lima limón 18gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 18gr', 'values' => ['Peso' => '18gr'], 'sku' => 'REF-BOK-LIM-18', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Refresco Boka Sabor Naranja', 'descripcion' => 'Refresco en polvo sabor naranja 18gr', 'precio_base' => 0.79, 'variantes' => [['nombre' => 'Peso: 18gr', 'values' => ['Peso' => '18gr'], 'sku' => 'REF-BOK-NAR-18', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                ],
                'Santa Monica' => [
                    ['nombre' => 'Arroz Zafiro Santa Monica', 'descripcion' => 'Arroz blanco de alta calidad', 'precio_base' => 1.30, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-SMO-ZAF-1KG', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-SMO-ZAF-5KG', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Arroz Premium Santa Monica', 'descripcion' => 'Arroz blanco de calidad premium', 'precio_base' => 1.55, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-SMO-PREM-1KG', 'precio_ajuste' => 0, 'cantidad' => 80], ['nombre' => 'Peso: 5kg', 'values' => ['Peso' => '5kg'], 'sku' => 'ARZ-SMO-PREM-5KG', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                ],
                'Amanecer' => [
                    ['nombre' => 'Arroz Blanco Tipo I 900gr Amanecer', 'descripcion' => 'Arroz blanco tipo I de 900gr', 'precio_base' => 1.73, 'variantes' => [['nombre' => 'Peso: 900gr', 'values' => ['Peso' => '900gr'], 'sku' => 'ARZ-AMA-900', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Arroz Blanco Tipo I 1kg Amanecer', 'descripcion' => 'Arroz blanco tipo I de 1kg', 'precio_base' => 1.91, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ARZ-AMA-1KG', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Café en Granos 1kg Amanecer', 'descripcion' => 'Café en granos de 1kg', 'precio_base' => 16.64, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'CAF-AMA-GR-1KG', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Café Gourmet 100gr Amanecer', 'descripcion' => 'Café gourmet de 100gr', 'precio_base' => 1.56, 'variantes' => [['nombre' => 'Peso: 100gr', 'values' => ['Peso' => '100gr'], 'sku' => 'CAF-AMA-GOU-100', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Maíz para Cotufa 400gr Amanecer', 'descripcion' => 'Maíz para cotufa de 400gr', 'precio_base' => 1.54, 'variantes' => [['nombre' => 'Peso: 400gr', 'values' => ['Peso' => '400gr'], 'sku' => 'MAI-AMA-400', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Azúcar Amanecer 1kg', 'descripcion' => 'Azúcar refinada 1kg', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'AZU-AMA-1KG', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                ],
                'Capri' => [
                    ['nombre' => 'Pasta Premium Codito 1kg Capri', 'descripcion' => 'Pasta premium codito de 1kg', 'precio_base' => 2.23, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-COD-1KG', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Pasta Premium Spaghetti 1kg Capri', 'descripcion' => 'Pasta premium spaghetti de 1kg', 'precio_base' => 2.23, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-SPA-1KG', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Pasta Premium Linguini 1kg Capri', 'descripcion' => 'Pasta premium linguini de 1kg', 'precio_base' => 2.23, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-LIN-1KG', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Pasta Premium Macarron 1kg Capri', 'descripcion' => 'Pasta premium macarron de 1kg', 'precio_base' => 2.23, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-MAC-1KG', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Pasta Premium Pluma 1kg Capri', 'descripcion' => 'Pasta premium pluma de 1kg', 'precio_base' => 2.23, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-PLU-1KG', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Pasta Premium Fettuccine 1kg Capri', 'descripcion' => 'Pasta premium fettuccine de 1kg', 'precio_base' => 2.35, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-FET-1KG', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Pasta Premium Caracola 1kg Capri', 'descripcion' => 'Pasta premium caracola de 1kg', 'precio_base' => 2.23, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-CAR-1KG', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Pasta Premium Tornillo 1kg Capri', 'descripcion' => 'Pasta premium tornillo de 1kg', 'precio_base' => 2.23, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-TOR-1KG', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Pasta Premium Angel Hair 1kg Capri', 'descripcion' => 'Pasta premium cabello de angel de 1kg', 'precio_base' => 2.30, 'variantes' => [['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'PAS-CAP-ANG-1KG', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Pasta Premium Lasagna 500gr Capri', 'descripcion' => 'Pasta premium lasagna de 500gr', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-CAP-LAS-500', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Pasta Premium Ravioli 500gr Capri', 'descripcion' => 'Pasta premium ravioli de 500gr', 'precio_base' => 2.45, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-CAP-RAV-500', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Pasta Premium Farfalle 500gr Capri', 'descripcion' => 'Pasta premium farfalle de 500gr', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-CAP-FAR-500', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Pasta Premium Rigatoni 500gr Capri', 'descripcion' => 'Pasta premium rigatoni de 500gr', 'precio_base' => 1.90, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-CAP-RIG-500', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Pasta Premium Pappardelle 500gr Capri', 'descripcion' => 'Pasta premium pappardelle de 500gr', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-CAP-PAP-500', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                    ['nombre' => 'Pasta Premium Tagliatelle 500gr Capri', 'descripcion' => 'Pasta premium tagliatelle de 500gr', 'precio_base' => 2.10, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-CAP-TAG-500', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Pasta Premium Gnocchi 500gr Capri', 'descripcion' => 'Pasta premium gnocchi de 500gr', 'precio_base' => 2.55, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-CAP-GNO-500', 'precio_ajuste' => 0, 'cantidad' => 45]]],
                    ['nombre' => 'Pasta Premium Orzo 500gr Capri', 'descripcion' => 'Pasta premium orzo de 500gr', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-CAP-ORZ-500', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Allegri' => [
                    ['nombre' => 'Pasta Allegri Rigatones 12x500mg', 'descripcion' => 'Pasta rigatones caja de 12 unidades de 500g', 'precio_base' => 18.50, 'variantes' => [['nombre' => 'Caja: 12 x 500g', 'values' => ['Caja' => '12 x 500g'], 'sku' => 'PAS-ALL-RIG-12X500', 'precio_ajuste' => 0, 'cantidad' => 25]]],
                    ['nombre' => 'Pasta Larga Spaghetti 500gr Allegri', 'descripcion' => 'Pasta larga spaghetti de 500gr', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-SPA-500', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Pasta Larga Linguini 500gr Allegri', 'descripcion' => 'Pasta larga linguini de 500gr', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-LIN-500', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Pasta Corta Macarron 500gr Allegri', 'descripcion' => 'Pasta corta macarron de 500gr', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-MAC-500', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Pasta Corta Pluma 500gr Allegri', 'descripcion' => 'Pasta corta pluma de 500gr', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-PLU-500', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Pasta Corta Codito 500gr Allegri', 'descripcion' => 'Pasta corta codito de 500gr', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-COD-500', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Pasta Larga Fettuccine 500gr Allegri', 'descripcion' => 'Pasta larga fettuccine de 500gr', 'precio_base' => 1.55, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-FET-500', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Pasta Corta Caracola 500gr Allegri', 'descripcion' => 'Pasta corta caracola de 500gr', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-CAR-500', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Pasta Larga Angel Hair 500gr Allegri', 'descripcion' => 'Pasta larga cabello de angel de 500gr', 'precio_base' => 1.50, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-ANG-500', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Pasta Corta Tornillo 500gr Allegri', 'descripcion' => 'Pasta corta tornillo de 500gr', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Peso: 500gr', 'values' => ['Peso' => '500gr'], 'sku' => 'PAS-ALL-TOR-500', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                ],
                'Dobón' => [
                    ['nombre' => 'Leche Entera UHT 1L Dobón', 'descripcion' => 'Leche entera UHT de 1 litro', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'LEC-DOB-1L', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Leche Descremada UHT 1L Dobón', 'descripcion' => 'Leche descremada UHT de 1 litro', 'precio_base' => 2.10, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'LEC-DOB-DESC-1L', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                ],
                'Pampero' => [
                    ['nombre' => 'Ketchup Pampero 340gr', 'descripcion' => 'Salsa de tomate ketchup de 340gr', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Peso: 340gr', 'values' => ['Peso' => '340gr'], 'sku' => 'KET-PAM-340', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Salsa de Tomate Pampero 198ml', 'descripcion' => 'Salsa de tomate 198ml', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 198ml', 'values' => ['Peso' => '198ml'], 'sku' => 'SAL-PAM-198', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                ],
                'Iberia' => [
                    ['nombre' => 'Aceite de Oliva Extra Virgen 500ml Iberia', 'descripcion' => 'Aceite de oliva extra virgen de 500ml', 'precio_base' => 4.50, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'ACE-IBE-OLI-500', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Vinagre Blanco 500ml Iberia', 'descripcion' => 'Vinagre blanco de 500ml', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'VIN-IBE-BLA-500', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Vinagre de Manzana 500ml Iberia', 'descripcion' => 'Vinagre de manzana de 500ml', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'VIN-IBE-MAN-500', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Orégano Molido 15gr Iberia', 'descripcion' => 'Orégano molido de 15gr', 'precio_base' => 0.85, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'ORE-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Comino Molido 15gr Iberia', 'descripcion' => 'Comino molido de 15gr', 'precio_base' => 0.90, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'COM-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 95]]],
                    ['nombre' => 'Pimienta Negra Molida 15gr Iberia', 'descripcion' => 'Pimienta negra molida de 15gr', 'precio_base' => 1.10, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'PIM-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Paprika Dulce 15gr Iberia', 'descripcion' => 'Paprika dulce de 15gr', 'precio_base' => 1.05, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'PAP-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Ajo en Polvo 15gr Iberia', 'descripcion' => 'Ajo en polvo de 15gr', 'precio_base' => 0.95, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'AJO-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Cebolla en Polvo 15gr Iberia', 'descripcion' => 'Cebolla en polvo de 15gr', 'precio_base' => 0.90, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'CEB-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 95]]],
                    ['nombre' => 'Laurel Entero 10gr Iberia', 'descripcion' => 'Hojas de laurel enteras de 10gr', 'precio_base' => 0.80, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'LAU-IBE-10', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Tomillo Seco 10gr Iberia', 'descripcion' => 'Tomillo seco de 10gr', 'precio_base' => 0.85, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'TOM-IBE-10', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Romero Seco 10gr Iberia', 'descripcion' => 'Romero seco de 10gr', 'precio_base' => 0.90, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'ROM-IBE-10', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Albahaca Seca 10gr Iberia', 'descripcion' => 'Albahaca seca de 10gr', 'precio_base' => 0.95, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'ALB-IBE-10', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Perejil Seco 10gr Iberia', 'descripcion' => 'Perejil seco de 10gr', 'precio_base' => 0.80, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'PER-IBE-10', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Cilantro Seco 10gr Iberia', 'descripcion' => 'Cilantro seco de 10gr', 'precio_base' => 0.85, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'CIL-IBE-10', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Canela en Polvo 15gr Iberia', 'descripcion' => 'Canela en polvo de 15gr', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'CAN-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Nuez Moscada Molida 10gr Iberia', 'descripcion' => 'Nuez moscada molida de 10gr', 'precio_base' => 1.35, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'NUE-IBE-10', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Clavo de Olor Entero 10gr Iberia', 'descripcion' => 'Clavo de olor entero de 10gr', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 10gr', 'values' => ['Peso' => '10gr'], 'sku' => 'CLA-IBE-10', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                    ['nombre' => 'Jengibre en Polvo 15gr Iberia', 'descripcion' => 'Jengibre en polvo de 15gr', 'precio_base' => 1.20, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'JEN-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Curry en Polvo 15gr Iberia', 'descripcion' => 'Curry en polvo de 15gr', 'precio_base' => 1.30, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'CUR-IBE-15', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                    ['nombre' => 'Cúrcuma en Polvo 15gr Iberia', 'descripcion' => 'Cúrcuma en polvo de 15gr', 'precio_base' => 1.10, 'variantes' => [['nombre' => 'Peso: 15gr', 'values' => ['Peso' => '15gr'], 'sku' => 'CUR-IBE-POL-15', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Adobo Completo 25gr Iberia', 'descripcion' => 'Adobo completo de 25gr', 'precio_base' => 1.40, 'variantes' => [['nombre' => 'Peso: 25gr', 'values' => ['Peso' => '25gr'], 'sku' => 'ADO-IBE-25', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Sazón Completo 25gr Iberia', 'descripcion' => 'Sazón completo de 25gr', 'precio_base' => 1.35, 'variantes' => [['nombre' => 'Peso: 25gr', 'values' => ['Peso' => '25gr'], 'sku' => 'SAZ-IBE-25', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                ],
                'Maggi' => [
                    ['nombre' => 'Cubitos Maggi Pollo', 'descripcion' => 'Cubitos de caldo sabor pollo', 'precio_base' => 0.85, 'variantes' => [['nombre' => 'Cantidad: 8 cubitos', 'values' => ['Cantidad' => '8 cubitos'], 'sku' => 'MAG-POL-8', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                    ['nombre' => 'Cubitos Maggi Res', 'descripcion' => 'Cubitos de caldo sabor res', 'precio_base' => 0.85, 'variantes' => [['nombre' => 'Cantidad: 8 cubitos', 'values' => ['Cantidad' => '8 cubitos'], 'sku' => 'MAG-RES-8', 'precio_ajuste' => 0, 'cantidad' => 110]]],
                    ['nombre' => 'Sopa Maggi Pollo con Fideos', 'descripcion' => 'Sopa instantánea sabor pollo con fideos', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 70g', 'values' => ['Peso' => '70g'], 'sku' => 'SOP-MAG-POL-70', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Knorr' => [
                    ['nombre' => 'Sopa Knorr Pollo', 'descripcion' => 'Sopa deshidratada sabor pollo', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Peso: 65g', 'values' => ['Peso' => '65g'], 'sku' => 'SOP-KNO-POL-65', 'precio_ajuste' => 0, 'cantidad' => 95]]],
                    ['nombre' => 'Sopa Knorr Verduras', 'descripcion' => 'Sopa deshidratada sabor verduras', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Peso: 65g', 'values' => ['Peso' => '65g'], 'sku' => 'SOP-KNO-VER-65', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Aderezos Knorr Suiza', 'descripcion' => 'Aderezo cremoso sabor suiza', 'precio_base' => 2.45, 'variantes' => [['nombre' => 'Peso: 400ml', 'values' => ['Peso' => '400ml'], 'sku' => 'ADE-KNO-SUI-400', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
                'Hellmanns' => [
                    ['nombre' => 'Mayonesa Hellmanns Real', 'descripcion' => 'Mayonesa real con huevos frescos', 'precio_base' => 3.25, 'variantes' => [['nombre' => 'Peso: 475ml', 'values' => ['Peso' => '475ml'], 'sku' => 'MAY-HEL-475', 'precio_ajuste' => 0, 'cantidad' => 65], ['nombre' => 'Peso: 445ml', 'values' => ['Peso' => '445ml'], 'sku' => 'MAY-HEL-445', 'precio_ajuste' => -0.20, 'cantidad' => 70], ['nombre' => 'Peso: 175ml', 'values' => ['Peso' => '175ml'], 'sku' => 'MAY-HEL-175', 'precio_ajuste' => -1.50, 'cantidad' => 90]]],
                    ['nombre' => 'Mayonesa Hellmanns Light', 'descripcion' => 'Mayonesa baja en grasa', 'precio_base' => 3.45, 'variantes' => [['nombre' => 'Peso: 475ml', 'values' => ['Peso' => '475ml'], 'sku' => 'MAY-HEL-LIG-475', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                ],
                'Heinz' => [
                    ['nombre' => 'Ketchup Heinz', 'descripcion' => 'Salsa de tomate ketchup', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Peso: 397g', 'values' => ['Peso' => '397g'], 'sku' => 'KET-HEI-397', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Mostaza Heinz', 'descripcion' => 'Mostaza amarilla clásica', 'precio_base' => 2.65, 'variantes' => [['nombre' => 'Peso: 226g', 'values' => ['Peso' => '226g'], 'sku' => 'MOS-HEI-226', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
                'Tapa Amarilla' => [
                    ['nombre' => 'Margarina Tapa Amarilla', 'descripcion' => 'Margarina vegetal clásica', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'MAR-TAP-500', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Aceite Tapa Amarilla', 'descripcion' => 'Aceite vegetal para cocinar', 'precio_base' => 3.15, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'ACE-TAP-900', 'precio_ajuste' => 0, 'cantidad' => 70], ['nombre' => 'Volumen: 1.8L', 'values' => ['Volumen' => '1.8L'], 'sku' => 'ACE-TAP-1800', 'precio_ajuste' => 2.85, 'cantidad' => 50]]],
                    ['nombre' => 'Mantequilla Tapa Amarilla', 'descripcion' => 'Mantequilla con sal', 'precio_base' => 3.45, 'variantes' => [['nombre' => 'Peso: 250g', 'values' => ['Peso' => '250g'], 'sku' => 'MAN-TAP-250', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Aceite en Spray Tapa Amarilla', 'descripcion' => 'Aceite vegetal en spray antiadherente', 'precio_base' => 4.25, 'variantes' => [['nombre' => 'Volumen: 190ml', 'values' => ['Volumen' => '190ml'], 'sku' => 'ACE-SPR-TAP-190', 'precio_ajuste' => 0, 'cantidad' => 45]]],
                    ['nombre' => 'Manteca Tapa Amarilla', 'descripcion' => 'Manteca vegetal para cocinar', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'MTC-TAP-500', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                    ['nombre' => 'Margarina Light Tapa Amarilla', 'descripcion' => 'Margarina baja en grasa', 'precio_base' => 2.65, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'MAR-LIG-TAP-500', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Aceite de Maíz Tapa Amarilla', 'descripcion' => 'Aceite puro de maíz', 'precio_base' => 3.45, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'ACE-MAI-TAP-900', 'precio_ajuste' => 0, 'cantidad' => 55]]],
                    ['nombre' => 'Aceite Vegetal de Palma', 'descripcion' => 'Aceite vegetal de palma para cocinar', 'precio_base' => 2.95, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'ACE-PAL-TAP-900', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                ],
            ],
            'Confitería' => [
                'Pringles' => [
                    ['nombre' => 'Pringles Original', 'descripcion' => 'Papas fritas sabor original en presentación de 110g', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'PRI-ORI-110', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 158g', 'values' => ['Peso' => '158g'], 'sku' => 'PRI-ORI-158', 'precio_ajuste' => 1.20, 'cantidad' => 80]]],
                    ['nombre' => 'Pringles Crema y Cebolla', 'descripcion' => 'Papas fritas sabor crema y cebolla en presentación de 110g', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'PRI-CYC-110', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 158g', 'values' => ['Peso' => '158g'], 'sku' => 'PRI-CYC-158', 'precio_ajuste' => 1.20, 'cantidad' => 80]]],
                    ['nombre' => 'Pringles Queso Cheddar', 'descripcion' => 'Papas fritas sabor queso cheddar en presentación de 110g', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'PRI-QUE-110', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 158g', 'values' => ['Peso' => '158g'], 'sku' => 'PRI-QUE-158', 'precio_ajuste' => 1.20, 'cantidad' => 80]]],
                    ['nombre' => 'Pringles Barbacoa BBQ', 'descripcion' => 'Papas fritas sabor barbacoa BBQ en presentación de 110g', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'PRI-BBQ-110', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 158g', 'values' => ['Peso' => '158g'], 'sku' => 'PRI-BBQ-158', 'precio_ajuste' => 1.20, 'cantidad' => 80]]],
                    ['nombre' => 'Pringles Ranch', 'descripcion' => 'Papas fritas sabor ranch en presentación de 110g', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'PRI-RAN-110', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 158g', 'values' => ['Peso' => '158g'], 'sku' => 'PRI-RAN-158', 'precio_ajuste' => 1.20, 'cantidad' => 80]]],
                    ['nombre' => 'Pringles Pizza', 'descripcion' => 'Papas fritas sabor pizza en presentación de 110g', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'PRI-PIZ-110', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 158g', 'values' => ['Peso' => '158g'], 'sku' => 'PRI-PIZ-158', 'precio_ajuste' => 1.20, 'cantidad' => 80]]],
                    ['nombre' => 'Pringles Enchilada', 'descripcion' => 'Papas fritas sabor enchilada en presentación de 124g', 'precio_base' => 2.70, 'variantes' => [['nombre' => 'Peso: 124g', 'values' => ['Peso' => '124g'], 'sku' => 'PRI-ENC-124', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Pringles Habanero', 'descripcion' => 'Papas fritas sabor habanero en presentación de 124g', 'precio_base' => 2.70, 'variantes' => [['nombre' => 'Peso: 124g', 'values' => ['Peso' => '124g'], 'sku' => 'PRI-HAB-124', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Tom' => [
                    ['nombre' => 'Tostones con Ajo Tom', 'descripcion' => 'Deliciosos tostones con sabor a ajo', 'precio_base' => 1.43, 'variantes' => [['nombre' => 'Peso: 140gr', 'values' => ['Peso' => '140gr'], 'sku' => 'TOS-TOM-AG-140', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 80gr', 'values' => ['Peso' => '80gr'], 'sku' => 'TOS-TOM-AG-80', 'precio_ajuste' => -0.56, 'cantidad' => 100], ['nombre' => 'Peso: 28gr', 'values' => ['Peso' => '28gr'], 'sku' => 'TOS-TOM-AG-28', 'precio_ajuste' => -1.11, 'cantidad' => 100]]],
                    ['nombre' => 'Tostones Natural Tom', 'descripcion' => 'Tostones naturales sin condimentos', 'precio_base' => 1.43, 'variantes' => [['nombre' => 'Peso: 140gr', 'values' => ['Peso' => '140gr'], 'sku' => 'TOS-TOM-NAT-140', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 80gr', 'values' => ['Peso' => '80gr'], 'sku' => 'TOS-TOM-NAT-80', 'precio_ajuste' => -0.56, 'cantidad' => 100], ['nombre' => 'Peso: 28gr', 'values' => ['Peso' => '28gr'], 'sku' => 'TOS-TOM-NAT-28', 'precio_ajuste' => -1.11, 'cantidad' => 100]]],
                ],
                'Pin Pop' => [
                    ['nombre' => 'Pin Pop Bomba Gigante 24x1', 'descripcion' => 'Caramelos bomba gigante caja de 24 unidades', 'precio_base' => 12.50, 'variantes' => [['nombre' => 'Caja: 24 x 1 und', 'values' => ['Caja' => '24 x 1 und'], 'sku' => 'PIN-BOM-GIG-24X1', 'precio_ajuste' => 0, 'cantidad' => 30]]],
                    ['nombre' => 'Pin Pop Bomba Gigante 48x1', 'descripcion' => 'Caramelos bomba gigante caja de 48 unidades', 'precio_base' => 24.00, 'variantes' => [['nombre' => 'Caja: 48 x 1 und', 'values' => ['Caja' => '48 x 1 und'], 'sku' => 'PIN-BOM-GIG-48X1', 'precio_ajuste' => 0, 'cantidad' => 15]]],
                    ['nombre' => 'Pin Pop Bomba Surtidos', 'descripcion' => 'Caramelos bomba sabores surtidos', 'precio_base' => 8.50, 'variantes' => [['nombre' => 'Caja: Surtidos', 'values' => ['Caja' => 'Surtidos'], 'sku' => 'PIN-BOM-SUR', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                    ['nombre' => 'Pin Pop Gigante Mora Fresa 24x1', 'descripcion' => 'Caramelos gigantes sabor mora fresa caja de 24', 'precio_base' => 13.25, 'variantes' => [['nombre' => 'Caja: 24 x 1 und', 'values' => ['Caja' => '24 x 1 und'], 'sku' => 'PIN-GIG-MOR-FRE-24X1', 'precio_ajuste' => 0, 'cantidad' => 25]]],
                ],
                'Tam Cikolatam' => [
                    ['nombre' => 'Tam Cikolatam 83g', 'descripcion' => 'Chocolate Tam Cikolatam 83g', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Peso: 83g', 'values' => ['Peso' => '83g'], 'sku' => 'TAM-CIK-83', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                ],
            ],
            'Lavandería' => [
                'Tapa Amarilla' => [
                    ['nombre' => 'Cloro Tapa Amarilla Original', 'descripcion' => 'Cloro desinfectante original', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'CLO-TAP-ORI-1L', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Cloro Tapa Amarilla Lavanda', 'descripcion' => 'Cloro desinfectante aroma lavanda', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'CLO-TAP-LAV-1L', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Cloro Tapa Amarilla Limón', 'descripcion' => 'Cloro desinfectante aroma limón', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'CLO-TAP-LIM-1L', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Cloro Tapa Amarilla Floral', 'descripcion' => 'Cloro desinfectante aroma floral', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'CLO-TAP-FLO-1L', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Cloro Tapa Amarilla Manzana', 'descripcion' => 'Cloro desinfectante aroma manzana', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'CLO-TAP-MAN-1L', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Desinfectante Tapa Amarilla Pino', 'descripcion' => 'Desinfectante multiusos aroma pino', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'DES-TAP-PIN-1L', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                    ['nombre' => 'Desinfectante Tapa Amarilla Eucalipto', 'descripcion' => 'Desinfectante multiusos aroma eucalipto', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'DES-TAP-EUC-1L', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                ],
                'Alive' => [
                    ['nombre' => 'Alive Azul', 'descripcion' => 'Detergente para todo tipo de ropa', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-AZ-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ALV-AZ-1KG', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Alive Morado', 'descripcion' => 'Detergente Floral', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-MOR-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ALV-MOR-1KG', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Alive Verde', 'descripcion' => 'Detergente Limón', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-VER-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ALV-VER-1KG', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Alive Rosa', 'descripcion' => 'Detergente Suavizante', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-ROS-500', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Alive Blanqueador', 'descripcion' => 'Detergente con Blanqueador', 'precio_base' => 2.75, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-BLAN-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ALV-BLAN-1KG', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                ],
                'Oso Blanco' => [
                    ['nombre' => 'Oso Blanco Cloro Gel', 'descripcion' => 'Cloro Gel Desinfectante para Baño y Cocina', 'precio_base' => 1.75, 'variantes' => [['nombre' => 'Presentación: 500ml', 'values' => ['Presentación' => '500ml'], 'sku' => 'OSO-CLR-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-CLR-1LT', 'precio_ajuste' => 1.00, 'cantidad' => 80]]],
                    ['nombre' => 'Papel Higienico Luciano', 'descripcion' => 'Papel Higienico Luciano', 'precio_base' => 1.75, 'variantes' => [['nombre' => 'Presentación: Paquete de 4 rollos', 'values' => ['Presentación' => 'Paquete de 4 rollos'], 'sku' => 'OSO-CL-PAPER-500', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Oso Blanco Lavavajillas', 'descripcion' => 'Detergente para lavavajillas', 'precio_base' => 1.50, 'variantes' => [['nombre' => 'Presentación: 500ml', 'values' => ['Presentación' => '500ml'], 'sku' => 'OSO-LVJ-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-LVJ-1LT', 'precio_ajuste' => 1.00, 'cantidad' => 80]]],
                    ['nombre' => 'Oso Blanco Suavizante', 'descripcion' => 'Suavizante para telas', 'precio_base' => 2.00, 'variantes' => [['nombre' => 'Presentación: 900ml', 'values' => ['Presentación' => '900ml'], 'sku' => 'OSO-SUAV-900', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Presentación: 1800ml', 'values' => ['Presentación' => '1800ml'], 'sku' => 'OSO-SUAV-1.8', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Oso Blanco Multiusos', 'descripcion' => 'Limpiador multiusos', 'precio_base' => 1.75, 'variantes' => [['nombre' => 'Presentación: 900ml', 'values' => ['Presentación' => '900ml'], 'sku' => 'OSO-MULT-900', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Presentación: 1800ml', 'values' => ['Presentación' => '1800ml'], 'sku' => 'OSO-MULT-1.8', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Oso Blanco Desinfectante', 'descripcion' => 'Desinfectante clorado', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-DES-1LT', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Presentación: 2000ml', 'values' => ['Presentación' => '2000ml'], 'sku' => 'OSO-DES-2LT', 'precio_ajuste' => 1.75, 'cantidad' => 80]]],
                    ['nombre' => 'Oso Blanco Desinfectante Lavanda', 'descripcion' => 'Desinfectante clorado con fragancia a lavanda', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-DES-LAV', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Oso Blanco Desinfectante Manzana', 'descripcion' => 'Desinfectante clorado con fragancia a manzana', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-DES-MAN', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Oso Blanco Desinfectante Cherry', 'descripcion' => 'Desinfectante clorado con fragancia a cereza', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-DES-CHR', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Oso Blanco Desinfectante Fresca Sensación', 'descripcion' => 'Desinfectante clorado con fragancia fresca', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-DES-FRSC', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Desinfectante Oso Blanco Intensa Floral', 'descripcion' => 'Desinfectante con fragancia intensa floral', 'precio_base' => 2.35, 'variantes' => [['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-DES-INT-FLO', 'precio_ajuste' => 0, 'cantidad' => 95]]],
                    ['nombre' => 'Desinfectante Oso Blanco Manzana Verde', 'descripcion' => 'Desinfectante con fragancia a manzana verde', 'precio_base' => 2.30, 'variantes' => [['nombre' => 'Presentación: 1000ml', 'values' => ['Presentación' => '1000ml'], 'sku' => 'OSO-DES-MAN-VER', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Alive' => [
                    ['nombre' => 'Alive Azul', 'descripcion' => 'Detergente para todo tipo de ropa', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-AZ-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ALV-AZ-1KG', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Alive Morado', 'descripcion' => 'Detergente Floral', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-MOR-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ALV-MOR-1KG', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Alive Verde', 'descripcion' => 'Detergente Limón', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-VER-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ALV-VER-1KG', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Alive Rosa', 'descripcion' => 'Detergente Suavizante', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-ROS-500', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Alive Blanqueador', 'descripcion' => 'Detergente con Blanqueador', 'precio_base' => 2.75, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-BLAN-500', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Peso: 1kg', 'values' => ['Peso' => '1kg'], 'sku' => 'ALV-BLAN-1KG', 'precio_ajuste' => 1.50, 'cantidad' => 80]]],
                    ['nombre' => 'Alive Blanco', 'descripcion' => 'Detergente para ropa blanca', 'precio_base' => 2.60, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-BLA-500', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Alive Rosado', 'descripcion' => 'Detergente rosado suavizante', 'precio_base' => 2.55, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'ALV-ROSA-500', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Alive Toallas Femeninas', 'descripcion' => 'Toallas sanitarias femeninas', 'precio_base' => 3.25, 'variantes' => [['nombre' => 'Cantidad: 10 und', 'values' => ['Cantidad' => '10 und'], 'sku' => 'ALV-TOA-FEM-10', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Toallas Alive Nocturnas', 'descripcion' => 'Toallas sanitarias nocturnas', 'precio_base' => 3.45, 'variantes' => [['nombre' => 'Cantidad: 8 und', 'values' => ['Cantidad' => '8 und'], 'sku' => 'TOA-ALV-NOC-8', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Toallas Alive Diaria', 'descripcion' => 'Toallas sanitarias de uso diario', 'precio_base' => 2.95, 'variantes' => [['nombre' => 'Cantidad: 15 und', 'values' => ['Cantidad' => '15 und'], 'sku' => 'TOA-ALV-DIA-15', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                ],
                'Aromax' => [
                    ['nombre' => 'Limpiapisos en Polvo Aromax', 'descripcion' => 'Limpiador de pisos en polvo', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Peso: 500g', 'values' => ['Peso' => '500g'], 'sku' => 'LIM-ARO-500', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                ],
                'Suavitel' => [
                    ['nombre' => 'Suavitel Cuidado Superior Frescura de Manzana', 'descripcion' => 'Suavizante con fragancia a manzana', 'precio_base' => 3.15, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'SUA-MAN-900', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Suavitel Cuidado Superior Frescura de Primavera', 'descripcion' => 'Suavizante con fragancia primaveral', 'precio_base' => 3.15, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'SUA-PRI-900', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Suavitel Cuidado Superior Fresa y Chocolate', 'descripcion' => 'Suavizante con fragancia a fresa y chocolate', 'precio_base' => 3.25, 'variantes' => [['nombre' => 'Volumen: 900ml', 'values' => ['Volumen' => '900ml'], 'sku' => 'SUA-FRE-CHO-900', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                ],
                'Blue Softy' => [
                    ['nombre' => 'Servilletas Blue Softy', 'descripcion' => 'Servilletas de papel suaves', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Cantidad: 100 und', 'values' => ['Cantidad' => '100 und'], 'sku' => 'SER-BLU-100', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                ],
                'Tapa Amarilla' => [
                    ['nombre' => 'Cloro Limpieza Extrema 1L', 'descripcion' => 'Cloro de limpieza extrema', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'CLO-LIM-EXT-1L', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                ],
            ],
            'Cigarrilos' => [
                'Chimo' => [
                    ['nombre' => 'Chimo el apureño', 'descripcion' => 'Chimo con extracto de tabaco y sabores naturales.', 'precio_base' => 1.43, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CHI-APU-1', 'precio_ajuste' => 0, 'cantidad' => 1000]]],
                    ['nombre' => 'Chimo El Tigrito Amarillo', 'descripcion' => 'Chimo El Tigrito Amarillo', 'precio_base' => 0.51, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CHI-TIG-AMA', 'precio_ajuste' => 0, 'cantidad' => 200]]],
                ],
                'Lucky Strike' => [
                    ['nombre' => 'Cigarrillos Nova Lucky Strike', 'descripcion' => 'Cigarrillos Nova Lucky Strike', 'precio_base' => 3.65, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CIG-LUC-NOV', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Cigarrillos Lucky Strike Red', 'descripcion' => 'Cigarrillos Lucky Strike Red', 'precio_base' => 3.39, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CIG-LUC-RED', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Cigarros Eclipse Lucky Strike', 'descripcion' => 'Cigarros Eclipse Lucky Strike', 'precio_base' => 3.73, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CIG-LUC-ECL', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Belmont' => [
                    ['nombre' => 'Cigarros 10und Belmont', 'descripcion' => 'Cigarros Belmont 10 unidades', 'precio_base' => 1.54, 'variantes' => [['nombre' => 'Cantidad: 10 und', 'values' => ['Cantidad' => '10 und'], 'sku' => 'CIG-BEL-10', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Cigarros 20und Belmont', 'descripcion' => 'Cigarros Belmont 20 unidades', 'precio_base' => 3.07, 'variantes' => [['nombre' => 'Cantidad: 20 und', 'values' => ['Cantidad' => '20 und'], 'sku' => 'CIG-BEL-20', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Pall Mall' => [
                    ['nombre' => 'Cigarrillos Grandes Pall Mall', 'descripcion' => 'Cigarrillos Grandes Pall Mall', 'precio_base' => 3.05, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CIG-PAL-GRAN', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Viceroy' => [
                    ['nombre' => 'Cigarros Viceroy', 'descripcion' => 'Cigarros Viceroy', 'precio_base' => 2.12, 'variantes' => [['nombre' => 'Cantidad: 1 und', 'values' => ['Cantidad' => '1 und'], 'sku' => 'CIG-VIC-UND', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Universal' => [
                    ['nombre' => 'Cigarros 20und Universal', 'descripcion' => 'Cigarros Universal 20 unidades', 'precio_base' => 2.09, 'variantes' => [['nombre' => 'Cantidad: 20 und', 'values' => ['Cantidad' => '20 und'], 'sku' => 'CIG-UNI-20', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Cigarrillos Cosmic 20und', 'descripcion' => 'Cigarrillos Cosmic 20 unidades', 'precio_base' => 3.73, 'variantes' => [['nombre' => 'Cantidad: 20 und', 'values' => ['Cantidad' => '20 und'], 'sku' => 'CIG-COS-20', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Marlboro' => [
                    ['nombre' => 'Marlboro Red 20und', 'descripcion' => 'Cigarrillos Marlboro Red 20 unidades', 'precio_base' => 4.25, 'variantes' => [['nombre' => 'Cantidad: 20 und', 'values' => ['Cantidad' => '20 und'], 'sku' => 'MAR-RED-20', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Marlboro Gold 20und', 'descripcion' => 'Cigarrillos Marlboro Gold 20 unidades', 'precio_base' => 4.25, 'variantes' => [['nombre' => 'Cantidad: 20 und', 'values' => ['Cantidad' => '20 und'], 'sku' => 'MAR-GOL-20', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Marlboro Menthol 20und', 'descripcion' => 'Cigarrillos Marlboro Menthol 20 unidades', 'precio_base' => 4.25, 'variantes' => [['nombre' => 'Cantidad: 20 und', 'values' => ['Cantidad' => '20 und'], 'sku' => 'MAR-MEN-20', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
                'Camel' => [
                    ['nombre' => 'Camel Blue 20und', 'descripcion' => 'Cigarrillos Camel Blue 20 unidades', 'precio_base' => 3.95, 'variantes' => [['nombre' => 'Cantidad: 20 und', 'values' => ['Cantidad' => '20 und'], 'sku' => 'CAM-BLU-20', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Camel Yellow 20und', 'descripcion' => 'Cigarrillos Camel Yellow 20 unidades', 'precio_base' => 3.95, 'variantes' => [['nombre' => 'Cantidad: 20 und', 'values' => ['Cantidad' => '20 und'], 'sku' => 'CAM-YEL-20', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
            ],
             'Higiene Personal' => [
                'Nutribela' => [
                    [
                        'nombre' => 'Nutribela Biokeratina',
                        'descripcion' => 'Tratamiento capilar Biokeratina 300ml',
                        'precio_base' => 18.40,
                        'variantes' => [
                            ['nombre' => 'Presentación: 300ml', 'values' => ['Presentación' => '300ml'], 'sku' => 'NUT-BIO-300', 'precio_ajuste' => 0, 'cantidad' => 50],
                            ['nombre' => 'Presentación: 12 sobres', 'values' => ['Presentación' => '12 sobres'], 'sku' => 'NUT-BIO-SOB', 'precio_ajuste' => -1.90, 'cantidad' => 100],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribela Enzimoterapia',
                        'descripcion' => 'Tratamiento capilar Enzimoterapia 300ml',
                        'precio_base' => 18.40,
                        'variantes' => [
                            ['nombre' => 'Presentación: 300ml', 'values' => ['Presentación' => '300ml'], 'sku' => 'NUT-ENZ-300', 'precio_ajuste' => 0, 'cantidad' => 50],
                            ['nombre' => 'Presentación: 12 sobres', 'values' => ['Presentación' => '12 sobres'], 'sku' => 'NUT-ENZ-SOB', 'precio_ajuste' => -1.90, 'cantidad' => 100],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribela Células Madres',
                        'descripcion' => 'Tratamiento capilar Células Madres 300ml',
                        'precio_base' => 18.40,
                        'variantes' => [
                            ['nombre' => 'Presentación: 300ml', 'values' => ['Presentación' => '300ml'], 'sku' => 'NUT-CEL-300', 'precio_ajuste' => 0, 'cantidad' => 50],
                            ['nombre' => 'Presentación: 12 sobres', 'values' => ['Presentación' => '12 sobres'], 'sku' => 'NUT-CEL-SOB', 'precio_ajuste' => -1.90, 'cantidad' => 100],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribela Pro Hialurónico',
                        'descripcion' => 'Tratamiento capilar Pro Hialurónico 300ml',
                        'precio_base' => 18.40,
                        'variantes' => [
                            ['nombre' => 'Presentación: 300ml', 'values' => ['Presentación' => '300ml'], 'sku' => 'NUT-HIA-300', 'precio_ajuste' => 0, 'cantidad' => 50],
                            ['nombre' => 'Presentación: 12 sobres', 'values' => ['Presentación' => '12 sobres'], 'sku' => 'NUT-HIA-SOB', 'precio_ajuste' => -1.90, 'cantidad' => 100],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribela Termoprotección',
                        'descripcion' => 'Tratamiento capilar Termoprotección 300ml',
                        'precio_base' => 18.40,
                        'variantes' => [
                            ['nombre' => 'Presentación: 300ml', 'values' => ['Presentación' => '300ml'], 'sku' => 'NUT-TER-300', 'precio_ajuste' => 0, 'cantidad' => 50],
                            ['nombre' => 'Presentación: 12 sobres', 'values' => ['Presentación' => '12 sobres'], 'sku' => 'NUT-TER-SOB', 'precio_ajuste' => -1.90, 'cantidad' => 100],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribela Reparación Intensiva',
                        'descripcion' => 'Tratamiento capilar Reparación Intensiva 300ml',
                        'precio_base' => 18.40,
                        'variantes' => [
                            ['nombre' => 'Presentación: 300ml', 'values' => ['Presentación' => '300ml'], 'sku' => 'NUT-REP-300', 'precio_ajuste' => 0, 'cantidad' => 50],
                            ['nombre' => 'Presentación: 12 sobres', 'values' => ['Presentación' => '12 sobres'], 'sku' => 'NUT-REP-SOB', 'precio_ajuste' => -1.90, 'cantidad' => 100],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribela Nutrición Avanzada',
                        'descripcion' => 'Tratamiento capilar Nutrición Avanzada 300ml',
                        'precio_base' => 18.40,
                        'variantes' => [
                            ['nombre' => 'Presentación: 300ml', 'values' => ['Presentación' => '300ml'], 'sku' => 'NUT-NUT-300', 'precio_ajuste' => 0, 'cantidad' => 50],
                            ['nombre' => 'Presentación: 12 sobres', 'values' => ['Presentación' => '12 sobres'], 'sku' => 'NUT-NUT-SOB', 'precio_ajuste' => -1.90, 'cantidad' => 100],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribela Repolarización en Frío',
                        'descripcion' => 'Tratamiento capilar Repolarización en Frío 300ml',
                        'precio_base' => 18.40,
                        'variantes' => [
                            ['nombre' => 'Presentación: 300ml', 'values' => ['Presentación' => '300ml'], 'sku' => 'NUT-REF-300', 'precio_ajuste' => 0, 'cantidad' => 50],
                            ['nombre' => 'Presentación: 12 sobres', 'values' => ['Presentación' => '12 sobres'], 'sku' => 'NUT-REF-SOB', 'precio_ajuste' => -1.90, 'cantidad' => 100],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Acondicionador 12x15ml Con células madres',
                        'descripcion' => 'Acondicionador con células madres 12 sobres de 15ml',
                        'precio_base' => 16.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 15ml', 'values' => ['Presentación' => '12 x 15ml'], 'sku' => 'NUT15-ACO-CEL-12X15', 'precio_ajuste' => 0, 'cantidad' => 80],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Shampoo 12x15ml Reparación Intensiva',
                        'descripcion' => 'Shampoo reparación intensiva 12 sobres de 15ml',
                        'precio_base' => 15.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 15ml', 'values' => ['Presentación' => '12 x 15ml'], 'sku' => 'NUT15-SHA-REP-12X15', 'precio_ajuste' => 0, 'cantidad' => 85],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Tratamiento Capilar 12x1',
                        'descripcion' => 'Tratamiento capilar 12 unidades',
                        'precio_base' => 18.75,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 1 und', 'values' => ['Presentación' => '12 x 1 und'], 'sku' => 'NUT15-TRA-CAP-12X1', 'precio_ajuste' => 0, 'cantidad' => 60],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Acondicionador 12x15ml Enzimoterapia',
                        'descripcion' => 'Acondicionador enzimoterapia 12 sobres de 15ml',
                        'precio_base' => 16.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 15ml', 'values' => ['Presentación' => '12 x 15ml'], 'sku' => 'NUT15-ACO-ENZ-12X15', 'precio_ajuste' => 0, 'cantidad' => 80],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Acondicionador 12x15ml Pro hialurónico',
                        'descripcion' => 'Acondicionador pro hialurónico 12 sobres de 15ml',
                        'precio_base' => 16.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 15ml', 'values' => ['Presentación' => '12 x 15ml'], 'sku' => 'NUT15-ACO-HIA-12X15', 'precio_ajuste' => 0, 'cantidad' => 80],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Acondicionador 12x15ml Reparación Intensiva',
                        'descripcion' => 'Acondicionador reparación intensiva 12 sobres de 15ml',
                        'precio_base' => 16.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 15ml', 'values' => ['Presentación' => '12 x 15ml'], 'sku' => 'NUT15-ACO-REP-12X15', 'precio_ajuste' => 0, 'cantidad' => 80],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Acondicionador 12x18ml Pro hialurónico',
                        'descripcion' => 'Acondicionador pro hialurónico 12 sobres de 18ml',
                        'precio_base' => 17.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 18ml', 'values' => ['Presentación' => '12 x 18ml'], 'sku' => 'NUT15-ACO-HIA-12X18', 'precio_ajuste' => 0, 'cantidad' => 75],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Acondicionador Nutrición 12x1',
                        'descripcion' => 'Acondicionador nutrición 12 unidades',
                        'precio_base' => 18.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 1 und', 'values' => ['Presentación' => '12 x 1 und'], 'sku' => 'NUT15-ACO-NUT-12X1', 'precio_ajuste' => 0, 'cantidad' => 60],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Acondicionador Termoprotección 12x1',
                        'descripcion' => 'Acondicionador termoprotección 12 unidades',
                        'precio_base' => 18.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 1 und', 'values' => ['Presentación' => '12 x 1 und'], 'sku' => 'NUT15-ACO-TER-12X1', 'precio_ajuste' => 0, 'cantidad' => 60],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Shampoo 12x18ml Enzimoterapia',
                        'descripcion' => 'Shampoo enzimoterapia 12 sobres de 18ml',
                        'precio_base' => 16.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 18ml', 'values' => ['Presentación' => '12 x 18ml'], 'sku' => 'NUT15-SHA-ENZ-12X18', 'precio_ajuste' => 0, 'cantidad' => 80],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Shampoo Nutrición 12x1',
                        'descripcion' => 'Shampoo nutrición 12 unidades',
                        'precio_base' => 17.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 1 und', 'values' => ['Presentación' => '12 x 1 und'], 'sku' => 'NUT15-SHA-NUT-12X1', 'precio_ajuste' => 0, 'cantidad' => 70],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Shampoo Termoprotección 12x1',
                        'descripcion' => 'Shampoo termoprotección 12 unidades',
                        'precio_base' => 17.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 1 und', 'values' => ['Presentación' => '12 x 1 und'], 'sku' => 'NUT15-SHA-TER-12X1', 'precio_ajuste' => 0, 'cantidad' => 70],
                        ]
                    ],
                    [
                        'nombre' => 'Nutribella15 Tratamiento Capilar 12x24ml Enzimoterapia',
                        'descripcion' => 'Tratamiento capilar enzimoterapia 12 sobres de 24ml',
                        'precio_base' => 19.50,
                        'variantes' => [
                            ['nombre' => 'Presentación: 12 x 24ml', 'values' => ['Presentación' => '12 x 24ml'], 'sku' => 'NUT15-TRA-ENZ-12X24', 'precio_ajuste' => 0, 'cantidad' => 55],
                        ]
                    ],
                ],
                'Alident' => [
                    ['nombre' => 'Alident Gel Triple Acción', 'descripcion' => 'Pasta dental gel triple acción', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Peso: 100ml', 'values' => ['Peso' => '100ml'], 'sku' => 'ALI-GEL-TRI-100', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Alident Gel Verde Azul', 'descripcion' => 'Pasta dental gel verde azul', 'precio_base' => 2.75, 'variantes' => [['nombre' => 'Peso: 100ml', 'values' => ['Peso' => '100ml'], 'sku' => 'ALI-GEL-VER-AZ-100', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Alident Gel Verde Fresco', 'descripcion' => 'Pasta dental gel verde fresco', 'precio_base' => 2.75, 'variantes' => [['nombre' => 'Peso: 100ml', 'values' => ['Peso' => '100ml'], 'sku' => 'ALI-GEL-VER-FRE-100', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                ],
                'Colgate' => [
                    ['nombre' => 'Colgate 12 x 90g', 'descripcion' => 'Pasta dental Colgate caja de 12 tubos de 90g', 'precio_base' => 28.50, 'variantes' => [['nombre' => 'Caja: 12 x 90g', 'values' => ['Caja' => '12 x 90g'], 'sku' => 'COL-12X90', 'precio_ajuste' => 0, 'cantidad' => 20]]],
                ],
                'Lady Speed Stick' => [
                    ['nombre' => 'Lady Speed Stick 18g', 'descripcion' => 'Desodorante femenino en barra', 'precio_base' => 3.45, 'variantes' => [['nombre' => 'Peso: 18g', 'values' => ['Peso' => '18g'], 'sku' => 'LAD-SPE-18', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                ],
                'Speed Stick' => [
                    ['nombre' => 'Speed Stick 18g', 'descripcion' => 'Desodorante masculino en barra', 'precio_base' => 3.25, 'variantes' => [['nombre' => 'Peso: 18g', 'values' => ['Peso' => '18g'], 'sku' => 'SPE-STI-18', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                ],
                'Anita' => [
                    ['nombre' => 'Jabón Anita Rosado', 'descripcion' => 'Jabón de tocador rosado', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'JAB-ANI-ROS-110', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Frita' => [
                    ['nombre' => 'Jabón Frita Azul', 'descripcion' => 'Jabón de tocador azul', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'JAB-FRI-AZ-110', 'precio_ajuste' => 0, 'cantidad' => 95]]],
                ],
            ],
            'Bebidas' => [
                'Coca Cola' => [
                    ['nombre' => 'Coca Cola Original 350ml', 'descripcion' => 'Bebida gaseosa sabor cola', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'COC-ORI-350', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Coca Cola Original 355ml', 'descripcion' => 'Bebida gaseosa sabor cola lata', 'precio_base' => 1.30, 'variantes' => [['nombre' => 'Volumen: 355ml', 'values' => ['Volumen' => '355ml'], 'sku' => 'COC-ORI-355', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                    ['nombre' => 'Coca Cola Original 500ml', 'descripcion' => 'Bebida gaseosa sabor cola', 'precio_base' => 1.75, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'COC-ORI-500', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Coca Cola Zero 350ml', 'descripcion' => 'Bebida gaseosa sin azúcar', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'COC-ZER-350', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Refresco Coca Cola', 'descripcion' => 'Refresco Coca Cola', 'precio_base' => 1.35, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-COC-COL-350', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-COC-COL-500', 'precio_ajuste' => 0.35, 'cantidad' => 100], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-COC-COL-1L', 'precio_ajuste' => 0.75, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-COC-COL-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-COC-COL-2L', 'precio_ajuste' => 1.55, 'cantidad' => 50]]],
                ],
                'Pepsi' => [
                    ['nombre' => 'Pepsi Original 350ml', 'descripcion' => 'Bebida gaseosa sabor cola', 'precio_base' => 1.20, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'PEP-ORI-350', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Pepsi Original 500ml', 'descripcion' => 'Bebida gaseosa sabor cola', 'precio_base' => 1.70, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'PEP-ORI-500', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Refresco Pepsi Cola', 'descripcion' => 'Refresco Pepsi Cola', 'precio_base' => 1.30, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-PEP-COL-350', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-PEP-COL-500', 'precio_ajuste' => 0.35, 'cantidad' => 100], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-PEP-COL-1L', 'precio_ajuste' => 0.75, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-PEP-COL-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-PEP-COL-2L', 'precio_ajuste' => 1.55, 'cantidad' => 50]]],
                ],
                'Sprite' => [
                    ['nombre' => 'Sprite Original 350ml', 'descripcion' => 'Bebida gaseosa sabor limón', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'SPR-ORI-350', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Sprite Original 500ml', 'descripcion' => 'Bebida gaseosa sabor limón', 'precio_base' => 1.75, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'SPR-ORI-500', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Fanta' => [
                    ['nombre' => 'Fanta Naranja 350ml', 'descripcion' => 'Bebida gaseosa sabor naranja', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'FAN-NAR-350', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Fanta Uva 350ml', 'descripcion' => 'Bebida gaseosa sabor uva', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'FAN-UVA-350', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                ],
                'Polar' => [
                    ['nombre' => 'Malta Retornable 36x222ml Maltil Polar', 'descripcion' => 'Malta en envase retornable caja de 36 unidades', 'precio_base' => 28.50, 'variantes' => [['nombre' => 'Caja: 36 x 222ml', 'values' => ['Caja' => '36 x 222ml'], 'sku' => 'MAL-POL-RET-36X222', 'precio_ajuste' => 0, 'cantidad' => 20]]],
                    ['nombre' => 'Malta Desechable 12x250ml Maltin Polar', 'descripcion' => 'Malta en envase desechable caja de 12 unidades', 'precio_base' => 12.75, 'variantes' => [['nombre' => 'Caja: 12 x 250ml', 'values' => ['Caja' => '12 x 250ml'], 'sku' => 'MAL-POL-DES-12X250', 'precio_ajuste' => 0, 'cantidad' => 30]]],
                    ['nombre' => 'Malta de Lata 24x330ml Maltin Polar', 'descripcion' => 'Malta en lata caja de 24 unidades', 'precio_base' => 22.80, 'variantes' => [['nombre' => 'Caja: 24 x 330ml', 'values' => ['Caja' => '24 x 330ml'], 'sku' => 'MAL-POL-LAT-24X330', 'precio_ajuste' => 0, 'cantidad' => 25]]],
                    ['nombre' => 'Malta 1.5Lt Maltin Polar', 'descripcion' => 'Malta en botella de 1.5 litros', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'MAL-POL-1.5L', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Maltin 355ml', 'descripcion' => 'Malta en lata de 355ml', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Volumen: 355ml', 'values' => ['Volumen' => '355ml'], 'sku' => 'MAL-POL-355', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Carabobo' => [
                    ['nombre' => 'Leche Líquida Carabobo Entera', 'descripcion' => 'Leche líquida entera pasteurizada', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'LEC-CAR-ENT-1L', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Leche Líquida Carabobo Descremada', 'descripcion' => 'Leche líquida descremada pasteurizada', 'precio_base' => 1.90, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'LEC-CAR-DESC-1L', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Leche Líquida Carabobo Semi-descremada', 'descripcion' => 'Leche líquida semi-descremada pasteurizada', 'precio_base' => 1.92, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'LEC-CAR-SEMI-1L', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
                'Frut Cola' => [
                    ['nombre' => 'Refresco Frut Cola 2L', 'descripcion' => 'Refresco sabor cola en botella de 2 litros', 'precio_base' => 1.65, 'variantes' => [['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-FRU-COL-2L', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                ],
                'Minalba' => [
                    ['nombre' => 'Agua Minalba 600ml', 'descripcion' => 'Agua mineral natural en botella de 600ml', 'precio_base' => 0.85, 'variantes' => [['nombre' => 'Volumen: 600ml', 'values' => ['Volumen' => '600ml'], 'sku' => 'AGU-MIN-600', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                ],
                'Cool' => [
                    ['nombre' => 'Agua Mineral Cool 1L', 'descripcion' => 'Agua mineral natural en botella de 1 litro', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'AGU-COO-1L', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Agua Mineral Cool 400ml', 'descripcion' => 'Agua mineral natural en botella de 400ml', 'precio_base' => 0.75, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'AGU-COO-400', 'precio_ajuste' => 0, 'cantidad' => 150]]],
                ],
                'Vista' => [
                    ['nombre' => 'Agua Vista 1.5L', 'descripcion' => 'Agua purificada en botella de 1.5 litros', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'AGU-VIS-1.5L', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Cristal' => [
                    ['nombre' => 'Agua Cristal 350ml', 'descripcion' => 'Agua purificada en botella de 350ml', 'precio_base' => 0.65, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'AGU-CRI-350', 'precio_ajuste' => 0, 'cantidad' => 150]]],
                    ['nombre' => 'Agua Cristal 500ml', 'descripcion' => 'Agua purificada en botella de 500ml', 'precio_base' => 0.80, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'AGU-CRI-500', 'precio_ajuste' => 0, 'cantidad' => 130]]],
                    ['nombre' => 'Agua Cristal 600ml', 'descripcion' => 'Agua purificada en botella de 600ml', 'precio_base' => 0.90, 'variantes' => [['nombre' => 'Volumen: 600ml', 'values' => ['Volumen' => '600ml'], 'sku' => 'AGU-CRI-600', 'precio_ajuste' => 0, 'cantidad' => 120]]],
                    ['nombre' => 'Agua Cristal 1L', 'descripcion' => 'Agua purificada en botella de 1 litro', 'precio_base' => 1.20, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'AGU-CRI-1L', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                    ['nombre' => 'Agua Cristal 1.5L', 'descripcion' => 'Agua purificada en botella de 1.5 litros', 'precio_base' => 1.50, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'AGU-CRI-1.5L', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Agua Cristal 2L', 'descripcion' => 'Agua purificada en botella de 2 litros', 'precio_base' => 1.80, 'variantes' => [['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'AGU-CRI-2L', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
                'Frubys' => [
                    ['nombre' => 'Frubys Manzana 12x1.5L', 'descripcion' => 'Jugo de manzana caja de 12 botellas de 1.5L', 'precio_base' => 18.50, 'variantes' => [['nombre' => 'Caja: 12 x 1.5L', 'values' => ['Caja' => '12 x 1.5L'], 'sku' => 'FRU-MAN-12X1.5', 'precio_ajuste' => 0, 'cantidad' => 15]]],
                    ['nombre' => 'Frubys Pera 12x1.5L', 'descripcion' => 'Jugo de pera caja de 12 botellas de 1.5L', 'precio_base' => 18.50, 'variantes' => [['nombre' => 'Caja: 12 x 1.5L', 'values' => ['Caja' => '12 x 1.5L'], 'sku' => 'FRU-PER-12X1.5', 'precio_ajuste' => 0, 'cantidad' => 15]]],
                ],
                'Frutea' => [
                    ['nombre' => 'Frutea Limón 1.5L', 'descripcion' => 'Té helado sabor limón', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'FRT-LIM-1.5L', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Frutea Durazno 1.5L', 'descripcion' => 'Té helado sabor durazno', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'FRT-DUR-1.5L', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                ],
                'Fruttsy' => [
                    ['nombre' => 'Fruttsy Passion 250ml', 'descripcion' => 'Bebida de fruta de la pasión', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Volumen: 250ml', 'values' => ['Volumen' => '250ml'], 'sku' => 'FTS-PAS-250', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Fruttsy Frutos Rojos 250ml', 'descripcion' => 'Bebida de frutos rojos', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Volumen: 250ml', 'values' => ['Volumen' => '250ml'], 'sku' => 'FTS-FRR-250', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Fruttsy Naranja 250ml', 'descripcion' => 'Bebida de naranja', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Volumen: 250ml', 'values' => ['Volumen' => '250ml'], 'sku' => 'FTS-NAR-250', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Jugo Fruttsy Frutos Rojos 24x250ml', 'descripcion' => 'Jugo frutos rojos caja de 24 unidades de 250ml', 'precio_base' => 24.00, 'variantes' => [['nombre' => 'Caja: 24 x 250ml', 'values' => ['Caja' => '24 x 250ml'], 'sku' => 'FTS-FRR-24X250', 'precio_ajuste' => 0, 'cantidad' => 15]]],
                ],
                'Frutys' => [
                    ['nombre' => 'Frutys Durazno 1.5L', 'descripcion' => 'Néctar de durazno', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'FRY-DUR-1.5L', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Frutys Naranja 1.5L', 'descripcion' => 'Néctar de naranja', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'FRY-NAR-1.5L', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Frutys Mandarina 1.5L', 'descripcion' => 'Néctar de mandarina', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'FRY-MAN-1.5L', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                ],
                'El Valle' => [
                    ['nombre' => 'Jugos El Valle Naranja 1.5L', 'descripcion' => 'Jugo de naranja natural', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'VAL-NAR-1.5L', 'precio_ajuste' => 0, 'cantidad' => 45]]],
                ],
                'Lipton' => [
                    ['nombre' => 'Lipton Ice Tea Durazno 500ml', 'descripcion' => 'Té helado sabor durazno', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'LIP-DUR-500', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Lipton Ice Tea Limón 500ml', 'descripcion' => 'Té helado sabor limón', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'LIP-LIM-500', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
                'Jucosa' => [
                    ['nombre' => 'Jucosa Refrescante Pera', 'descripcion' => 'Bebida refrescante sabor pera', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'JUC-PER-1L', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Jucosa Refrescante Manzana', 'descripcion' => 'Bebida refrescante sabor manzana', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'JUC-MAN-1L', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                ],
                'Natulac' => [
                    ['nombre' => 'Natulac Pera', 'descripcion' => 'Bebida láctea sabor pera', 'precio_base' => 1.75, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'NAT-PER-500', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                    ['nombre' => 'Jugo Natulac Manzana', 'descripcion' => 'Jugo de manzana', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'JUG-NAT-MAN-500', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Jugo Natulac Durazno', 'descripcion' => 'Jugo de durazno', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'JUG-NAT-DUR-500', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                ],
                'Valencia' => [
                    ['nombre' => 'Jugo Valencia Coctel', 'descripcion' => 'Jugo coctel de frutas', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'JUG-VAL-COC-1L', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Jugo Valencia Naranjada', 'descripcion' => 'Jugo de naranja', 'precio_base' => 2.15, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'JUG-VAL-NAR-1L', 'precio_ajuste' => 0, 'cantidad' => 55]]],
                    ['nombre' => 'Jugo Valencia Durazno', 'descripcion' => 'Jugo de durazno', 'precio_base' => 2.20, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'JUG-VAL-DUR-1L', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                    ['nombre' => 'Jugo Valencia Pera', 'descripcion' => 'Jugo de pera', 'precio_base' => 2.20, 'variantes' => [['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'JUG-VAL-PER-1L', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                ],
                'Justy' => [
                    ['nombre' => 'Jugo Justy Naranja', 'descripcion' => 'Jugo de naranja natural', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'JUG-JUS-NAR-500', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                ],
                'Speed' => [
                    ['nombre' => 'Jugo Speed', 'descripcion' => 'Bebida energética de frutas', 'precio_base' => 2.45, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'JUG-SPE-500', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Jugo Speed Lata', 'descripcion' => 'Bebida energética de frutas en lata', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Volumen: 355ml', 'values' => ['Volumen' => '355ml'], 'sku' => 'JUG-SPE-LAT-355', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Power Max' => [
                    ['nombre' => 'Power Max 400ml', 'descripcion' => 'Bebida energética', 'precio_base' => 2.45, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'POW-MAX-400', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                ],
                'Aldor' => [
                    ['nombre' => 'Aldor Tamarindo 100x1', 'descripcion' => 'Bebida de tamarindo caja de 100 unidades', 'precio_base' => 45.00, 'variantes' => [['nombre' => 'Caja: 100 x 1 und', 'values' => ['Caja' => '100 x 1 und'], 'sku' => 'ALD-TAM-100X1', 'precio_ajuste' => 0, 'cantidad' => 10]]],
                ],
                'Fruton' => [
                    ['nombre' => 'Jugo Fruton Fruit Punch 6x1.5L', 'descripcion' => 'Jugo fruit punch caja de 6 botellas de 1.5L', 'precio_base' => 12.50, 'variantes' => [['nombre' => 'Caja: 6 x 1.5L', 'values' => ['Caja' => '6 x 1.5L'], 'sku' => 'FRU-FRU-6X1.5', 'precio_ajuste' => 0, 'cantidad' => 25]]],
                    ['nombre' => 'Jugo Fruton Citrus Punch 6x1.5L', 'descripcion' => 'Jugo citrus punch caja de 6 botellas de 1.5L', 'precio_base' => 12.50, 'variantes' => [['nombre' => 'Caja: 6 x 1.5L', 'values' => ['Caja' => '6 x 1.5L'], 'sku' => 'FRU-CIT-6X1.5', 'precio_ajuste' => 0, 'cantidad' => 25]]],
                ],
                'Glup' => [
                    // Glup Naranja - Todas las presentaciones
                    ['nombre' => 'Glup Naranja', 'descripcion' => 'Refresco sabor naranja', 'precio_base' => 0.95, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-NAR-400', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-NAR-1L', 'precio_ajuste' => 0.85, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-NAR-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-NAR-2L', 'precio_ajuste' => 1.65, 'cantidad' => 50]]],
                    // Glup Limón - Todas las presentaciones
                    ['nombre' => 'Glup Limón', 'descripcion' => 'Refresco sabor limón', 'precio_base' => 0.95, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-LIM-400', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-LIM-1L', 'precio_ajuste' => 0.85, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-LIM-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-LIM-2L', 'precio_ajuste' => 1.65, 'cantidad' => 50]]],
                    // Glup Uva - Todas las presentaciones
                    ['nombre' => 'Glup Uva', 'descripcion' => 'Refresco sabor uva', 'precio_base' => 0.95, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-UVA-400', 'precio_ajuste' => 0, 'cantidad' => 115], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-UVA-1L', 'precio_ajuste' => 0.85, 'cantidad' => 75], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-UVA-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 55], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-UVA-2L', 'precio_ajuste' => 1.65, 'cantidad' => 45]]],
                    // Glup Manzana - Todas las presentaciones
                    ['nombre' => 'Glup Manzana', 'descripcion' => 'Refresco sabor manzana', 'precio_base' => 0.95, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-MAN-400', 'precio_ajuste' => 0, 'cantidad' => 115], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-MAN-1L', 'precio_ajuste' => 0.85, 'cantidad' => 75], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-MAN-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 55], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-MAN-2L', 'precio_ajuste' => 1.65, 'cantidad' => 45]]],
                    // Glup Tropical - Todas las presentaciones
                    ['nombre' => 'Glup Tropical', 'descripcion' => 'Refresco sabor tropical', 'precio_base' => 1.00, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-TRO-400', 'precio_ajuste' => 0, 'cantidad' => 110], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-TRO-1L', 'precio_ajuste' => 0.85, 'cantidad' => 70], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-TRO-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 50], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-TRO-2L', 'precio_ajuste' => 1.65, 'cantidad' => 40]]],
                    // Glup Fresa - Todas las presentaciones
                    ['nombre' => 'Glup Fresa', 'descripcion' => 'Refresco sabor fresa', 'precio_base' => 1.00, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-FRE-400', 'precio_ajuste' => 0, 'cantidad' => 110], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-FRE-1L', 'precio_ajuste' => 0.85, 'cantidad' => 70], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-FRE-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 50], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-FRE-2L', 'precio_ajuste' => 1.65, 'cantidad' => 40]]],
                    // Glup Cola Negra - Todas las presentaciones
                    ['nombre' => 'Glup Cola Negra', 'descripcion' => 'Refresco sabor cola negra', 'precio_base' => 1.05, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-COL-400', 'precio_ajuste' => 0, 'cantidad' => 105], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-COL-1L', 'precio_ajuste' => 0.85, 'cantidad' => 65], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-COL-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 45], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-COL-2L', 'precio_ajuste' => 1.65, 'cantidad' => 35]]],
                    // Glup Colita - Todas las presentaciones
                    ['nombre' => 'Glup Colita', 'descripcion' => 'Refresco sabor colita', 'precio_base' => 1.05, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-COLI-400', 'precio_ajuste' => 0, 'cantidad' => 105], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-COLI-1L', 'precio_ajuste' => 0.85, 'cantidad' => 65], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-COLI-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 45], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-COLI-2L', 'precio_ajuste' => 1.65, 'cantidad' => 35]]],
                    // Glup Fresh - Todas las presentaciones
                    ['nombre' => 'Glup Fresh', 'descripcion' => 'Refresco sabor fresh', 'precio_base' => 1.05, 'variantes' => [['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'GLP-FRESH-400', 'precio_ajuste' => 0, 'cantidad' => 105], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GLP-FRESH-1L', 'precio_ajuste' => 0.85, 'cantidad' => 65], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'GLP-FRESH-1.5L', 'precio_ajuste' => 1.25, 'cantidad' => 45], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'GLP-FRESH-2L', 'precio_ajuste' => 1.65, 'cantidad' => 35]]],
                ],
                'Gatorade' => [
                    ['nombre' => 'Gatorade Azul', 'descripcion' => 'Bebida deportiva sabor azul', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'GAT-AZU-500', 'precio_ajuste' => 0, 'cantidad' => 60], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GAT-AZU-1L', 'precio_ajuste' => 1.15, 'cantidad' => 40]]],
                    ['nombre' => 'Gatorade Mandarina', 'descripcion' => 'Bebida deportiva sabor mandarina', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'GAT-MAN-500', 'precio_ajuste' => 0, 'cantidad' => 60], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GAT-MAN-1L', 'precio_ajuste' => 1.15, 'cantidad' => 40]]],
                    ['nombre' => 'Gatorade Tropical', 'descripcion' => 'Bebida deportiva sabor tropical', 'precio_base' => 2.85, 'variantes' => [['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'GAT-TRO-500', 'precio_ajuste' => 0, 'cantidad' => 60], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'GAT-TRO-1L', 'precio_ajuste' => 1.15, 'cantidad' => 40]]],
                ],
                'Mi Brisa' => [
                    ['nombre' => 'Agua Mineral Mi Brisa', 'descripcion' => 'Agua mineral natural', 'precio_base' => 0.75, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'AGU-MIB-350', 'precio_ajuste' => 0, 'cantidad' => 150], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'AGU-MIB-500', 'precio_ajuste' => 0.15, 'cantidad' => 130], ['nombre' => 'Volumen: 600ml', 'values' => ['Volumen' => '600ml'], 'sku' => 'AGU-MIB-600', 'precio_ajuste' => 0.25, 'cantidad' => 120], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'AGU-MIB-1L', 'precio_ajuste' => 0.55, 'cantidad' => 100], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'AGU-MIB-1.5L', 'precio_ajuste' => 0.85, 'cantidad' => 80], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'AGU-MIB-2L', 'precio_ajuste' => 1.15, 'cantidad' => 70]]],
                ],
                'Heal' => [
                    ['nombre' => 'Agua Mineral Heal', 'descripcion' => 'Agua mineral purificada', 'precio_base' => 0.80, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'AGU-HEA-350', 'precio_ajuste' => 0, 'cantidad' => 150], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'AGU-HEA-500', 'precio_ajuste' => 0.15, 'cantidad' => 130], ['nombre' => 'Volumen: 600ml', 'values' => ['Volumen' => '600ml'], 'sku' => 'AGU-HEA-600', 'precio_ajuste' => 0.25, 'cantidad' => 120], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'AGU-HEA-1L', 'precio_ajuste' => 0.55, 'cantidad' => 100], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'AGU-HEA-1.5L', 'precio_ajuste' => 0.85, 'cantidad' => 80], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'AGU-HEA-2L', 'precio_ajuste' => 1.15, 'cantidad' => 70]]],
                ],
                'Golden' => [
                    ['nombre' => 'Refresco Golden Colita', 'descripcion' => 'Refresco sabor colita', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-GOL-COL-350', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-GOL-COL-500', 'precio_ajuste' => 0.35, 'cantidad' => 100], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-GOL-COL-1L', 'precio_ajuste' => 0.75, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-GOL-COL-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-GOL-COL-2L', 'precio_ajuste' => 1.55, 'cantidad' => 50]]],
                    ['nombre' => 'Refresco Golden Piña', 'descripcion' => 'Refresco sabor piña', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-GOL-PIN-350', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-GOL-PIN-500', 'precio_ajuste' => 0.35, 'cantidad' => 100], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-GOL-PIN-1L', 'precio_ajuste' => 0.75, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-GOL-PIN-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-GOL-PIN-2L', 'precio_ajuste' => 1.55, 'cantidad' => 50]]],
                    ['nombre' => 'Refresco Golden Manzana', 'descripcion' => 'Refresco sabor manzana', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-GOL-MAN-350', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-GOL-MAN-500', 'precio_ajuste' => 0.35, 'cantidad' => 100], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-GOL-MAN-1L', 'precio_ajuste' => 0.75, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-GOL-MAN-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-GOL-MAN-2L', 'precio_ajuste' => 1.55, 'cantidad' => 50]]],
                    ['nombre' => 'Refresco Golden 7UP', 'descripcion' => 'Refresco sabor lima limón', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-GOL-7UP-350', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-GOL-7UP-500', 'precio_ajuste' => 0.35, 'cantidad' => 100], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-GOL-7UP-1L', 'precio_ajuste' => 0.75, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-GOL-7UP-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-GOL-7UP-2L', 'precio_ajuste' => 1.55, 'cantidad' => 50]]],
                ],
                'Frescolita' => [
                    ['nombre' => 'Refresco Frescolita', 'descripcion' => 'Refresco sabor fresa', 'precio_base' => 1.35, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-FRE-350', 'precio_ajuste' => 0, 'cantidad' => 110], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-FRE-500', 'precio_ajuste' => 0.35, 'cantidad' => 90], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-FRE-1L', 'precio_ajuste' => 0.75, 'cantidad' => 70], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-FRE-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 50], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-FRE-2L', 'precio_ajuste' => 1.55, 'cantidad' => 40]]],
                ],
                'Chinotto' => [
                    ['nombre' => 'Refresco Chinotto', 'descripcion' => 'Refresco sabor chinotto', 'precio_base' => 1.45, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-CHI-350', 'precio_ajuste' => 0, 'cantidad' => 100], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-CHI-500', 'precio_ajuste' => 0.35, 'cantidad' => 80], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-CHI-1L', 'precio_ajuste' => 0.75, 'cantidad' => 60], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-CHI-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 45], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-CHI-2L', 'precio_ajuste' => 1.55, 'cantidad' => 35]]],
                ],
                'Hit' => [
                    ['nombre' => 'Refresco Hit Naranja', 'descripcion' => 'Refresco sabor naranja', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-HIT-NAR-350', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-HIT-NAR-500', 'precio_ajuste' => 0.35, 'cantidad' => 100], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-HIT-NAR-1L', 'precio_ajuste' => 0.75, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-HIT-NAR-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-HIT-NAR-2L', 'precio_ajuste' => 1.55, 'cantidad' => 50]]],
                    ['nombre' => 'Refresco Hit Uva', 'descripcion' => 'Refresco sabor uva', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Volumen: 350ml', 'values' => ['Volumen' => '350ml'], 'sku' => 'REF-HIT-UVA-350', 'precio_ajuste' => 0, 'cantidad' => 120], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'REF-HIT-UVA-500', 'precio_ajuste' => 0.35, 'cantidad' => 100], ['nombre' => 'Volumen: 1L', 'values' => ['Volumen' => '1L'], 'sku' => 'REF-HIT-UVA-1L', 'precio_ajuste' => 0.75, 'cantidad' => 80], ['nombre' => 'Volumen: 1.5L', 'values' => ['Volumen' => '1.5L'], 'sku' => 'REF-HIT-UVA-1.5L', 'precio_ajuste' => 1.15, 'cantidad' => 60], ['nombre' => 'Volumen: 2L', 'values' => ['Volumen' => '2L'], 'sku' => 'REF-HIT-UVA-2L', 'precio_ajuste' => 1.55, 'cantidad' => 50]]],
                ],
            ],
            'Snacks' => [
                'Doritos' => [
                    ['nombre' => 'Doritos Nacho Cheese', 'descripcion' => 'Tortilla chips sabor queso nacho', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Peso: 150g', 'values' => ['Peso' => '150g'], 'sku' => 'DOR-NAC-150', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Doritos Spicy Sweet Chili', 'descripcion' => 'Tortilla chips sabor chile dulce picante', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Peso: 150g', 'values' => ['Peso' => '150g'], 'sku' => 'DOR-CHI-150', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Doritos Mega Queso', 'descripcion' => 'Tortilla chips mega queso', 'precio_base' => 2.45, 'variantes' => [['nombre' => 'Peso: 160g', 'values' => ['Peso' => '160g'], 'sku' => 'DOR-MEG-160', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
                'Cheese Tris' => [
                    ['nombre' => 'Cheese Tris 50g', 'descripcion' => 'Snack de queso triangular', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 50g', 'values' => ['Peso' => '50g'], 'sku' => 'CHE-TRI-50', 'precio_ajuste' => 0, 'cantidad' => 100]]],
                ],
                'Puffy' => [
                    ['nombre' => 'Puffy Sabor a Pimienta con Limón', 'descripcion' => 'Snack inflado sabor pimienta limón', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Peso: 80g', 'values' => ['Peso' => '80g'], 'sku' => 'PUF-PIM-80', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Puffy Sabor a Queso', 'descripcion' => 'Snack inflado sabor queso', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Peso: 80g', 'values' => ['Peso' => '80g'], 'sku' => 'PUF-QUE-80', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Puffy Sabor a Tomate con Albahaca', 'descripcion' => 'Snack inflado sabor tomate albahaca', 'precio_base' => 1.15, 'variantes' => [['nombre' => 'Peso: 80g', 'values' => ['Peso' => '80g'], 'sku' => 'PUF-TOM-80', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                ],
                'Saltitacos' => [
                    ['nombre' => 'Saltitacos', 'descripcion' => 'Snack salado tipo taco', 'precio_base' => 1.35, 'variantes' => [['nombre' => 'Peso: 100g', 'values' => ['Peso' => '100g'], 'sku' => 'SAL-TAC-100', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Lays' => [
                    ['nombre' => 'Lays Clásicas', 'descripcion' => 'Papas fritas clásicas con sal', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Peso: 140g', 'values' => ['Peso' => '140g'], 'sku' => 'LAY-CLA-140', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Lays Limón', 'descripcion' => 'Papas fritas sabor limón', 'precio_base' => 1.85, 'variantes' => [['nombre' => 'Peso: 140g', 'values' => ['Peso' => '140g'], 'sku' => 'LAY-LIM-140', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                ],
                'Ruffles' => [
                    ['nombre' => 'Ruffles Original', 'descripcion' => 'Papas fritas onduladas originales', 'precio_base' => 2.10, 'variantes' => [['nombre' => 'Peso: 150g', 'values' => ['Peso' => '150g'], 'sku' => 'RUF-ORI-150', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Ruffles Queso y Cebolla', 'descripcion' => 'Papas fritas onduladas sabor queso y cebolla', 'precio_base' => 2.10, 'variantes' => [['nombre' => 'Peso: 150g', 'values' => ['Peso' => '150g'], 'sku' => 'RUF-QUE-150', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                ],
                'Cheetos' => [
                    ['nombre' => 'Cheetos Crunchy', 'descripcion' => 'Snacks de maíz sabor queso cheddar', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Peso: 130g', 'values' => ['Peso' => '130g'], 'sku' => 'CHE-CRU-130', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Cheetos Flamin Hot', 'descripcion' => 'Snacks de maíz sabor picante', 'precio_base' => 1.95, 'variantes' => [['nombre' => 'Peso: 130g', 'values' => ['Peso' => '130g'], 'sku' => 'CHE-FLA-130', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                ],
                'Oreo' => [
                    ['nombre' => 'Oreo Original', 'descripcion' => 'Galletas de chocolate con crema', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 154g', 'values' => ['Peso' => '154g'], 'sku' => 'ORE-ORI-154', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                    ['nombre' => 'Oreo Golden', 'descripcion' => 'Galletas doradas con crema de vainilla', 'precio_base' => 2.50, 'variantes' => [['nombre' => 'Peso: 154g', 'values' => ['Peso' => '154g'], 'sku' => 'ORE-GOL-154', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                ],
                'Ritz' => [
                    ['nombre' => 'Ritz Crackers Original', 'descripcion' => 'Galletas saladas redondas', 'precio_base' => 2.25, 'variantes' => [['nombre' => 'Peso: 200g', 'values' => ['Peso' => '200g'], 'sku' => 'RIT-ORI-200', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                    ['nombre' => 'Ritz Crackers Queso', 'descripcion' => 'Galletas saladas con sabor a queso', 'precio_base' => 2.35, 'variantes' => [['nombre' => 'Peso: 200g', 'values' => ['Peso' => '200g'], 'sku' => 'RIT-QUE-200', 'precio_ajuste' => 0, 'cantidad' => 70]]],
                ],
            ],
            'Desayuno' => [
                'Kelloggs' => [
                    ['nombre' => 'Corn Flakes Kelloggs', 'descripcion' => 'Cereal de maíz tostado', 'precio_base' => 4.50, 'variantes' => [['nombre' => 'Peso: 350g', 'values' => ['Peso' => '350g'], 'sku' => 'KEL-COR-350', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                    ['nombre' => 'Froot Loops Kelloggs', 'descripcion' => 'Cereal de colores con sabor a frutas', 'precio_base' => 5.25, 'variantes' => [['nombre' => 'Peso: 345g', 'values' => ['Peso' => '345g'], 'sku' => 'KEL-FRO-345', 'precio_ajuste' => 0, 'cantidad' => 55]]],
                    ['nombre' => 'Special K Kelloggs', 'descripcion' => 'Cereal de arroz y trigo', 'precio_base' => 5.75, 'variantes' => [['nombre' => 'Peso: 380g', 'values' => ['Peso' => '380g'], 'sku' => 'KEL-SPE-380', 'precio_ajuste' => 0, 'cantidad' => 50]]],
                ],
                'Quaker' => [
                    ['nombre' => 'Avena Quaker Original', 'descripcion' => 'Avena instantánea natural', 'precio_base' => 3.25, 'variantes' => [['nombre' => 'Peso: 400g', 'values' => ['Peso' => '400g'], 'sku' => 'QUA-AVE-400', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Avena Quaker Canela', 'descripcion' => 'Avena instantánea sabor canela', 'precio_base' => 3.50, 'variantes' => [['nombre' => 'Peso: 400g', 'values' => ['Peso' => '400g'], 'sku' => 'QUA-CAN-400', 'precio_ajuste' => 0, 'cantidad' => 75]]],
                ],
                'Nestlé' => [
                    ['nombre' => 'Nesquik Chocolate', 'descripcion' => 'Polvo para bebida sabor chocolate', 'precio_base' => 4.75, 'variantes' => [['nombre' => 'Peso: 400g', 'values' => ['Peso' => '400g'], 'sku' => 'NES-CHO-400', 'precio_ajuste' => 0, 'cantidad' => 65]]],
                    ['nombre' => 'Nesquik Fresa', 'descripcion' => 'Polvo para bebida sabor fresa', 'precio_base' => 4.75, 'variantes' => [['nombre' => 'Peso: 400g', 'values' => ['Peso' => '400g'], 'sku' => 'NES-FRE-400', 'precio_ajuste' => 0, 'cantidad' => 60]]],
                ],
                'Rexona' => [
                    ['nombre' => 'Jabón de Baño Rexona Antibacterial', 'descripcion' => 'Jabón antibacterial protección 12h', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'JAB-REX-ANT-110', 'precio_ajuste' => 0, 'cantidad' => 90]]],
                    ['nombre' => 'Jabón de Baño Rexona Aloe Vera', 'descripcion' => 'Jabón con extracto de aloe vera', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'JAB-REX-ALO-110', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Jabón de Baño Rexona Glicerina', 'descripcion' => 'Jabón con glicerina hidratante', 'precio_base' => 1.25, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'JAB-REX-GLI-110', 'precio_ajuste' => 0, 'cantidad' => 85]]],
                    ['nombre' => 'Jabón de Baño Rexona Avena', 'descripcion' => 'Jabón exfoliante con avena', 'precio_base' => 1.35, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'JAB-REX-AVE-110', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Jabón de Baño Rexona Coco', 'descripcion' => 'Jabón hidratante con aceite de coco', 'precio_base' => 1.35, 'variantes' => [['nombre' => 'Peso: 110g', 'values' => ['Peso' => '110g'], 'sku' => 'JAB-REX-COC-110', 'precio_ajuste' => 0, 'cantidad' => 80]]],
                    ['nombre' => 'Jabón de Baño Rexona Surtidos 8x110g', 'descripcion' => 'Pack de 8 jabones surtidos de 110g cada uno', 'precio_base' => 8.95, 'variantes' => [['nombre' => 'Pack: 8 unidades x 110g', 'values' => ['Pack' => '8 unidades x 110g'], 'sku' => 'JAB-REX-SUR-8X110', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                ],
            ],
            'Cuidado del Bebé' => [
                'Baby Finger' => [
                    ['nombre' => 'Pañales Baby Finger P', 'descripcion' => 'Pañales talla P', 'precio_base' => 8.50, 'variantes' => [['nombre' => 'Talla: P', 'values' => ['Talla' => 'P'], 'sku' => 'PAN-BF-P', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                    ['nombre' => 'Pañales Baby Finger M', 'descripcion' => 'Pañales talla M', 'precio_base' => 8.50, 'variantes' => [['nombre' => 'Talla: M', 'values' => ['Talla' => 'M'], 'sku' => 'PAN-BF-M', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                    ['nombre' => 'Pañales Baby Finger G', 'descripcion' => 'Pañales talla G', 'precio_base' => 8.50, 'variantes' => [['nombre' => 'Talla: G', 'values' => ['Talla' => 'G'], 'sku' => 'PAN-BF-G', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                    ['nombre' => 'Pañales Baby Finger RN', 'descripcion' => 'Pañales talla recién nacido', 'precio_base' => 8.50, 'variantes' => [['nombre' => 'Talla: RN', 'values' => ['Talla' => 'RN'], 'sku' => 'PAN-BF-RN', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                    ['nombre' => 'Pañales Baby Finger XXG', 'descripcion' => 'Pañales talla XXG', 'precio_base' => 8.50, 'variantes' => [['nombre' => 'Talla: XXG', 'values' => ['Talla' => 'XXG'], 'sku' => 'PAN-BF-XXG', 'precio_ajuste' => 0, 'cantidad' => 40]]],
                ],
                'Hugme' => [
                    ['nombre' => 'Pañales Hugme P', 'descripcion' => 'Pañales talla P', 'precio_base' => 9.50, 'variantes' => [['nombre' => 'Pack: 10 x 20', 'values' => ['Pack' => '10 x 20'], 'sku' => 'PAN-HUG-P-10X20', 'precio_ajuste' => 0, 'cantidad' => 30], ['nombre' => 'Pack: 5 x 20', 'values' => ['Pack' => '5 x 20'], 'sku' => 'PAN-HUG-P-5X20', 'precio_ajuste' => -4.50, 'cantidad' => 50]]],
                    ['nombre' => 'Pañales Hugme M', 'descripcion' => 'Pañales talla M', 'precio_base' => 9.50, 'variantes' => [['nombre' => 'Pack: 10 x 20', 'values' => ['Pack' => '10 x 20'], 'sku' => 'PAN-HUG-M-10X20', 'precio_ajuste' => 0, 'cantidad' => 30], ['nombre' => 'Pack: 5 x 20', 'values' => ['Pack' => '5 x 20'], 'sku' => 'PAN-HUG-M-5X20', 'precio_ajuste' => -4.50, 'cantidad' => 50]]],
                    ['nombre' => 'Pañales Hugme G', 'descripcion' => 'Pañales talla G', 'precio_base' => 9.50, 'variantes' => [['nombre' => 'Pack: 10 x 20', 'values' => ['Pack' => '10 x 20'], 'sku' => 'PAN-HUG-G-10X20', 'precio_ajuste' => 0, 'cantidad' => 30], ['nombre' => 'Pack: 5 x 20', 'values' => ['Pack' => '5 x 20'], 'sku' => 'PAN-HUG-G-5X20', 'precio_ajuste' => -4.50, 'cantidad' => 50]]],
                    ['nombre' => 'Pañales Hugme XG', 'descripcion' => 'Pañales talla XG', 'precio_base' => 9.50, 'variantes' => [['nombre' => 'Pack: 10 x 20', 'values' => ['Pack' => '10 x 20'], 'sku' => 'PAN-HUG-XG-10X20', 'precio_ajuste' => 0, 'cantidad' => 30], ['nombre' => 'Pack: 5 x 20', 'values' => ['Pack' => '5 x 20'], 'sku' => 'PAN-HUG-XG-5X20', 'precio_ajuste' => -4.50, 'cantidad' => 50]]],
                    ['nombre' => 'Pañales Hugme RN', 'descripcion' => 'Pañales recién nacido', 'precio_base' => 8.75, 'variantes' => [['nombre' => 'Pack: 10 x 24', 'values' => ['Pack' => '10 x 24'], 'sku' => 'PAN-HUG-RN-10X24', 'precio_ajuste' => 0, 'cantidad' => 25], ['nombre' => 'Pack: 5 x 24', 'values' => ['Pack' => '5 x 24'], 'sku' => 'PAN-HUG-RN-5X24', 'precio_ajuste' => -4.00, 'cantidad' => 40]]],
                    ['nombre' => 'Toallitas Húmedas Hugme', 'descripcion' => 'Toallitas húmedas para bebé', 'precio_base' => 3.25, 'variantes' => [['nombre' => 'Pack: 80 und', 'values' => ['Pack' => '80 und'], 'sku' => 'TOA-HUG-80', 'precio_ajuste' => 0, 'cantidad' => 60], ['nombre' => 'Pack: 160 und', 'values' => ['Pack' => '160 und'], 'sku' => 'TOA-HUG-160', 'precio_ajuste' => 2.75, 'cantidad' => 40]]],
                    ['nombre' => 'Shampoo Hugme Bebé', 'descripcion' => 'Shampoo suave para bebé', 'precio_base' => 4.50, 'variantes' => [['nombre' => 'Volumen: 200ml', 'values' => ['Volumen' => '200ml'], 'sku' => 'SHA-HUG-200', 'precio_ajuste' => 0, 'cantidad' => 45], ['nombre' => 'Volumen: 400ml', 'values' => ['Volumen' => '400ml'], 'sku' => 'SHA-HUG-400', 'precio_ajuste' => 3.50, 'cantidad' => 30]]],
                    ['nombre' => 'Jabón Líquido Hugme', 'descripcion' => 'Jabón líquido para bebé', 'precio_base' => 4.25, 'variantes' => [['nombre' => 'Volumen: 250ml', 'values' => ['Volumen' => '250ml'], 'sku' => 'JAB-HUG-250', 'precio_ajuste' => 0, 'cantidad' => 50], ['nombre' => 'Volumen: 500ml', 'values' => ['Volumen' => '500ml'], 'sku' => 'JAB-HUG-500', 'precio_ajuste' => 3.75, 'cantidad' => 35]]],
                    ['nombre' => 'Crema Hugme Bebé', 'descripcion' => 'Crema hidratante para bebé', 'precio_base' => 5.75, 'variantes' => [['nombre' => 'Peso: 100g', 'values' => ['Peso' => '100g'], 'sku' => 'CRE-HUG-100', 'precio_ajuste' => 0, 'cantidad' => 40], ['nombre' => 'Peso: 200g', 'values' => ['Peso' => '200g'], 'sku' => 'CRE-HUG-200', 'precio_ajuste' => 4.25, 'cantidad' => 25]]],
                    ['nombre' => 'Aceite Hugme Bebé', 'descripcion' => 'Aceite corporal para bebé', 'precio_base' => 3.95, 'variantes' => [['nombre' => 'Volumen: 100ml', 'values' => ['Volumen' => '100ml'], 'sku' => 'ACE-HUG-100', 'precio_ajuste' => 0, 'cantidad' => 55], ['nombre' => 'Volumen: 200ml', 'values' => ['Volumen' => '200ml'], 'sku' => 'ACE-HUG-200', 'precio_ajuste' => 2.50, 'cantidad' => 35]]],
                ],
            ],
        ];
    }

    /**
     * Obtener categoría por nombre
     */
    private function obtenerCategoria(string $nombre, ...$categorias)
    {
        foreach ($categorias as $categoria) {
            if ($categoria->nombre === $nombre) {
                return $categoria;
            }
        }
        return $categorias[0]; // Por defecto Víveres
    }

    /**
     * Obtener marca por nombre
     */
    private function obtenerMarca(string $nombre)
    {
        return Marca::where('nombre', $nombre)->first();
    }

    /**
     * Crear productos y sus variantes.
     */
    private function crearProductos(array $productos, Categoria $categoria, Marca $marca): void
    {
        foreach ($productos as $productoData) {
            $producto = Producto::create([
                'code' => strtoupper(substr(Str::slug($productoData['nombre']), 0, 3)) . '-' . rand(10000, 99999),
                'name' => $productoData['nombre'],
                'price' => $productoData['precio_base'],
                'precio_bs' => $productoData['precio_base'] * 35,
                'description' => $productoData['descripcion'],
                'quantity' => array_sum(array_column($productoData['variantes'], 'cantidad')),
                'quantity_alert' => 10,
                'status' => true,
                'empresa_id' => $categoria->empresa_id,
                'categoria_id' => $categoria->id,
                'marca_id' => $marca->id,
            ]);

            foreach ($productoData['variantes'] as $varianteData) {
                ProductoVariant::create([
                    'producto_id' => $producto->id,
                    'name' => $varianteData['nombre'],
                    'values' => $varianteData['values'],
                    'sku' => $varianteData['sku'],
                    'price_adjustment' => $varianteData['precio_ajuste'],
                    'quantity' => $varianteData['cantidad'],
                    'image_path' => null,
                    'image_thumbnail_path' => null,
                    'alt_text' => $varianteData['nombre'] . ' - ' . $productoData['nombre'],
                    'empresa_id' => $categoria->empresa_id,
                ]);
            }
        }
    }

    /**
     * Obtiene la marca correspondiente según el nombre del producto
     *
     * @param string $nombreProducto
     * @return Marca
     */
    private function getMarcaPorNombre($nombreProducto)
    {
        // Verificar marcas de cigarros
        if (str_contains($nombreProducto, 'Lucky Strike')) {
            return $this->marcaLuckyStrike;
        } elseif (str_contains($nombreProducto, 'Belmont')) {
            return $this->marcaBelmont;
        } elseif (str_contains($nombreProducto, 'Chimo')) {
            return $this->marcaChimo;
        } elseif (str_contains($nombreProducto, 'Pall Mall')) {
            return $this->marcaPallMall;
        } elseif (str_contains($nombreProducto, 'Viceroy')) {
            return $this->marcaViceroy;
        } elseif (str_contains($nombreProducto, 'Nova')) {
            return $this->marcaNova;
        } elseif (str_contains($nombreProducto, 'Eclipse')) {
            return $this->marcaEclipse;
        } elseif (str_contains($nombreProducto, 'Universal')) {
            return $this->marcaUniversal;
        } elseif (str_contains($nombreProducto, 'Alive')) {
            return $this->marcaAlive;
        } elseif (str_contains($nombreProducto, 'Oso Blanco')) {
            return $this->marcaOsoBlanco;
        }

        // Por defecto, devolver una marca genérica
        return Marca::where('nombre', 'Polar')->first();
    }

    /**
     * Mostrar mensajes de éxito
     */
    private function mostrarMensajesExito(): void
    {
        $mensaje = 'Productos creados exitosamente!';
        $totalProductos = Producto::count();
        $totalVariantes = ProductoVariant::count();

        if ($this->command) {
            $this->command->info($mensaje);
            $this->command->info('Total de productos: ' . $totalProductos);
            $this->command->info('Total de variantes: ' . $totalVariantes);
        } else {
            \Log::info($mensaje);
            \Log::info('Total de productos: ' . $totalProductos);
            \Log::info('Total de variantes: ' . $totalVariantes);
        }
    }


}