<?php

require_once 'vendor/autoload.php';

$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "🔍 AUDITORÍA CRÍTICA DE LA API WHATSAPP\n";
echo "=======================================\n\n";

$issues = [];
$improvements = [];

// 1. Seguridad
echo "🔒 ANÁLISIS DE SEGURIDAD:\n";
echo "-------------------------\n";

// Verificar si las contraseñas se envían en texto plano
$jobFile = file_get_contents('app/Jobs/SendWelcomeWhatsAppMessage.php');
if (strpos($jobFile, '$this->originalPassword') !== false) {
    $issues[] = "🚨 CRÍTICO: Contraseñas en texto plano en Jobs (riesgo de logs/dumps)";
    $improvements[] = "Encriptar contraseñas antes de enviar o usar tokens temporales";
}

// Verificar rate limiting
$whatsappService = file_get_contents('app/Services/WhatsAppService.php');
if (strpos($whatsappService, 'rate') === false && strpos($whatsappService, 'throttle') === false) {
    $issues[] = "⚠️  MEDIO: Sin rate limiting explícito en WhatsApp Service";
    $improvements[] = "Implementar rate limiting para prevenir spam";
}

// 2. Validación de datos
echo "📝 ANÁLISIS DE VALIDACIÓN:\n";
echo "--------------------------\n";

// Verificar validación de números de teléfono
$indexFile = file_get_contents('app/Livewire/Front/Index.php');
if (strpos($indexFile, 'regex') === false || strpos($indexFile, 'phone') === false) {
    $issues[] = "⚠️  MEDIO: Validación básica de números telefónicos";
    $improvements[] = "Añadir validación regex específica por país";
}

// 3. Manejo de errores
echo "🛡️  ANÁLISIS DE ERRORES:\n";
echo "------------------------\n";

// Verificar si hay información sensible en logs
if (strpos($whatsappService, 'Log::info') !== false && strpos($whatsappService, 'password') !== false) {
    $issues[] = "🚨 CRÍTICO: Posible logging de información sensible";
    $improvements[] = "Sanitizar logs para evitar exposición de datos sensibles";
}

// 4. Rendimiento
echo "⚡ ANÁLISIS DE RENDIMIENTO:\n";
echo "---------------------------\n";

// Verificar conexiones de DB en loops
if (strpos($indexFile, 'where(') !== false && strpos($indexFile, 'foreach') !== false) {
    $issues[] = "⚠️  MEDIO: Posibles consultas N+1 en componentes Livewire";
    $improvements[] = "Optimizar consultas con eager loading";
}

// 5. Escalabilidad
echo "📈 ANÁLISIS DE ESCALABILIDAD:\n";
echo "-----------------------------\n";

// Verificar límites de procesamiento
if (strpos($whatsappService, 'batch') === false && strpos($whatsappService, 'chunk') === false) {
    $issues[] = "⚠️  MEDIO: Sin procesamiento por lotes para alto volumen";
    $improvements[] = "Implementar procesamiento por lotes para mensajes masivos";
}

// 6. Configuración
echo "⚙️  ANÁLISIS DE CONFIGURACIÓN:\n";
echo "------------------------------\n";

$configFile = 'config/whatsapp.php';
if (!file_exists($configFile)) {
    $issues[] = "⚠️  MEDIO: Configuración hardcodeada en lugar de archivo config";
    $improvements[] = "Centralizar configuración en archivo config/whatsapp.php";
}

// 7. Testing
echo "🧪 ANÁLISIS DE TESTING:\n";
echo "-----------------------\n";

$testFiles = glob('tests/**/*WhatsApp*Test.php');
if (empty($testFiles)) {
    $issues[] = "🚨 CRÍTICO: Sin tests unitarios para WhatsApp Service";
    $improvements[] = "Crear suite completa de tests unitarios e integración";
}

// 8. Monitoreo
echo "📊 ANÁLISIS DE MONITOREO:\n";
echo "-------------------------\n";

if (!class_exists('App\Services\WhatsAppHealthService')) {
    $issues[] = "⚠️  MEDIO: Sin métricas de negocio (conversión, engagement)";
    $improvements[] = "Añadir métricas de negocio y analytics";
}

// Mostrar resultados
echo "\n🚨 PROBLEMAS IDENTIFICADOS:\n";
echo "===========================\n";
foreach ($issues as $i => $issue) {
    echo ($i + 1) . ". {$issue}\n";
}

echo "\n💡 MEJORAS SUGERIDAS:\n";
echo "=====================\n";
foreach ($improvements as $i => $improvement) {
    echo ($i + 1) . ". {$improvement}\n";
}

// Calcular puntuación de seguridad
$criticalIssues = count(array_filter($issues, fn($issue) => strpos($issue, 'CRÍTICO') !== false));
$mediumIssues = count(array_filter($issues, fn($issue) => strpos($issue, 'MEDIO') !== false));

$securityScore = max(0, 100 - ($criticalIssues * 30) - ($mediumIssues * 10));

echo "\n📊 PUNTUACIÓN DE SEGURIDAD: {$securityScore}%\n";

if ($securityScore >= 90) {
    echo "🟢 Estado: EXCELENTE\n";
} elseif ($securityScore >= 70) {
    echo "🟡 Estado: BUENO - Mejoras recomendadas\n";
} elseif ($securityScore >= 50) {
    echo "🟠 Estado: REGULAR - Mejoras necesarias\n";
} else {
    echo "🔴 Estado: CRÍTICO - Acción inmediata requerida\n";
}

echo "\n=======================================\n";