<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    private $baseUrl;
    private $apiKey;
    private $timeout;

    public function __construct()
    {
        $this->baseUrl = config('whatsapp.api_url', 'http://localhost:3001');
        $this->apiKey = config('whatsapp.api_key', 'test-api-key-vargas-centro');
        $this->timeout = config('whatsapp.timeout', 30);
    }

     public function testConnection(): array
    {
         $key = $apiKey ?? $this->apiKey;

        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(['X-API-Key' => $key])
                ->get("{$this->baseUrl}/api/whatsapp/status");

            return $response->successful() ? $response->json() : null;
        } catch (\Exception $e) {
            Log::error('WhatsApp Status Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Obtener el estado de la conexión WhatsApp
     */
    public function getStatus($apiKey = null)
    {
        $key = $apiKey ?? $this->apiKey;

        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(['X-API-Key' => $key])
                ->get("{$this->baseUrl}/api/whatsapp/status");

            return $response->successful() ? $response->json() : null;
        } catch (\Exception $e) {
            Log::error('WhatsApp Status Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Crear empresa en el servicio de WhatsApp
     */
    public function createCompany($companyId, $companyName, $apiKey, $webhookUrl = null)
    {
        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(['X-API-Key' => $apiKey])  // Usar el API key de la empresa, no el del servicio
                ->post("{$this->baseUrl}/api/companies", [
                    'id' => $companyId,
                    'name' => $companyName,
                    'apiKey' => $apiKey,
                    'webhookUrl' => $webhookUrl,
                    'rateLimitPerMinute' => 60,
                    'isActive' => true
                ]);

            if ($response->successful()) {
                Log::info('Empresa creada en WhatsApp service', [
                    'company_id' => $companyId,
                    'company_name' => $companyName,
                    'api_key_used' => $apiKey
                ]);
                return true;
            } else {
                Log::error('Error al crear empresa en WhatsApp service', [
                    'company_id' => $companyId,
                    'status' => $response->status(),
                    'response' => $response->body(),
                    'api_key_used' => $apiKey
                ]);
                return false;
            }
        } catch (\Exception $e) {
            Log::error('Excepción al crear empresa en WhatsApp service', [
                'company_id' => $companyId,
                'error' => $e->getMessage(),
                'api_key_used' => $apiKey
            ]);
            return false;
        }
    }

    /**
     * Obtener código QR para conectar WhatsApp
     */
    public function getQRCode()
    {
        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(['X-API-Key' => $this->apiKey])
                ->get("{$this->baseUrl}/api/whatsapp/qr");

            return $response->successful() ? $response->json() : null;
        } catch (\Exception $e) {
            Log::error('WhatsApp QR Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Enviar mensaje de texto
     */
    public function sendMessage(string $to, string $message)
    {
        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(['X-API-Key' => $this->apiKey])
                ->post("{$this->baseUrl}/api/whatsapp/send", [
                    'to' => $to,
                    'message' => $message,
                    'type' => 'text'
                ]);

            return $response->successful() ? $response->json() : null;
        } catch (\Exception $e) {
            Log::error('WhatsApp Send Message Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Obtener historial de mensajes
     */
    public function getMessages(array $filters = [])
    {
        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(['X-API-Key' => $this->apiKey])
                ->get("{$this->baseUrl}/api/whatsapp/messages", $filters);

            return $response->successful() ? $response->json() : null;
        } catch (\Exception $e) {
            Log::error('WhatsApp Get Messages Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Conectar WhatsApp
     */
    public function connect()
    {
        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(['X-API-Key' => $this->apiKey])
                ->post("{$this->baseUrl}/api/whatsapp/connect");

            return $response->successful() ? $response->json() : null;
        } catch (\Exception $e) {
            Log::error('WhatsApp Connect Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Desconectar WhatsApp
     */
    public function disconnect()
    {
        try {
            $response = Http::timeout($this->timeout)
                ->withHeaders(['X-API-Key' => $this->apiKey])
                ->delete("{$this->baseUrl}/api/whatsapp/disconnect");

            return $response->successful() ? $response->json() : null;
        } catch (\Exception $e) {
            Log::error('WhatsApp Disconnect Error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Limpiar y formatear número de teléfono
     */
    public function cleanPhoneNumber(string $phoneNumber): string
    {
        // Remover espacios, guiones, paréntesis y símbolos
        $cleanNumber = preg_replace('/[\s\-\(\)\+]/', '', $phoneNumber);

        // Si el número empieza con 0, removerlo (formato local venezolano)
        if (substr($cleanNumber, 0, 1) === '0') {
            $cleanNumber = substr($cleanNumber, 1);
        }

        // Si no tiene código de país y parece ser venezolano (10 dígitos que empiezan con 2 o 4), agregar 58
        if (strlen($cleanNumber) === 10 && in_array(substr($cleanNumber, 0, 1), ['2', '4'])) {
            $cleanNumber = '58' . $cleanNumber;
        }

        // Si ya tiene código de país (58) pero no empieza con 58, asegurar que tenga 58
        if (strlen($cleanNumber) === 11 && substr($cleanNumber, 0, 2) !== '58' && in_array(substr($cleanNumber, 0, 1), ['2', '4'])) {
            $cleanNumber = '58' . substr($cleanNumber, 1);
        }

        // Si el número tiene 9 dígitos después de quitar el 0 y empieza con 2 o 4, agregar 58
        if (strlen($cleanNumber) === 9 && in_array(substr($cleanNumber, 0, 1), ['2', '4'])) {
            $cleanNumber = '58' . $cleanNumber;
        }

        return $cleanNumber;
    }

    /**
     * Enviar notificación de pedido al cliente
     *
     * @param \App\Models\Pedido $pedido
     * @return array
     */
    public function sendOrderNotification($pedido): array
    {
        try {
            $user = $pedido->user;

            if (!$user || !$user->telefono) {
                return [
                    'success' => false,
                    'error' => 'Usuario no tiene número de teléfono registrado'
                ];
            }

            $message = $this->formatOrderMessage($pedido);
            $cleanPhone = $this->cleanPhoneNumber($user->telefono);

            return $this->sendMessage($cleanPhone, $message);
        } catch (\Exception $e) {
            Log::error('Error en sendOrderNotification: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => 'Error al procesar notificación: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Formatear mensaje de notificación de pedido
     *
     * @param \App\Models\Pedido $pedido
     * @return string
     */
    private function formatOrderMessage($pedido): string
    {
        $user = $pedido->user;
        $empresa = $user->empresa ?? null;

        $message = "📦 *¡Pedido Realizado!* 📦\n\n";
        $message .= "*Hola {$user->name}!*\n\n";
        $message .= "✅ Tu pedido *#{$pedido->codigo}* ha sido recibido y está siendo procesado.\n\n";

        $currency = $pedido->moneda;

        // Detalles del pedido
        $message .= "📋 *Detalles del Pedido:*\n";
        $message .= "• Código: #{$pedido->codigo}\n";
        $message .= "• Fecha: " . $pedido->created_at->format('d/m/Y H:i') . "\n";
        if ($currency == 'usd') {
            $message .= "• Total: " .  format_money($pedido->total_usd, 2) . "\n";
        } else {
            $message .= "• Total: " .  format_money($pedido->total_bs, 2) . "\n";
        }
        $message .= "• Estado: {$pedido->estado}\n";
        $message .= "• Método de Pago: {$pedido->metodo_pago}\n\n";

        // Productos
        if ($pedido->detalles && $pedido->detalles->count() > 0 &&   $currency == 'usd') {
            $message .= "🛍️ *Productos:*\n";

            foreach ($pedido->detalles as $detalle) {
                $productoNombre = $detalle->product_name ?? $detalle->producto->name ?? 'Producto';
                $message .= "• {$productoNombre} x{$detalle->quantity} = " . format_money($detalle->total_usd, 2) . "\n";
            }
            $message .= "\n";
        }
        else
            {
                 $message .= "🛍️ *Productos:*\n";

                foreach ($pedido->detalles as $detalle) {
                    $productoNombre = $detalle->product_name ?? $detalle->producto->name ?? 'Producto';
                    $message .= "• {$productoNombre} x{$detalle->quantity} = " . format_money($detalle->total_bs, 2) . "\n";
                }
                $message .= "\n";
            }

        // Información de entrega
        if ($pedido->ubicacion) {
            $message .= "📍 *Dirección de Entrega:*\n{$pedido->ubicacion}\n\n";
        }

        // Próximos pasos
        $message .= "⏳ *Próximos Pasos:*\n";
        $message .= "• Revisaremos tu pedido\n";
        $message .= "• Confirmaremos la disponibilidad de los productos\n";
        $message .= "• Te notificaremos cuando esté listo para entrega\n\n";

        $message .= "¡Gracias por tu compra! 😊\n\n";

        if ($empresa) {
            $message .= "Atentamente,\n{$empresa->razon_social}\n";
            if ($empresa->telefono) {
                $message .= "📞 {$empresa->telefono}\n";
            }
        }

        return $message;
    }
}
