<?php

namespace App\Services;

use App\Models\Pedido;
use App\Models\Empleado;
use App\Models\User;
use App\Models\HistoricoPedido;
use App\Models\PedidoPago;
use App\Services\WhatsAppService;
use App\Events\PedidoActualizado;
use Illuminate\Support\Facades\DB;

class PedidoService
{
    protected $whatsappService;

    public function __construct(WhatsAppService $whatsappService)
    {
        $this->whatsappService = $whatsappService;
    }

    public function confirmarPago(Pedido $pedido)
    {
        DB::transaction(function () use ($pedido) {
            $pedido->update(['estado' => 'pagado']);

            // Crear registro de pago en caja
            $this->crearPagoDesdePedido($pedido);

            // Registrar en histórico
            $this->registrarHistorico($pedido, "Pago confirmado - {$this->obtenerDetalleMetodoPagoCliente($pedido)}");

            // Enviar WhatsApp al cliente
            $this->enviarConfirmacionPago($pedido);

            // Broadcast evento
            broadcast(new PedidoActualizado($pedido, 'pago_confirmado'));
        });
    }

    public function asignarEmpleado(Pedido $pedido, Empleado $empleado)
    {
        DB::transaction(function () use ($pedido, $empleado) {
            $pedido->empleados()->sync([$empleado->id]);
            $pedido->update(['estado' => 'asignado']);

            // Registrar en histórico
            $this->registrarHistorico($pedido, "Empleado asignado: {$empleado->nombres} {$empleado->apellidos}");

            // Cargar relaciones necesarias para WhatsApp
            $pedido->load(['user.empresa.pais', 'detalles.producto', 'pagos']);
            $empleado->load(['user']);

            // Enviar notificaciones por WhatsApp de forma inmediata (sin cola)
            $notificacionesExitosas = 0;
            $notificacionesFallidas = 0;
            $errores = [];

            try {
                // Notificar al empleado sobre la asignación
                $resultadoEmpleado = $this->enviarNotificacionAsignacionEmpleado($pedido, $empleado);
                if ($resultadoEmpleado['exito']) {
                    $notificacionesExitosas++;
                } else {
                    $notificacionesFallidas++;
                    $errores[] = 'Empleado: ' . $resultadoEmpleado['mensaje'];
                }

                // Notificar al cliente sobre la asignación del empleado
                $resultadoCliente = $this->enviarNotificacionAsignacionCliente($pedido, $empleado);
                if ($resultadoCliente['exito']) {
                    $notificacionesExitosas++;
                } else {
                    $notificacionesFallidas++;
                    $errores[] = 'Cliente: ' . $resultadoCliente['mensaje'];
                }

                \Log::info('✅ Notificaciones de asignación procesadas', [
                    'pedido_id' => $pedido->id,
                    'empleado_id' => $empleado->id,
                    'exitosas' => $notificacionesExitosas,
                    'fallidas' => $notificacionesFallidas,
                    'errores' => $errores
                ]);

            } catch (\Exception $e) {
                dd($e);
                \Log::error('❌ Error al procesar notificaciones de asignación', [
                    'pedido_id' => $pedido->id,
                    'empleado_id' => $empleado->id,
                    'error' => $e->getMessage()
                ]);
                $notificacionesFallidas = 2;
                $errores[] = 'Excepción: ' . $e->getMessage();
            }

            // Broadcast evento
            broadcast(new PedidoActualizado($pedido, 'empleado_asignado'));
        });
    }

    public function cancelarPedido(Pedido $pedido, $motivo = null)
    {
        DB::transaction(function () use ($pedido, $motivo) {
            $pedido->update([
                'estado' => 'cancelado',
                'nota' => $motivo ? "Cancelado: {$motivo}" : 'Pedido cancelado'
            ]);

            // Devolver productos al almacén
            $this->devolverProductosAlmacen($pedido);

            // Registrar en histórico
            $this->registrarHistorico($pedido, "Pedido cancelado: " . ($motivo ?? 'Sin motivo especificado'));

            // Enviar WhatsApp al cliente
            $this->enviarCancelacionPedido($pedido, $motivo);

            // Broadcast evento
            broadcast(new PedidoActualizado($pedido, 'pedido_cancelado'));
        });
    }

    public function finalizarEntrega(Pedido $pedido)
    {
        $pedido->update(['estado' => 'entregado']);

        // Registrar en histórico
        $this->registrarHistorico($pedido, "Entrega finalizada");

        // Enviar confirmación de entrega
        $this->enviarConfirmacionEntrega($pedido);

        // Broadcast evento
        broadcast(new PedidoActualizado($pedido, 'entrega_finalizada'));
    }

    /**
     * Revertir un pedido (incluso si está pagado)
     * Permite al cliente notificar cambios o solicitar reversión
     * Integración completa: también revierte el pago en caja si existe
     */
    public function revertirPedido(Pedido $pedido, $motivo = null, $tipoReversion = 'cambio')
    {
        DB::transaction(function () use ($pedido, $motivo, $tipoReversion) {
            // Determinar el nuevo estado basado en el tipo de reversión
            $nuevoEstado = match($tipoReversion) {
                'cancelacion' => 'cancelado',
                'cambio' => 'en_revision',
                'devolucion' => 'devuelto',
                default => 'en_revision'
            };

            // Actualizar el pedido con el motivo de la reversión
            $notaActualizada = $pedido->nota ? $pedido->nota . "\n" : "";
            $notaActualizada .= now()->format('d/m/Y H:i') . " - ";
            $notaActualizada .= ucfirst($tipoReversion) . ": " . ($motivo ?? 'Sin motivo especificado');

            $pedido->update([
                'estado' => $nuevoEstado,
                'nota' => $notaActualizada
            ]);

            // Devolver productos al almacén (solo si el pedido estaba confirmado o entregado)
            if (in_array($pedido->estado, ['pagado', 'asignado', 'entregado'])) {
                $this->devolverProductosAlmacen($pedido);
            }

            // Revertir el pago en caja si el pedido estaba pagado
            if ($pedido->estado === 'pagado') {
                $this->revertirPagoEnCaja($pedido, $motivo, $tipoReversion);
            }

            // Registrar en histórico
            $this->registrarHistorico($pedido, "Pedido revertido - {$tipoReversion}: " . ($motivo ?? 'Sin motivo'));

            // Enviar notificaciones según el tipo de reversión
            $this->enviarNotificacionReversion($pedido, $motivo, $tipoReversion);

            // Broadcast evento
            $eventoTipo = match($tipoReversion) {
                'cancelacion' => 'pedido_cancelado',
                'cambio' => 'pedido_en_revision',
                'devolucion' => 'pedido_devuelto',
                default => 'pedido_revertido'
            };

            broadcast(new PedidoActualizado($pedido, $eventoTipo));
        });
    }

    /**
     * Crear registro de pago en caja cuando se confirma el pago de un pedido
     * Integración completa: crea tanto PedidoPago como Pago en caja
     */
    private function crearPagoDesdePedido(Pedido $pedido)
    {
        try {
            // Obtener la caja abierta actual
            $cajaAbierta = \App\Models\Caja::obtenerCajaAbierta(
                $pedido->empresaId,
                $pedido->sucursal_id ?? 1,
                now()->toDateString()
            );

            if (!$cajaAbierta) {
                // Si no hay caja abierta, crear una automáticamente
                $cajaAbierta = \App\Models\Caja::crearCajaDiaria(
                    $pedido->empresaId,
                    $pedido->sucursal_id ?? 1,
                    0,
                    'Caja automática para pedido #' . $pedido->id,
                    auth()->id() ?? $pedido->userId
                );
            }

                        // 2. Crear Pago en caja para el sistema contable
            $this->crearPagoEnCaja($pedido);

        } catch (\Exception $e) {
            // Si hay algún error al crear el pago, registrar en el log
            \Log::error('Error al crear pago desde pedido: ' . $e->getMessage(), [
                'pedido_id' => $pedido->id,
                'pedido_codigo' => $pedido->codigo
            ]);
        }
    }

    /**
     * Crear pago en el sistema de cajas (contabilidad)
     */
    private function crearPagoEnCaja(Pedido $pedido)
    {
        // Obtener o crear caja abierta
        $caja = \App\Models\Caja::obtenerCajaAbierta(
            $pedido->empresaId,
            $pedido->sucursal_id ?? 1, // Asumiendo sucursal 1 por defecto
            now()->toDateString()
        );

        if (!$caja) {
            // Si no hay caja abierta, crear una
            $caja = \App\Models\Caja::crearCajaDiaria(
                $pedido->empresaId,
                $pedido->sucursal_id ?? 1,
                0, // monto inicial
                'Caja creada automáticamente para pago de pedido',
                auth()->id() ?? $pedido->userId
            );
        }

        // Obtener concepto de pago para pedidos
        //$conceptoPago = $this->obtenerConceptoPagoPedidos();

        $pago = \App\Models\PedidoPago::where('pedidoId', $pedido->id)->update([
            'caja_id' => $caja->id,
            'status'  => 'Pagado'
        ]);

        // Actualizar totales de la caja
        $caja->calcularTotales();
    }

    /**
     * Obtener o crear concepto de pago para pedidos
     */
    private function obtenerConceptoPagoPedidos()
    {
        $concepto = \App\Models\ConceptoPago::where('nombre', 'Pedidos')
            ->where('empresa_id', auth()->user()->empresa_id ?? 1)
            ->where('sucursal_id', auth()->user()->sucursal_id ?? 1)
            ->first();

        if (!$concepto) {
            $concepto = \App\Models\ConceptoPago::create([
                'nombre' => 'Pedidos',
                'descripcion' => 'Ventas de pedidos de productos',
                'activo' => true,
                'empresa_id' => auth()->user()->empresa_id ?? 1,
                'sucursal_id' => auth()->user()->sucursal_id ?? 1
            ]);
        }

        return $concepto;
    }

    /**
     * Calcular total en bolívares basado en la tasa de cambio actual
     */
    private function calcularTotalBolivares($totalUsd)
    {
        $tasaCambio = $this->obtenerTasaCambioActual();
        return $tasaCambio ? $totalUsd * $tasaCambio : 0;
    }

    /**
     * Revertir el pago en caja cuando se revierte un pedido
     */
    private function revertirPagoEnCaja(Pedido $pedido, $motivo = null, $tipoReversion = 'cambio')
    {
        try {
            // Buscar el PedidoPago asociado al pedido
            $pedidoPago = PedidoPago::where('pedidoId', $pedido->id)->first();

            if ($pedidoPago) {
                // Si hay una caja asociada, recalcular sus totales
                if ($pedidoPago->caja) {
                    $pedidoPago->caja->calcularTotales();
                }

                // TODO: Si PedidoPago tuviera un campo estado, marcarlo como cancelado
                // Por ahora, podríamos agregar una nota o simplemente dejar el registro como está
                // ya que el pedido revertido ya indica que el pago fue revertido
            }

            // Buscar también en el sistema de caja escolar (si existe)
            if (class_exists('App\Models\Pago')) {
                $pago = \App\Models\Pago::where('referencia', $pedido->codigo)
                    ->where('empresa_id', $pedido->empresaId)
                    ->where('estado', \App\Models\Pago::ESTADO_APROBADO)
                    ->first();

                if ($pago) {
                    // Cancelar el pago
                    $pago->update([
                        'estado' => \App\Models\Pago::ESTADO_CANCELADO,
                        'observaciones' => ($pago->observaciones ? $pago->observaciones . "\n" : '') .
                            "Reversión de pedido - {$tipoReversion}: " . ($motivo ?? 'Sin motivo especificado')
                    ]);

                    // Actualizar totales de la caja
                    if ($pago->caja) {
                        $pago->caja->calcularTotales();
                    }
                }
            }

        } catch (\Exception $e) {
            \Log::error('Error al revertir pago en caja: ' . $e->getMessage(), [
                'pedido_id' => $pedido->id,
                'pedido_codigo' => $pedido->codigo
            ]);
        }
    }

    /**
     * Obtener tasa de cambio actual
     */
    private function obtenerTasaCambioActual()
    {
        $tasa = \App\Models\ExchangeRate::where('fecha', now()->toDateString())
            ->orderBy('created_at', 'desc')
            ->first();

        return $tasa ? $tasa->tasa : null;
    }

    /**
     * Obtener el método de pago desde el pedido
     */
    private function obtenerMetodoPagoDesdePedido(Pedido $pedido)
    {
        // Buscar el método de pago en los detalles del pedido o en la información del cliente
        // Por defecto usamos 'efectivo' si no se encuentra información específica

        if ($pedido->metodo_pago) {
            return $pedido->metodo_pago;
        }

        // Si el pedido tiene información de pago en algún campo adicional
        if ($pedido->payment_method) {
            return $pedido->payment_method;
        }

        // Buscar en los detalles si hay información de método de pago
        $detallesPago = $pedido->detalles()->where('tipo', 'pago')->first();
        if ($detallesPago && $detallesPago->metodo_pago) {
            return $detallesPago->metodo_pago;
        }

        // Por defecto
        return 'efectivo';
    }

    /**
     * Devolver productos al almacén cuando se revierte un pedido
     */
    private function devolverProductosAlmacen(Pedido $pedido)
    {
        $detalles = $pedido->detalles()->with('producto')->get();

        foreach ($detalles as $detalle) {
            $producto = $detalle->producto;
            if ($producto) {
                // Validar que la cantidad sea un valor numérico válido
                $cantidad = $detalle->cantidad;

                // Manejar caso especial de cantidad NULL
                if (is_null($cantidad)) {
                    \Log::error("Cantidad NULL encontrada en detalle ID: {$detalle->id}. Este detalle será ignorado para evitar errores.");
                    continue;
                }

                if (!is_numeric($cantidad) || $cantidad <= 0) {
                    \Log::warning("Cantidad inválida para detalle ID: {$detalle->id}, cantidad: " . var_export($cantidad, true));
                    continue;
                }

                // Aumentar el stock del producto
                $producto->increment('quantity', $cantidad);

                // Registrar en histórico el movimiento de inventario
                $this->registrarHistorico($pedido, "Producto devuelto al almacén: {$producto->name} (Cant: {$cantidad})");
            }
        }
    }

    /**
     * Registrar evento en el histórico del pedido
     */
    private function registrarHistorico(Pedido $pedido, string $descripcion)
    {
        // Determinar un título apropiado basado en la descripción
        $titulo = match(true) {
            str_contains($descripcion, 'Pago confirmado') => 'Pago Confirmado',
            str_contains($descripcion, 'Empleado asignado') => 'Empleado Asignado',
            str_contains($descripcion, 'Pedido cancelado') => 'Pedido Cancelado',
            str_contains($descripcion, 'Pedido revertido') => 'Pedido Revertido',
            str_contains($descripcion, 'Entrega finalizada') => 'Entrega Finalizada',
            str_contains($descripcion, 'Producto devuelto') => 'Devolución de Producto',
            default => 'Actualización de Pedido'
        };

        HistoricoPedido::create([
            'fecha' => now()->format('Y-m-d'),
            'titulo' => $titulo,
            'descripcion' => $descripcion,
            'empresaId' => $pedido->empresaId,
            'userId' => auth()->id() ?? $pedido->userId, // Usar usuario autenticado o el del pedido
            'pedidoId' => $pedido->id
        ]);
    }

    private function enviarConfirmacionPago(Pedido $pedido)
    {
        $cliente = $pedido->user;
        if (!$cliente->telefono) return;
        $telefonoLimpio = $this->cleanPhoneNumber($cliente->telefono);


        $message = "¡Hola {$cliente->name}! 🎉\n\n";
        $message .= "Tu pedido ha sido confirmado exitosamente.\n\n";
        $message .= "📋 **Detalles del pedido:**\n";
        $message .= "Código: {$pedido->codigo}\n";
        if ($pedido->moneda == 'usd') {
            $message .= "Total: \${$pedido->total_usd}\n";
        } else {
            $message .= "Total: \${$pedido->total_bs}\n";
        }


        // Obtener detalle mejorado del método de pago para el cliente
        $detallePago = $this->obtenerDetalleMetodoPagoCliente($pedido);
        $message .= $detallePago;

        $message .= "\nEn breves momentos nuestro empleado asignado se contactará contigo para coordinar la entrega.\n\n";
        $message .= "¡Gracias por tu compra! 🚀";

        $this->whatsappService->sendMessage($telefonoLimpio, $message);
    }


    /**
     * Enviar mensaje de WhatsApp con sistema de reintentos
     *
     * @param string $telefono
     * @param string $mensaje
     * @param int $maxIntentos Número máximo de intentos
     * @param int $esperaSegundos Segundos entre intentos
     * @return array Resultado del envío
     */
    public function enviarWhatsAppConReintento(string $telefono, string $mensaje, int $maxIntentos = 3, int $esperaSegundos = 2): array
    {
        $intentos = 0;
        $ultimoError = null;

        while ($intentos < $maxIntentos) {
            try {
                $intentos++;

                // Log del intento
                \Log::info('Intentando enviar mensaje WhatsApp', [
                    'telefono' => $telefono,
                    'intento' => $intentos,
                    'max_intentos' => $maxIntentos
                ]);

                // Intentar enviar el mensaje
                $resultado = $this->whatsappService->sendMessage($telefono, $mensaje);

                // Si llegamos aquí, el envío fue exitoso
                \Log::info('✅ Mensaje WhatsApp enviado exitosamente', [
                    'telefono' => $telefono,
                    'intento' => $intentos,
                    'resultado' => $resultado
                ]);

                return [
                    'exito' => true,
                    'intento' => $intentos,
                    'resultado' => $resultado,
                    'mensaje' => 'Mensaje enviado exitosamente'
                ];

            } catch (\Exception $e) {
                $ultimoError = $e;

                // Log del error
                \Log::error('❌ Error al enviar mensaje WhatsApp', [
                    'telefono' => $telefono,
                    'intento' => $intentos,
                    'error' => $e->getMessage(),
                    'codigo_error' => $e->getCode(),
                    'traza' => $e->getTraceAsString()
                ]);

                // Si no es el último intento, esperar antes de reintentar
                if ($intentos < $maxIntentos) {
                    \Log::info('Esperando antes de reintentar...', [
                        'espera_segundos' => $esperaSegundos,
                        'proximo_intento' => $intentos + 1
                    ]);
                    sleep($esperaSegundos);
                }
            }
        }

        // Si llegamos aquí, todos los intentos fallaron
        \Log::error('❌ Todos los intentos de envío fallaron', [
            'telefono' => $telefono,
            'total_intentos' => $intentos,
            'ultimo_error' => $ultimoError->getMessage()
        ]);

        return [
            'exito' => false,
            'intento' => $intentos,
            'error' => $ultimoError->getMessage(),
            'mensaje' => 'No se pudo enviar el mensaje después de ' . $intentos . ' intentos'
        ];
    }

    /**
     * Guardar mensaje pendiente para reenvío posterior
     *
     * @param int $pedidoId
     * @param string $telefono
     * @param string $mensaje
     * @param string $tipo
     * @param string $error
     */
    private function guardarMensajePendiente(int $pedidoId, string $telefono, string $mensaje, string $tipo, string $error)
    {
        try {
            // Verificar si ya existe un mensaje pendiente similar
            $mensajeExistente = \App\Models\MensajePendiente::where('pedido_id', $pedidoId)
                ->where('telefono', $telefono)
                ->where('tipo', $tipo)
                ->where('estado', 'pendiente')
                ->first();

            if ($mensajeExistente) {
                // Actualizar el mensaje existente
                $mensajeExistente->update([
                    'mensaje' => $mensaje,
                    'error' => $error,
                    'intentos' => $mensajeExistente->intentos + 1,
                    'ultimo_intento' => now()
                ]);

                \Log::info('Mensaje pendiente actualizado', [
                    'id' => $mensajeExistente->id,
                    'pedido_id' => $pedidoId,
                    'tipo' => $tipo,
                    'intentos' => $mensajeExistente->intentos
                ]);
            } else {
                // Crear nuevo mensaje pendiente
                $mensajePendiente = \App\Models\MensajePendiente::create([
                    'pedido_id' => $pedidoId,
                    'telefono' => $telefono,
                    'mensaje' => $mensaje,
                    'tipo' => $tipo,
                    'error' => $error,
                    'estado' => 'pendiente',
                    'intentos' => 1,
                    'ultimo_intento' => now()
                ]);

                \Log::info('Mensaje pendiente creado', [
                    'id' => $mensajePendiente->id,
                    'pedido_id' => $pedidoId,
                    'tipo' => $tipo
                ]);
            }
        } catch (\Exception $e) {
            \Log::error('Error al guardar mensaje pendiente', [
                'pedido_id' => $pedidoId,
                'telefono' => $telefono,
                'tipo' => $tipo,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Enviar notificación al empleado sobre la asignación del pedido
     * Este método envía la notificación de forma inmediata sin usar cola
     */
    private function enviarNotificacionAsignacionEmpleado(Pedido $pedido, Empleado $empleado)
    {
        if (!$empleado->telefono) return;

        // Limpiar y formatear número de teléfono
        $telefonoLimpio = $this->cleanPhoneNumber($empleado->telefono);

        $cliente = $pedido->user;
        // Load variants along with the detalles and productos
        $detalles = \App\Models\PedidoDetalle::with('producto.variants')
        ->where('pedidoId', $pedido->id)
        ->get();
        //dd($detalles);

        // Obtener detalle completo del método de pago
        $metodoPagoDetalle = $this->obtenerDetalleMetodoPago($pedido);

        $message = "¡Hola {$empleado->nombres} {$empleado->apellidos}! 📦\n\n";
        $message .= "Se te ha asignado un nuevo pedido:\n\n";
        $message .= "📋 **Información del pedido:**\n";
        $message .= "Código: {$pedido->codigo}\n";
        $message .= "Cliente: {$cliente->name}\n";
        $message .= "Teléfono: {$cliente->telefono}\n";
        $message .= "Total: \${$pedido->total_usd}\n";
        $message .= "Método de pago: {$metodoPagoDetalle}\n\n";

        $message .= "🛍️ **Productos:**\n";

        foreach ($detalles as $detalle) {
            // 1. Seguridad básica
            if (!$detalle || !isset($detalle->producto) || !$detalle->producto) {
                continue;
            }

            $quantity = $detalle->quantity ?? 0;
            $productName = $detalle->producto->name ?? 'Producto Desconocido';

            if (!empty($detalle->variant_id)) {
                // Asegúrate de usar eager loading: Detalle::with('producto.variants')->get();
                $variant = $detalle->producto->variants->firstWhere('id', $detalle->variant_id);

                if ($variant && isset($variant->alt_text)) {

                    // --- PASO 1: Limpiar el Nombre del Producto ---
                    // Elimina volúmenes al final (ej: "1Lt", "500ml")
                    $patternName = '/\s+(\d+(?:\.\d+)?(?:x\d+)?(?:L|ml|g|kg|oz|unidades)?)$/i';
                    $cleanName = preg_replace($patternName, '', $productName);
                    $cleanName = trim(preg_replace('/\s+/', ' ', $cleanName));

                    // --- PASO 2: Limpiar el Texto de la Variante (CORREGIDO) ---
                    $variantText = $variant->alt_text;

                    // NUEVA REGEX: Busca "CualquierTexto: CualquierValor - " al inicio
                    // [^:]+  -> Todo lo que NO sea dos puntos (el nombre de la propiedad, ej: Volumen)
                    // :      -> Los dos puntos literal
                    // \s*    -> Espacios opcionales
                    // .*?    -> El valor de la propiedad (ej: 1 Lt) de forma no codiciosa
                    // \s*-\s* -> El separador guion con sus espacios
                    $patternVariant = '/^[^:]+:\s*.*?\s*-\s*/';

                    // Aplicamos la limpieza
                    $cleanVariantText = preg_replace($patternVariant, '', $variantText);
                    $cleanVariantText = trim($cleanVariantText);

                    // --- PASO 3: Decidir qué mostrar ---
                    // Verificamos si el nombre limpio está dentro de la variante limpia
                    if (stripos($cleanVariantText, $cleanName) !== false) {
                        $message .= "• {$cleanVariantText} (Cant: {$quantity})\n";
                    } else {
                        $message .= "• {$cleanName} - {$cleanVariantText} (Cant: {$quantity})\n";
                    }

                } else {
                    $message .= "• {$productName} (Cant: {$quantity})\n";
                }
            } else {
                $message .= "• {$productName} (Cant: {$quantity})\n";
            }
        }


        if ($pedido->ubicacion) {
            $message .= "\n📍 **Ubicación:** {$pedido->ubicacion}\n";
        }

        $message .= "\n⚠️ **Recordatorio:** No olvides marcar como entregado una vez completada la entrega.\n\n";
        $message .= "¡Éxito en tu entrega! 🚀";

        // Enviar notificación con sistema de reintentos
        $resultado = $this->enviarWhatsAppConReintento($telefonoLimpio, $message);

        if (!$resultado['exito']) {
            // Si falló después de todos los reintentos, podemos guardar para reenvío posterior
            $this->guardarMensajePendiente($pedido->id, $telefonoLimpio, $message, 'asignacion_empleado', $resultado['error']);
        }

        return $resultado;
    }

    /**
     * Enviar notificación al cliente sobre la asignación del empleado
     * Este método envía la notificación de forma inmediata sin usar cola
     * y con el mismo nivel de detalle que la confirmación de pago
     */
    private function enviarNotificacionAsignacionCliente(Pedido $pedido, Empleado $empleado)
    {
        $cliente = $pedido->user;
        if (!$cliente->telefono) return false;


        $message = "¡Hola {$cliente->name}! 📦\n\n";
        $message .= "Tu pedido ha sido asignado a uno de nuestros empleados para su entrega.\n\n";

        $message .= "📋 **Resumen del pedido:**\n";
        $message .= "Código: {$pedido->codigo}\n";
        $message .= "Total: \${$pedido->total_usd}\n";

        // Obtener detalle del método de pago
        $detallePago = $this->obtenerDetalleMetodoPagoCliente($pedido);
        $message .= $detallePago;

        // Información del empleado asignado
        $message .= "\n👤 **Empleado asignado:**\n";
        $message .= "Nombre: {$empleado->nombres} {$empleado->apellidos}\n";
        if ($empleado->telefono) {
            $message .= "Teléfono: {$empleado->telefono}\n";
        }

        // Dirección de entrega
        if ($pedido->ubicacion) {
            $message .= "\n📍 **Dirección de entrega:**\n";
            $message .= "{$pedido->ubicacion}\n";
        }

        $message .= "\n📞 El empleado se contactará contigo para coordinar la entrega.\n";
        $message .= "¡Gracias por tu preferencia! 🚀";

        $phone = $this->cleanPhoneNumber($cliente->telefono);

        // Enviar con sistema de reintentos
        $resultado = $this->enviarWhatsAppConReintento($phone, $message);

        if (!$resultado['exito']) {
            // Si falló después de todos los reintentos, guardar para reenvío posterior
            $this->guardarMensajePendiente($pedido->id, $cliente->telefono, $message, 'asignacion_cliente', $resultado['error']);
        }

        return $resultado;
    }

    /**
     * Obtener el detalle completo del método de pago
     */
    private function obtenerDetalleMetodoPago($pedido)
    {
        $metodoBase = $pedido->metodo_pago;
        $service = new \App\Services\ExchangeRateService();
        $success = $service->fetchAndStoreRates();
        $todayRate = \App\Models\ExchangeRate::getTodayRate();
        $tasaDelDia = $todayRate ? $todayRate->usd_rate : 1;

        // Si hay pagos asociados (pago mixto)
        if ($pedido->pagos && $pedido->pagos->count() > 0) {
            $detalles = [];
            foreach ($pedido->pagos as $pago) {

                if ($pedido->moneda == 'usd') {
                   $monto = format_money($pago->total_usd);
                } else {
                    $monto = 'Bs. ' . number_format($pago->total_usd * $tasaDelDia, 2, ',', '.');
                }
                $detallePago = "{$pago->metodo_pago}: {$monto}";

                // Agregar referencia si existe
                if ($pago->referencia) {
                    $detallePago .= " (Ref: {$pago->referencia})";
                    }

                    $detalles[] = $detallePago;

            }

            return "(" . implode(', ', $detalles) . ")";
        }

        // Si hay referencia bancaria
        if ($pedido->referenciaBancaria) {
            $ref = $pedido->referenciaBancaria;
            $detalle = $metodoBase;

            if ($ref->referencia) {
                $detalle .= " (Ref: {$ref->referencia})";
            }

            if ($ref->monto_efectivo > 0 || $ref->monto_restante > 0) {
                $partes = [];
                if ($ref->monto_efectivo > 0) {
                    $partes[] = "Efectivo: " . format_money($ref->monto_efectivo);
                }
                if ($ref->monto_restante > 0) {
                    $partes[] = "Restante: " . format_money($ref->monto_restante);
                }
                $detalle .= " [" . implode(' + ', $partes) . "]";
            }

            return $detalle;
        }


        return $metodoBase;
    }

    /**
     * Obtener el detalle del método de pago formateado para el cliente (estilo numerado)
     */
    private function obtenerDetalleMetodoPagoCliente($pedido)
    {
        $metodoBase = $pedido->metodo_pago;

        // Si hay pagos asociados (pago mixto)
        if ($pedido->pagos && $pedido->pagos->count() > 0) {
            $detalles = [];
            $contador = 1;

              $tasaDelDia = \App\Models\ExchangeRate::whereDate('date', $pedido->created_at->toDateString())
                ->orderBy('fetch_time', 'desc')
                ->value('usd_rate') ?? 1;
            $montoUsdEquivalente = $pedido->total_usd * $tasaDelDia;

            foreach ($pedido->pagos as $pago) {
                $monto = format_money($pago->total_usd);
                if ($pedido->moneda == 'bs') {
                    number_format($pago->total_bs * $tasaDelDia, 2) . "Bs";
                } else {
                   $monto;
                }


                $detallePago = "{$contador}. {$pago->metodo_pago}: {$monto}";

                // Agregar referencia si existe
                if ($pago->referencia_bancaria) {
                    $detallePago .= " (Ref: {$pago->referencia_bancaria})";
                }

                $detalles[] = $detallePago;
                $contador++;
            }

            return "💳 **Método de pago:**\n" . implode("\n", $detalles);
        }

        // Si hay referencia bancaria con montos divididos
        if ($pedido->referenciaBancaria) {
            $ref = $pedido->referenciaBancaria;
            $detalles = [];
            $contador = 1;

            // Si hay monto en efectivo
            if ($ref->monto_efectivo > 0) {
                $detalles[] = "{$contador}. Efectivo: " . format_money($ref->monto_efectivo);
                $contador++;
            }

            // El método principal (transferencia, pago móvil, etc.)
            $metodoPrincipal = $metodoBase;
            if ($ref->referencia) {
                $metodoPrincipal .= " (Ref: {$ref->referencia})";
            }

            $montoRestante = $ref->monto_restante ?? 0;
            if ($montoRestante > 0) {
                $metodoPrincipal .= ": " . format_money($montoRestante);
            }

            $detalles[] = "{$contador}. {$metodoPrincipal}";

            return "💳 **Método de pago:**\n" . implode("\n", $detalles);
        }

        // Método de pago simple
        return "💳 **Método de pago:** {$metodoBase}";
    }

    private function enviarCancelacionPedido(Pedido $pedido, $motivo = null)
    {
        $cliente = $pedido->user;
        if (!$cliente->telefono) return false;

        $message = "Hola {$cliente->name} 😔\n\n";
        $message .= "Lamentamos informarte que tu pedido ha sido cancelado.\n\n";
        $message .= "📋 **Detalles del pedido:**\n";
        $message .= "Código: {$pedido->codigo}\n";
        $message .= "Total: \${$pedido->total_usd}\n\n";

        if ($motivo) {
            $message .= "**Motivo:** {$motivo}\n\n";
        }

        $message .= "Si tienes alguna pregunta, no dudes en contactarnos.\n\n";
        $message .= "Disculpas por las molestias ocasionadas.";

        $phone = $this->enviarWhatsAppConReintento($cliente->telefono);

        // Enviar con sistema de reintentos
        $resultado = $this->enviarWhatsAppConReintento($phone, $message);

        if (!$resultado['exito']) {
            $this->guardarMensajePendiente($pedido->id, $phone, $message, 'cancelacion_pedido', $resultado['error']);
        }

        return $resultado;
    }

    private function enviarConfirmacionEntrega(Pedido $pedido)
    {
        $cliente = $pedido->user;
        if (!$cliente->telefono) return false;

        $telefonoLimpio = $this->cleanPhoneNumber($cliente->telefono);

        $message = "¡Hola {$cliente->name}! ✅\n\n";
        $message .= "Tu pedido ha sido entregado exitosamente.\n\n";
        $message .= "📋 **Pedido:** {$pedido->codigo}\n";
        $message .= "💰 **Total:** \${$pedido->total_usd}\n\n";
        $message .= "¡Gracias por elegirnos! Esperamos verte pronto. 🎉";

        // Enviar con sistema de reintentos
        $resultado = $this->enviarWhatsAppConReintento($telefonoLimpio, $message);

        if (!$resultado['exito']) {
            $this->guardarMensajePendiente($pedido->id, $telefonoLimpio, $message, 'confirmacion_entrega', $resultado['error']);
        }

        return $resultado;
    }

    /**
     * Enviar notificación de reversión a cliente y empleados
     */
    private function enviarNotificacionReversion(Pedido $pedido, $motivo, $tipoReversion)
    {
        $resultados = [];

        // Notificar al cliente
        $resultadoCliente = $this->enviarNotificacionReversionCliente($pedido, $motivo, $tipoReversion);
        if ($resultadoCliente) {
            $resultados['cliente'] = $resultadoCliente;
        }

        // Notificar a empleados asignados
        $resultadoEmpleados = $this->enviarNotificacionReversionEmpleados($pedido, $motivo, $tipoReversion);
        if ($resultadoEmpleados) {
            $resultados['empleados'] = $resultadoEmpleados;
        }

        return $resultados;
    }

    /**
     * Enviar notificación de reversión al cliente
     */
    private function enviarNotificacionReversionCliente(Pedido $pedido, $motivo, $tipoReversion)
    {
        $cliente = $pedido->user;
        if (!$cliente->telefono) return false;

        $titulo = match($tipoReversion) {
            'cancelacion' => '❌ Pedido Cancelado',
            'cambio' => '🔁 Solicitud de Cambio',
            'devolucion' => '🔄 Devolución de Pedido',
            default => '📋 Pedido en Revisión'
        };

        $mensajeInicial = match($tipoReversion) {
            'cancelacion' => 'Tu pedido ha sido cancelado',
            'cambio' => 'Has solicitado un cambio en tu pedido',
            'devolucion' => 'Has solicitado la devolución de tu pedido',
            default => 'Tu pedido está en revisión'
        };

        $message = "¡Hola {$cliente->name}! {$titulo}\n\n";
        $message .= "{$mensajeInicial}.\n\n";

        $message .= "📋 **Información del pedido:**\n";
        $message .= "Código: {$pedido->codigo}\n";
       if ($pedido->moneda == 'usd') {
         $message .= "Total: \${$pedido->total_usd}\n";
       } else {
         $message .= "Total: \${$pedido->total_bs}\n";
       }


        // Obtener detalle del método de pago
        $detallePago = $this->obtenerDetalleMetodoPagoCliente($pedido);
        $message .= $detallePago;

        if ($motivo) {
            $message .= "\n📝 **Motivo:** {$motivo}\n";
        }

        $message .= "\n⏰ **Fecha de solicitud:** " . now()->format('d/m/Y H:i') . "\n";

        $mensajeFinal = match($tipoReversion) {
            'cancelacion' => 'Procesaremos tu cancelación y te notificaremos sobre el reembolso.',
            'cambio' => 'Nuestro equipo revisará tu solicitud y se contactará contigo para coordinar los cambios.',
            'devolucion' => 'Procesaremos tu devolución y te notificaremos sobre los pasos a seguir.',
            default => 'Nuestro equipo revisará tu solicitud y se contactará contigo.'
        };

        $message .= "\n{$mensajeFinal}\n\n";
        $message .= "¡Gracias por tu comprensión! 🙏";

        // Enviar con sistema de reintentos
        $resultado = $this->enviarWhatsAppConReintento($cliente->telefono, $message);

        if (!$resultado['exito']) {
            $this->guardarMensajePendiente($pedido->id, $cliente->telefono, $message, 'reversion_cliente', $resultado['error']);
        }

        return $resultado;
    }

    /**
     * Enviar notificación de reversión a empleados asignados
     */
    private function enviarNotificacionReversionEmpleados(Pedido $pedido, $motivo, $tipoReversion)
    {
        $empleados = $pedido->empleados;
        if ($empleados->isEmpty()) return false;

        $titulo = match($tipoReversion) {
            'cancelacion' => '⚠️ Pedido Cancelado',
            'cambio' => '🔁 Solicitud de Cambio',
            'devolucion' => '🔄 Devolución de Pedido',
            default => '📋 Pedido en Revisión'
        };

        $tipoAccion = match($tipoReversion) {
            'cancelacion' => 'cancelado',
            'cambio' => 'en proceso de cambio',
            'devolucion' => 'en proceso de devolución',
            default => 'en revisión'
        };

        $resultados = [];
        foreach ($empleados as $empleado) {
            if (!$empleado->telefono) continue;

            $message = "¡Hola {$empleado->nombres} {$empleado->apellidos}! {$titulo}\n\n";
            $message .= "El siguiente pedido está {$tipoAccion}:\n\n";

            $message .= "📋 **Detalles del pedido:**\n";
            $message .= "Código: {$pedido->codigo}\n";
            $message .= "Cliente: {$pedido->user->name}\n";
            $message .= "Teléfono: {$pedido->user->telefono}\n";
            $message .= "Total: \${$pedido->total_usd}\n";

            if ($motivo) {
                $message .= "\n📝 **Motivo del cliente:** {$motivo}\n";
            }

            $mensajeAccion = match($tipoReversion) {
                'cancelacion' => 'Por favor, detén cualquier entrega pendiente y notifica a administración.',
                'cambio' => 'Por favor, contacta al cliente para coordinar los cambios solicitados.',
                'devolucion' => 'Por favor, contacta al cliente para coordinar la devolución.',
                default => 'Por favor, contacta al cliente para más detalles.'
            };

            $message .= "\n{$mensajeAccion}\n\n";
            $message .= "Fecha de solicitud: " . now()->format('d/m/Y H:i');

            // Enviar con sistema de reintentos
            $resultado = $this->enviarWhatsAppConReintento($empleado->telefono, $message);

            if (!$resultado['exito']) {
                $this->guardarMensajePendiente($pedido->id, $empleado->telefono, $message, 'reversion_empleado', $resultado['error']);
            }

            $resultados[] = $resultado;
        }

        return !empty($resultados) ? $resultados : false;
    }

      /**
     * Limpiar y formatear número de teléfono
     */
    public function cleanPhoneNumber(string $phoneNumber): string
    {
        // Remover espacios, guiones, paréntesis y símbolos
        $cleanNumber = preg_replace('/[\s\-\(\)\+]/', '', $phoneNumber);

        // Si el número empieza con 0, removerlo (formato local venezolano)
        if (substr($cleanNumber, 0, 1) === '0') {
            $cleanNumber = substr($cleanNumber, 1);
        }

        // Si no tiene código de país y parece ser venezolano (10 dígitos que empiezan con 2 o 4), agregar 58
        if (strlen($cleanNumber) === 10 && in_array(substr($cleanNumber, 0, 1), ['2', '4'])) {
            $cleanNumber = '58' . $cleanNumber;
        }

        // Si ya tiene código de país (58) pero no empieza con 58, asegurar que tenga 58
        if (strlen($cleanNumber) === 11 && substr($cleanNumber, 0, 2) !== '58' && in_array(substr($cleanNumber, 0, 1), ['2', '4'])) {
            $cleanNumber = '58' . substr($cleanNumber, 1);
        }

        // Si el número tiene 9 dígitos después de quitar el 0 y empieza con 2 o 4, agregar 58
        if (strlen($cleanNumber) === 9 && in_array(substr($cleanNumber, 0, 1), ['2', '4'])) {
            $cleanNumber = '58' . $cleanNumber;
        }

        return $cleanNumber;
    }
}
