<?php

namespace App\Services;

use App\Models\Caja;
use App\Models\ExchangeRate;
use App\Models\Empresa;
use App\Models\Sucursal;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class CajaTasaAutomationService
{
    private ExchangeRateService $exchangeRateService;
    private CajaTasaConfigService $configService;

    public function __construct()
    {
        $this->exchangeRateService = new ExchangeRateService();
        $this->configService = new CajaTasaConfigService();
    }

    /**
     * Ejecutar automatizaciones según horario
     */
    public function ejecutarAutomatizaciones(): array
    {
        $resultados = [];
        $config = $this->configService->getConfiguracionActual();

        // 1. Verificar y actualizar tasa si es necesario
        if ($config['es_actualizacion_tasa'] || !$this->tasaActualizada()) {
            $resultados['tasa'] = $this->actualizarTasaAutomaticamente();
        }

        // 2. Realizar corte de caja si es la hora
        if ($config['hora_corte_caja']) {
            $resultados['corte_caja'] = $this->realizarCorteCajaAutomatico($config['hora_corte_caja']);
        }

        // 3. Verificar cajas abiertas y crear caja diaria si es necesario
        $resultados['verificacion_cajas'] = $this->verificarYCrearCajasDiarias();

        return $resultados;
    }

    /**
     * Actualizar tasa de cambio automáticamente
     */
    public function actualizarTasaAutomaticamente(): array
    {
        $resultado = [
            'exitoso' => false,
            'mensaje' => '',
            'tasa' => null,
            'hora' => Carbon::now()->format('H:i:s')
        ];

        try {
            Log::info('Iniciando actualización automática de tasa');
            
            $success = $this->exchangeRateService->fetchAndStoreRates();
            
            if ($success) {
                $todayRate = ExchangeRate::getTodayRate();
                
                if ($todayRate) {
                    $resultado['exitoso'] = true;
                    $resultado['tasa'] = $todayRate->usd_rate;
                    $resultado['mensaje'] = "Tasa actualizada automáticamente: USD = {$todayRate->usd_rate} Bs. (Fuente: {$todayRate->source})";
                    
                    Log::info('Tasa actualizada exitosamente', [
                        'tasa' => $todayRate->usd_rate,
                        'fuente' => $todayRate->source
                    ]);
                } else {
                    $resultado['mensaje'] = 'No se pudo obtener la tasa del día después de la actualización';
                    Log::warning('No se encontró tasa del día después de actualización');
                }
            } else {
                $resultado['mensaje'] = 'No se pudo obtener la tasa. Verifique la conexión a internet.';
                Log::error('Falló la obtención de tasas desde el servicio externo');
            }
        } catch (\Exception $e) {
            $resultado['mensaje'] = 'Error técnico al actualizar tasa: ' . $e->getMessage();
            Log::error('Error en actualización automática de tasa', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }

        return $resultado;
    }

    /**
     * Realizar corte de caja automático
     */
    public function realizarCorteCajaAutomatico(string $horaCorte): array
    {
        $resultado = [
            'exitoso' => false,
            'mensaje' => '',
            'cortes_realizados' => [],
            'hora_corte' => $horaCorte,
            'hora_ejecucion' => Carbon::now()->format('H:i:s')
        ];

        try {
            Log::info("Iniciando corte de caja automático a las {$horaCorte}");
            
            $empresas = Empresa::all();
            $totalCortes = 0;

            foreach ($empresas as $empresa) {
                $sucursales = Sucursal::where('empresa_id', $empresa->id)->get();
                
                foreach ($sucursales as $sucursal) {
                    $corte = $this->realizarCorteCajaSucursal($empresa->id, $sucursal->id, "Corte automático a las {$horaCorte}");
                    
                    if ($corte['exitoso']) {
                        $resultado['cortes_realizados'][] = [
                            'empresa' => $empresa->nombre,
                            'sucursal' => $sucursal->nombre,
                            'caja_id' => $corte['caja_id'],
                            'monto_final' => $corte['monto_final']
                        ];
                        $totalCortes++;
                    }
                }
            }

            $resultado['exitoso'] = true;
            $resultado['mensaje'] = "Se realizaron {$totalCortes} cortes de caja automáticos";
            
            Log::info("Corte de caja automático completado", [
                'total_cortes' => $totalCortes,
                'hora_corte' => $horaCorte
            ]);

        } catch (\Exception $e) {
            $resultado['mensaje'] = 'Error en corte de caja automático: ' . $e->getMessage();
            Log::error('Error en corte de caja automático', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }

        return $resultado;
    }

    /**
     * Realizar corte de caja para una sucursal específica
     */
    private function realizarCorteCajaSucursal(int $empresaId, int $sucursalId, string $observaciones): array
    {
        $resultado = [
            'exitoso' => false,
            'mensaje' => '',
            'caja_id' => null,
            'monto_final' => 0
        ];

        try {
            // Buscar caja abierta
            $cajaAbierta = Caja::obtenerCajaAbierta($empresaId, $sucursalId);
            
            if ($cajaAbierta) {
                // Cerrar caja actual
                $cajaAbierta->cerrar($observaciones);
                
                // Crear nueva caja con el monto final de la anterior
                $nuevaCaja = Caja::crearCajaDiaria(
                    $empresaId,
                    $sucursalId,
                    $cajaAbierta->monto_final,
                    'Apertura automática después de corte',
                    1 // Usuario sistema
                );

                $resultado['exitoso'] = true;
                $resultado['caja_id'] = $nuevaCaja->id;
                $resultado['monto_final'] = $cajaAbierta->monto_final;
                $resultado['mensaje'] = 'Corte realizado exitosamente';
            } else {
                // No hay caja abierta, crear una nueva
                $nuevaCaja = Caja::crearCajaDiaria(
                    $empresaId,
                    $sucursalId,
                    0,
                    'Apertura automática - no había caja abierta',
                    1 // Usuario sistema
                );

                $resultado['exitoso'] = true;
                $resultado['caja_id'] = $nuevaCaja->id;
                $resultado['monto_final'] = 0;
                $resultado['mensaje'] = 'No había caja abierta, se creó una nueva';
            }

        } catch (\Exception $e) {
            $resultado['mensaje'] = 'Error al realizar corte: ' . $e->getMessage();
            Log::error('Error en corte de caja sucursal', [
                'empresa_id' => $empresaId,
                'sucursal_id' => $sucursalId,
                'error' => $e->getMessage()
            ]);
        }

        return $resultado;
    }

    /**
     * Verificar y crear cajas diarias si es necesario
     */
    public function verificarYCrearCajasDiarias(): array
    {
        $resultado = [
            'exitoso' => false,
            'mensaje' => '',
            'cajas_creadas' => []
        ];

        try {
            $hoy = Carbon::today();
            $empresas = Empresa::all();
            $totalCreadas = 0;

            foreach ($empresas as $empresa) {
                $sucursales = Sucursal::where('empresa_id', $empresa->id)->get();
                
                foreach ($sucursales as $sucursal) {
                    // Verificar si ya existe caja para hoy
                    $cajaHoy = Caja::where('empresa_id', $empresa->id)
                        ->where('sucursal_id', $sucursal->id)
                        ->whereDate('fecha', $hoy)
                        ->first();

                    if (!$cajaHoy) {
                        // Crear caja diaria
                        $nuevaCaja = Caja::crearCajaDiaria(
                            $empresa->id,
                            $sucursal->id,
                            0,
                            'Caja diaria automática',
                            1 // Usuario sistema
                        );

                        $resultado['cajas_creadas'][] = [
                            'empresa' => $empresa->nombre,
                            'sucursal' => $sucursal->nombre,
                            'caja_id' => $nuevaCaja->id
                        ];
                        $totalCreadas++;
                    }
                }
            }

            $resultado['exitoso'] = true;
            $resultado['mensaje'] = "Se crearon {$totalCreadas} cajas diarias automáticas";

        } catch (\Exception $e) {
            $resultado['mensaje'] = 'Error al verificar cajas: ' . $e->getMessage();
            Log::error('Error en verificación de cajas diarias', [
                'error' => $e->getMessage()
            ]);
        }

        return $resultado;
    }

    /**
     * Verificar si la tasa está actualizada
     */
    private function tasaActualizada(): bool
    {
        $todayRate = ExchangeRate::getTodayRate();
        
        if (!$todayRate) {
            return false;
        }

        // Verificar que la tasa no tenga más de 6 horas
        return !$this->configService->tasaEstaDesactualizada($todayRate->fetch_time);
    }
}