<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use App\Traits\FiltersByEmpresaSucursal;
use Carbon\Carbon;

class Promotion extends Model
{
    use HasFactory, FiltersByEmpresaSucursal;

    protected $fillable = [
        'empresa_id',
        'name',
        'description',
        'type',
        'discount_type',
        'discount_value',
        'min_purchase_amount',
        'max_discount_amount',
        'starts_at',
        'ends_at',
        'usage_limit',
        'usage_count',
        'is_active',
        'is_automatic',
        'priority',
        'stackable',
        'applies_to',
        'metadata',
    ];

    protected $casts = [
        'discount_value' => 'decimal:2',
        'min_purchase_amount' => 'decimal:2',
        'max_discount_amount' => 'decimal:2',
        'starts_at' => 'datetime',
        'ends_at' => 'datetime',
        'usage_limit' => 'integer',
        'usage_count' => 'integer',
        'is_active' => 'boolean',
        'is_automatic' => 'boolean',
        'priority' => 'integer',
        'stackable' => 'boolean',
        'metadata' => 'array',
    ];

    public function empresa(): BelongsTo
    {
        return $this->belongsTo(Empresa::class);
    }

    public function coupons(): HasMany
    {
        return $this->hasMany(Coupon::class);
    }

    public function conditions(): HasMany
    {
        return $this->hasMany(PromotionCondition::class);
    }

    public function productos(): BelongsToMany
    {
        return $this->belongsToMany(Producto::class, 'promotion_product')
            ->withTimestamps();
    }

    public function categorias(): BelongsToMany
    {
        return $this->belongsToMany(Categoria::class, 'promotion_category')
            ->withTimestamps();
    }

    public function pedidoDiscounts(): HasMany
    {
        return $this->hasMany(PedidoDiscount::class);
    }

    public function scopeActive($query, $at = null)
    {
        $at = $at ?? Carbon::now();

        return $query->where('is_active', true)
            ->where(function ($q) use ($at) {
                $q->whereNull('starts_at')->orWhere('starts_at', '<=', $at);
            })
            ->where(function ($q) use ($at) {
                $q->whereNull('ends_at')->orWhere('ends_at', '>=', $at);
            });
    }

    public function scopeAutomatic($query)
    {
        return $query->where('is_automatic', true);
    }

    public function isValid($at = null): bool
    {
        $at = $at ?? Carbon::now();

        if (!$this->is_active) {
            return false;
        }

        if ($this->starts_at && $at->lt($this->starts_at)) {
            return false;
        }

        if ($this->ends_at && $at->gt($this->ends_at)) {
            return false;
        }

        if ($this->usage_limit !== null && $this->usage_count >= $this->usage_limit) {
            return false;
        }

        return true;
    }
}
