<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use App\Traits\Multitenantable;

class ProductBatch extends Model
{
    use Multitenantable;

    protected $fillable = [
        'producto_id', 'batch_number', 'initial_quantity', 'current_quantity',
        'manufacturing_date', 'expiry_date', 'supplier', 'cost_per_unit',
        'status', 'notes', 'empresa_id'
    ];

    protected $casts = [
        'manufacturing_date' => 'date',
        'expiry_date' => 'date',
        'cost_per_unit' => 'decimal:2',
        'initial_quantity' => 'integer',
        'current_quantity' => 'integer'
    ];

    public function producto()
    {
        return $this->belongsTo(Producto::class);
    }

    public function movements()
    {
        return $this->hasMany(BatchMovement::class, 'batch_id');
    }

    public function empresa()
    {
        return $this->belongsTo(Empresa::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active')->where('current_quantity', '>', 0);
    }

    public function scopeExpiringSoon($query, $days = 30)
    {
        return $query->where('expiry_date', '<=', now()->addDays($days))
                    ->where('status', 'active');
    }

    public function scopeExpired($query)
    {
        return $query->where('expiry_date', '<', now())
                    ->where('status', 'active');
    }

    // Métodos
    public function isExpired()
    {
        return $this->expiry_date < now();
    }

    public function daysUntilExpiry()
    {
        return now()->diffInDays($this->expiry_date, false);
    }

    public function reduceQuantity($quantity, $reason = null, $reference = null)
    {
        if ($this->current_quantity >= $quantity) {
            $this->current_quantity -= $quantity;
            $this->save();

            // Registrar movimiento
            $this->movements()->create([
                'type' => 'out',
                'quantity' => $quantity,
                'reason' => $reason,
                'reference' => $reference,
                'user_id' => auth()->id(),
                'empresa_id' => $this->empresa_id
            ]);

            return true;
        }
        return false;
    }
}