<?php

namespace App\Livewire\Front\Resena;

use Livewire\Component;
use App\Models\Resena as ResenaModel;
use App\Models\Pedido;
use Illuminate\Support\Facades\Auth;

class Index extends Component
{
    public $pedidoId;
    public $calificacion = 5;
    public $comentario = '';
    public $pedido;

    protected $rules = [
        'calificacion' => 'required|integer|min:1|max:5',
        'comentario' => 'nullable|string|max:500',
    ];

    protected $messages = [
        'calificacion.required' => 'Por favor, selecciona una calificación',
        'calificacion.min' => 'La calificación debe ser al menos 1 estrella',
        'calificacion.max' => 'La calificación no puede ser más de 5 estrellas',
        'comentario.max' => 'El comentario no puede exceder 500 caracteres',
    ];

    public function mount($pedidoId = null)
    {
        if ($pedidoId) {
            $this->pedidoId = $pedidoId;
            $this->pedido = Pedido::with('detalles.producto')->findOrFail($pedidoId);

            // Verificar que el pedido pertenece al usuario actual y está entregado
            if ($this->pedido->userId != Auth::id() || $this->pedido->estado != 'entregado') {
                abort(403, 'No tienes permiso para acceder a este pedido');
            }
        }
    }

    public function render()
    {
        $resenas = null;
        if ($this->pedidoId) {
            $resenas = ResenaModel::where('pedido_id', $this->pedidoId)
                ->where('aprobada', true)
                ->with('user')
                ->latest()
                ->get();
        }

        return view('livewire.front.resena', compact('resenas'));
    }


    public function guardarResena()
    {
        $this->validate();

        // Verificar si ya existe una reseña para este pedido
        $resenaExistente = ResenaModel::where('pedido_id', $this->pedidoId)
            ->where('user_id', Auth::id())
            ->first();

        if ($resenaExistente) {
            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'Ya has dejado una reseña para este pedido',
                'icon' => 'error',
            ]);
            return;
        }

        ResenaModel::create([
            'pedido_id' => $this->pedidoId,
            'user_id' => Auth::id(),
            'calificacion' => $this->calificacion,
            'comentario' => $this->comentario,
            'aprobada' => true, // Aprobada automáticamente para usuarios autenticados
        ]);

        $this->dispatch('alert', [
            'title' => '¡Gracias!',
            'text' => 'Tu reseña ha sido publicada exitosamente',
            'icon' => 'success',
        ]);

        // Refrescar las reseñas
        $this->render();
    }

    public function getEstrellasProperty()
    {
        return [
            1 => '★',
            2 => '★★',
            3 => '★★★',
            4 => '★★★★',
            5 => '★★★★★',
        ];
    }
}
