<?php

namespace App\Livewire\Front\Cart;

use App\Models\Cart;
use App\Models\Cupones;
use App\Models\DetallePedido;
use App\Models\Pedido;
use App\Models\Producto;
use App\Models\Serie;
use App\Models\Tasa;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Redirect;
use Livewire\Component;

class Index extends Component
{
    protected $listeners = ['cartUpdated' => '$refresh'];

    public $latitud, $longitud;
    public
        $city,
        $currency,
        $cupon,
        $cuponame,
        $country,
        $county,
        $municipality,
        $road,
        $postcode,
        $state,
        $metodo_pago,
        $propina,
        $ubicacion,
        $office;


    function mount()
    {


        $this->currency = (session()->has('currency')) ? session()->get('currency') : 'usd';

        $pedido = Pedido::where('userId', Auth::id())->count();



        $this->propina = 0.00;
    }

    public function render()
    {
        $cartItems = \Cart::session(userID())->getContent()->sortBy('id');

        $cartTotal = \Cart::session(userID())->getTotal();
        $cartCount = \Cart::session(userID())->getTotalQuantity();

        return view('livewire.front.cart.index', compact('cartItems', 'cartTotal', 'cartCount'))
            ->layout('layouts.front.app');
    }

    public function refreshCart()
    {
        // Método para forzar actualización
    }

    public function increaseQuantity($itemId)
    {
        try {
            // Obtener todos los items del carrito
            $cartItems = \Cart::session(userID())->getContent();

            // Verificar que el producto existe antes de actualizar
            $cartItem = $cartItems->firstWhere('id', $itemId);

            if (!$cartItem) {
                $this->dispatch('alert', [
                    'title' => 'Error',
                    'text' => 'Producto no encontrado en el carrito',
                    'icon' => 'error',
                ]);
                return;
            }

            // Nueva cantidad (aumentar en 1)
            $newQuantity = $cartItem->quantity + 1;

            // Verificar disponibilidad de stock si es relevante
            if (isset($cartItem->associatedModel) && $cartItem->associatedModel) {
                $product = $cartItem->associatedModel;
                if (isset($product->quantity) && $newQuantity > $product->quantity) {
                    $this->dispatch('alert', [
                        'title' => 'Stock insuficiente',
                        'text' => 'Solo hay ' . $product->quantity . ' unidades disponibles',
                        'icon' => 'warning',
                    ]);
                    return;
                }
            }

            // Actualizar la cantidad del producto específico
            // Usar quantity: 1 para aumentar en 1 unidad
            \Cart::update($itemId, [
                'quantity' => 1, // Aumentar en 1
            ]);

            // Recargar los datos del carrito
            $this->dispatch('cartUpdated');

        } catch (\Exception $e) {
            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudo actualizar la cantidad: ' . $e->getMessage(),
                'icon' => 'error',
            ]);
        }
    }

    public function decreaseQuantity($itemId)
    {
        try {
            // Obtener todos los items del carrito
            $cartItems = \Cart::session(userID())->getContent();

            // Verificar que el producto existe antes de actualizar
            $cartItem = $cartItems->firstWhere('id', $itemId);

            if (!$cartItem) {
                $this->dispatch('alert', [
                    'title' => 'Error',
                    'text' => 'Producto no encontrado en el carrito',
                    'icon' => 'error',
                ]);
                return;
            }

            // Si la cantidad es 1, no permitir disminuir más
            if ($cartItem->quantity <= 1) {
                $this->dispatch('alert', [
                    'title' => 'Advertencia',
                    'text' => 'La cantidad mínima es 1',
                    'icon' => 'warning',
                ]);
                return;
            }

            // Actualizar la cantidad del producto específico
            // Usar quantity: -1 para disminuir en 1 unidad
            \Cart::update($itemId, [
                'quantity' => -1, // Disminuir en 1
            ]);

            // Recargar los datos del carrito
            $this->dispatch('cartUpdated');

        } catch (\Exception $e) {
            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudo actualizar la cantidad: ' . $e->getMessage(),
                'icon' => 'error',
            ]);
        }
    }

    public function updateQuantity($itemId, $newQuantity = null)
    {
        try {
            // Si no se proporciona una nueva cantidad, mantener la cantidad actual (para otros usos del método)
            if ($newQuantity === null) {
                $quantity = 1;
            } else {
                $quantity = $newQuantity;
            }

            // Validar que la cantidad sea un número positivo
            if (!is_numeric($quantity) || $quantity < 0) {
                $this->dispatch('alert', [
                    'title' => 'Error',
                    'text' => 'Cantidad inválida',
                    'icon' => 'error',
                ]);
                return;
            }

            if ($quantity == 0) {
                $this->removeItem($itemId);
                return;
            }

            // Obtener todos los items del carrito
            $cartItems = \Cart::session(userID())->getContent();

            // Verificar que el producto existe antes de actualizar
            $cartItem = $cartItems->firstWhere('id', $itemId);

            if (!$cartItem) {
                $this->dispatch('alert', [
                    'title' => 'Error',
                    'text' => 'Producto no encontrado en el carrito',
                    'icon' => 'error',
                ]);
                return;
            }

            // Verificar disponibilidad de stock si es relevante
            if (isset($cartItem->associatedModel) && $cartItem->associatedModel) {
                $product = $cartItem->associatedModel;
                if (isset($product->quantity) && $quantity > $product->quantity) {
                    $this->dispatch('alert', [
                        'title' => 'Stock insuficiente',
                        'text' => 'Solo hay ' . $product->quantity . ' unidades disponibles',
                        'icon' => 'warning',
                    ]);
                    return;
                }
            }

            // Actualizar la cantidad del producto específico
            \Cart::update($itemId, [
                'quantity' => $quantity,
                //'name' => 'Nuevo Nombre del Producto',
            ]);

            // Recargar los datos del carrito
            $this->dispatch('cartUpdated');

            $this->dispatch('alert', [
                'title' => '¡Actualizado!',
                'text' => 'Cantidad actualizada correctamente',
                'icon' => 'success',
            ]);
        } catch (\Exception $e) {
            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudo actualizar la cantidad: ' . $e->getMessage(),
                'icon' => 'error',
            ]);
        }
    }

    public function removeItem($itemId)
    {
        try {
            $userId = userID();
            $cartSession = \Cart::session($userId);

            // Obtener todos los items del carrito
            $cartItems = $cartSession->getContent();

            // Verificar que el producto existe antes de eliminar
            $cartItem = $cartItems->firstWhere('id', $itemId);

            if (!$cartItem) {
                $this->dispatch('alert', [
                    'title' => 'Error',
                    'text' => 'Producto no encontrado en el carrito',
                    'icon' => 'error',
                ]);
                return;
            }

            // Almacenar información del producto para el mensaje de confirmación
            $productName = $cartItem->name;

            // Eliminar el producto específico
            \Cart::session(userID())->remove($itemId);

            // Recargar los datos del carrito
            $this->dispatch('cartUpdated');

            $this->dispatch('alert', [
                'title' => '¡Eliminado!',
                'text' => "El producto {$productName} ha sido eliminado del carrito",
                'icon' => 'success',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error al eliminar producto del carrito: ' . $e->getMessage(), [
                'user_id' => Auth::id(),
                'item_id' => $itemId,
                'session_id' => userID()
            ]);

            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudo eliminar el producto del carrito: ' . $e->getMessage(),
                'icon' => 'error',
            ]);
        }
    }

    public function clearCart()
    {
        try {
            // Verificar que el carrito no esté vacío
            $cartItems = \Cart::session(userID())->getContent();

            if ($cartItems->count() == 0) {
                $this->dispatch('alert', [
                    'title' => 'Carrito vacío',
                    'text' => 'El carrito ya está vacío',
                    'icon' => 'info',
                ]);
                return;
            }

            // Vaciar todo el carrito
            \Cart::session(userID())->clear();

            // Recargar los datos del carrito
            $this->dispatch('cartUpdated');

            $this->dispatch('alert', [
                'title' => '¡Carrito vaciado!',
                'text' => 'Todos los productos han sido eliminados del carrito',
                'icon' => 'success',
            ]);
        } catch (\Exception $e) {
            \Log::error('Error al vaciar carrito: ' . $e->getMessage(), [
                'user_id' => Auth::id()
            ]);

            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'No se pudo vaciar el carrito: ' . $e->getMessage(),
                'icon' => 'error',
            ]);
        }
    }


    function realizarPedido()
    {
        try {
            // Validaciones previas
            $cartItems = \Cart::session(userID())->getContent();
            if ($cartItems->isEmpty()) {
                $this->dispatch('alert', [
                    'title' => 'Carrito vacío',
                    'text' => 'No puedes realizar un pedido con un carrito vacío',
                    'icon' => 'warning',
                ]);
                return;
            }

            if (!$this->metodo_pago) {
                $this->dispatch('alert', [
                    'title' => 'Método de pago requerido',
                    'text' => 'Por favor, selecciona un método de pago',
                    'icon' => 'warning',
                ]);
                return;
            }

            DB::beginTransaction();

            // Obtener tasa actual
            $tasaRecord = Tasa::whereDate('created_at', date('Y-m-d'))->first();
            if (!$tasaRecord) {
                throw new \Exception('No se encontró la tasa de cambio para hoy');
            }
            $tasa = $tasaRecord->amount;

            // Generar número de serie y pedido
            $ultima_serie = Serie::first();
            if (!$ultima_serie) {
                throw new \Exception('No se encontró la configuración de series');
            }

            $ultimo_correlativo = (int)$ultima_serie->serie + 1;
            $nuevo_correlativo = str_pad($ultimo_correlativo, 5, '0', STR_PAD_LEFT);
            $nroPedido = $this->random_digits(7);

            $ultima_serie->serie = $nuevo_correlativo;
            $ultima_serie->save();

            // Calcular totales
            $cartTotal = \Cart::session(userID())->getTotal();
            $descuento = 0;
            $total = $cartTotal;

            if ($this->cuponame > 0) {
                $descuento = ($cartTotal * $this->cuponame) / 100;
                $total = $cartTotal - $descuento;
            }

            // Crear pedido principal
            $pedido = new Pedido();
            $pedido->fecha = date('Y-m-d');
            $pedido->total_usd = $total;
            $pedido->total_bs = $total * $tasa;
            $pedido->codigo = $nroPedido;
            $pedido->metodo_pago = $this->metodo_pago;
            $pedido->estado = 'Pendiente';
            $pedido->ubicacion = $this->ubicacion;
            $pedido->longitud = $this->longitud;
            $pedido->latitud = $this->latitud;
            $pedido->empresaId = Auth::user()->empresaId;
            $pedido->userId = Auth::user()->id;
            $pedido->save();

            // Crear detalles del pedido
            foreach ($cartItems as $item) {
                $producto = Producto::find($item->associatedModel->id);
                if (!$producto) {
                    throw new \Exception("Producto no encontrado: {$item->name}");
                }

                $itemTotalUSD = $item->price * $item->quantity;
                $itemDescuento = 0;
                $itemTotal = $itemTotalUSD;

                if ($this->cuponame > 0) {
                    $itemDescuento = ($itemTotalUSD * $this->cuponame) / 100;
                    $itemTotal = $itemTotalUSD - $itemDescuento;
                }

                $detalle = new DetallePedido();
                $detalle->fecha = date('Y-m-d');
                $detalle->total_usd = $itemTotal;
                $detalle->total_bs = $itemTotal * $tasa;
                $detalle->quantity = $item->quantity;
                $detalle->price = $item->price;
                $detalle->empresaId = Auth::user()->empresaId;
                $detalle->userId = Auth::user()->id;
                $detalle->productoId = $producto->id;
                $detalle->pedidoId = $pedido->id;

                // Debug: Log de información del item del carrito
                \Log::info('Guardando detalle de pedido', [
                    'item_name' => $item->name,
                    'variant_id' => $item->attributes['variant_id'] ?? null,
                    'item_attributes' => $item->attributes ?? []
                ]);

                // Guardar el nombre completo del producto (incluyendo variante si aplica)
                $detalle->product_name = $item->name;
                // Guardar ID de variante si existe
                $detalle->variant_id = $item->attributes['variant_id'] ?? null;
                $detalle->save();

                // Debug: Verificar que se guardó correctamente
                \Log::info('Detalle guardado', [
                    'detalle_id' => $detalle->id,
                    'product_name_saved' => $detalle->product_name,
                    'variant_id_saved' => $detalle->variant_id
                ]);
            }

            DB::commit();

            // Limpiar carrito después de crear el pedido
            \Cart::session(userID())->clear();

            $this->dispatch('cartUpdated');

            $this->dispatch('alert', [
                'title' => '¡Éxito!',
                'text' => '¡Pedido generado satisfactoriamente!',
                'icon' => 'success',
            ]);

            return redirect('front/pedido/detalle/' . $pedido->codigo);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error al procesar pedido: ' . $e->getMessage(), [
                'user_id' => Auth::id(),
                'exception' => $e
            ]);

            $this->dispatch('alert', [
                'title' => 'Error',
                'text' => 'Ocurrió un error al procesar tu pedido. Por favor, intenta nuevamente.',
                'icon' => 'error',
            ]);
        } catch (\Throwable $th) {
            DB::rollBack();

            \Log::error('Error fatal al procesar pedido: ' . $th->getMessage(), [
                'user_id' => Auth::id(),
                'throwable' => $th
            ]);

            $this->dispatch('alert', [
                'title' => 'Error Crítico',
                'text' => 'Ocurrió un error crítico al procesar tu pedido. Por favor, contacta al administrador.',
                'icon' => 'error',
            ]);
        }
    }

    function random_digits($length)
    {
        $result = '';

        for ($i = 0; $i < $length; $i++) {
            $result .= random_int(0, 9);
        }

        return $result;
    }

    function logout()
    {
        Auth::logout();
        Redirect::to('front/store');
    }
}
