<?php

namespace App\Livewire\Components;

use Livewire\Component;
use App\Models\Producto;
use App\Services\AIRecommendationService;

class AIChatbot extends Component
{
    public $isOpen = false;
    public $messages = [];
    public $currentMessage = '';
    public $isTyping = false;

    protected $aiService;

    public function mount()
    {
        $this->aiService = new AIRecommendationService();
        $this->messages = [
            [
                'type' => 'bot',
                'message' => '¡Hola! Soy tu asistente virtual. ¿En qué puedo ayudarte hoy?',
                'time' => now()->format('H:i')
            ]
        ];
    }

    public function toggleChat()
    {
        $this->isOpen = !$this->isOpen;
    }

    public function sendMessage()
    {
        if (empty(trim($this->currentMessage))) return;

        // Agregar mensaje del usuario
        $this->messages[] = [
            'type' => 'user',
            'message' => $this->currentMessage,
            'time' => now()->format('H:i')
        ];

        $userMessage = $this->currentMessage;
        $this->currentMessage = '';
        $this->isTyping = true;

        // Simular delay de respuesta
        $this->dispatch('scroll-to-bottom');
        
        // Procesar respuesta del bot
        $response = $this->processMessage($userMessage);
        
        $this->messages[] = [
            'type' => 'bot',
            'message' => $response,
            'time' => now()->format('H:i')
        ];

        $this->isTyping = false;
        $this->dispatch('scroll-to-bottom');
    }

    private function processMessage($message)
    {
        $message = strtolower($message);

        // Intenciones básicas
        if (str_contains($message, 'hola') || str_contains($message, 'buenos')) {
            return '¡Hola! ¿Cómo puedo ayudarte hoy? Puedo ayudarte a encontrar productos, responder preguntas sobre pedidos o recomendarte productos.';
        }

        if (str_contains($message, 'producto') || str_contains($message, 'buscar')) {
            return $this->searchProducts($message);
        }

        if (str_contains($message, 'recomendar') || str_contains($message, 'sugerir')) {
            return $this->getRecommendations();
        }

        if (str_contains($message, 'precio') || str_contains($message, 'costo')) {
            return 'Puedes ver los precios de todos nuestros productos en el catálogo. ¿Hay algún producto específico del que quieras saber el precio?';
        }

        if (str_contains($message, 'envío') || str_contains($message, 'delivery')) {
            return 'Ofrecemos envío gratuito en pedidos mayores a $50. El tiempo de entrega es de 2-5 días hábiles. ¿Necesitas más información sobre envíos?';
        }

        if (str_contains($message, 'gracias')) {
            return '¡De nada! ¿Hay algo más en lo que pueda ayudarte?';
        }

        // Respuesta por defecto
        return 'Entiendo que necesitas ayuda. Puedo ayudarte con: buscar productos, recomendaciones, información de envíos, precios y más. ¿Qué te interesa?';
    }

    private function searchProducts($message)
    {
        // Extraer términos de búsqueda
        $searchTerms = str_replace(['buscar', 'producto', 'quiero', 'necesito'], '', $message);
        $searchTerms = trim($searchTerms);

        if (empty($searchTerms)) {
            return 'Por favor, dime qué producto estás buscando. Por ejemplo: "buscar teléfonos" o "quiero una laptop".';
        }

        $products = Producto::where('name', 'like', "%{$searchTerms}%")
            ->orWhere('descripcion', 'like', "%{$searchTerms}%")
            ->where('status', 1)
            ->limit(3)
            ->get();

        if ($products->isEmpty()) {
            return "No encontré productos con '{$searchTerms}'. ¿Podrías ser más específico o probar con otros términos?";
        }

        $response = "Encontré estos productos relacionados con '{$searchTerms}':\n\n";
        foreach ($products as $product) {
            $response .= "• {$product->name} - \${$product->price}\n";
        }
        
        return $response . "\n¿Te interesa alguno de estos productos?";
    }

    private function getRecommendations()
    {
        $recommendations = $this->aiService->getRecommendations(auth()->id(), 3);
        
        if ($recommendations->isEmpty()) {
            return 'En este momento no tengo recomendaciones específicas, pero puedes explorar nuestro catálogo completo.';
        }

        $response = "Te recomiendo estos productos populares:\n\n";
        foreach ($recommendations as $product) {
            $response .= "• {$product->name} - \${$product->price}\n";
        }
        
        return $response . "\n¿Te interesa alguno de estos productos?";
    }

    public function render()
    {
        return view('livewire.components.ai-chatbot');
    }
}