<?php

namespace App\Livewire\Auth;

use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\RateLimiter;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use App\Models\ActiveSession;
use PragmaRX\Google2FA\Google2FA;

class Login extends Component
{
    public $email = '';
    public $password = '';
    public $remember = false;

    public $latitude;
    public $longitude;


    // Define un oyente para el evento 'set-coordinates'
    protected $listeners = ['setCoordinates' => 'setCoordinates'];

    // Propiedades para manejar errores de validación
    public $errors = [];

    public function rules()
    {
        return [
            'email' => 'required|string',
            'password' => 'required|string',
        ];
    }

    /**
     * Verificar si se excedió el límite de intentos de login
     */
    private function hasTooManyLoginAttempts()
    {
        return RateLimiter::tooManyAttempts($this->throttleKey(), 5);
    }

    /**
     * Obtener la clave para limitar intentos de login
     */
    private function throttleKey()
    {
        return Str::transliterate(Str::lower($this->email).'|'.request()->ip());
    }

    /**
     * Disparar evento de bloqueo por muchos intentos
     */
    private function fireLockoutEvent()
    {
        event(new \Illuminate\Auth\Events\Lockout(request()));
    }

    public function updated($field)
    {
        // Limpiar errores cuando el usuario empieza a escribir
        if (isset($this->errors[$field])) {
            unset($this->errors[$field]);
        }
    }

    public function authenticate()
    {
        $this->errors = []; // Limpiar errores anteriores

        try {
            $this->validate();
        } catch (ValidationException $e) {
            foreach ($e->errors() as $field => $messages) {
                $this->errors[$field] = $messages[0];
            }
            return;
        }

        // Verificar si se excedió el límite de intentos
        if ($this->hasTooManyLoginAttempts()) {
            $this->fireLockoutEvent();

            $seconds = RateLimiter::availableIn($this->throttleKey());
            $this->errors['email'] = trans('auth.throttle', ['seconds' => $seconds]);

            return;
        }

        // Determinar si el valor ingresado es un email o username
        $loginField = filter_var($this->email, FILTER_VALIDATE_EMAIL) ? 'email' : 'username';
        $credentials = [$loginField => $this->email, 'password' => $this->password];

        // Intentar autenticar al usuario
        if (!Auth::attempt($credentials, $this->remember)) {
            // Incrementar el contador de intentos fallidos
            RateLimiter::hit($this->throttleKey(), 300);

            // Establecer mensaje de error
            $this->errors['email'] = trans('auth.failed');
            return;
        }

        // Limpiar el historial de intentos fallidos
        RateLimiter::clear($this->throttleKey());

        // Obtener el usuario autenticado
        $user = Auth::user();

        // Generar api_token automáticamente para Administradores y Super Administradores
        if ($user->hasRole(['Administrador', 'Super Administrador'])) {
            if (empty($user->api_token)) {
                $user->api_token = \Illuminate\Support\Str::random(80);
                $user->save();
                \Log::info("API Token generado automáticamente para usuario: {$user->name} (ID: {$user->id})");
            }
        }

        // Verificar si el usuario es el super administrador (id = 1)
        if ($user->id === 1) {
            // Registrar la sesión activa
            $this->trackUserLogin();

            // Regenerar sesión para prevenir session fixation
            request()->session()->regenerate();

            // Redirigir al dashboard de super administrador
            return redirect()->route('superadmin.dashboard');
        }

        // Verificar si el usuario tiene 2FA habilitado
        if ($user->two_factor_enabled) {
            // Cerrar la sesión temporalmente
            Auth::logout();

            // Guardar información del usuario en la sesión para la verificación 2FA
            session([
                '2fa:user:id' => $user->id,
                '2fa:user:email' => $user->email
            ]);

            // Redirigir a la página de verificación 2FA
            return redirect()->route('two-factor.login');
        }

        // Registrar la sesión activa
        ActiveSession::create([
            'user_id' => Auth::id(),
            'session_id' => session()->getId(),
            'ip_address' => request()->ip(),
            'user_agent' => request()->userAgent(),
            'latitude' => $this->latitude,
            'longitude' => $this->longitude,
        ]);

        // Verificar si el usuario tiene 2FA habilitado
        if (Auth::user()->two_factor_enabled) {
            // Cerrar la sesión actual
            Auth::logout();

            // Almacenar las credenciales en la sesión para usar después de 2FA
            session()->put('login.id', Auth::id());
            session()->put('login.remember', $this->remember);
            session()->put('login.credentials', $credentials);

            // Redirigir a la página 2FA
            return redirect()->route('2fa.index');
        }

        // Disparar evento de login exitoso
        event(new \Illuminate\Auth\Events\Login(Auth::guard(), Auth::user(), $this->remember));

        // Redirigir al dashboard
        if ($user->hasRole('Administrador') || $user->hasRole('Recepcionista')) {
            return redirect()->route('admin.dashboard');
        } else {
            return redirect()->to('admin/empleados/dashboard');
        }
    }

    /**
     * Obtener información de geolocalización basada en coordenadas
     */
    private function getLocationData($ipAddress)
    {
        // Datos por defecto
        $locationData = [
            'location' => null,
            'latitude' => null,
            'longitude' => null,
        ];

        // Verificar si hay coordenadas en el componente
        if ($this->latitude && $this->longitude) {
            // Usar coordenadas del componente y hacer geocodificación inversa
            return $this->reverseGeocode($this->latitude, $this->longitude);
        }

        // Para IPs locales o cuando no hay coordenadas
        if ($ipAddress === '127.0.0.1' || $ipAddress === '::1' || strpos($ipAddress, '192.168.') === 0) {
            $locationData['location'] = 'Local';
            return $locationData;
        }

        // Si no hay coordenadas disponibles, usar valores predeterminados
        $locationData['location'] = 'Ubicación desconocida';
        return $locationData;
    }

    /**
     * Reverse geocode coordinates to get location information using Nominatim
     */
    private function reverseGeocode($lat, $lon)
    {
        $locationData = [
            'latitude' => $lat,
            'longitude' => $lon,
            'location' => null,
        ];

        try {
            // Usar Nominatim para geocodificación inversa con las coordenadas
            $url = "https://nominatim.openstreetmap.org/reverse?format=json&lat={$lat}&lon={$lon}&addressdetails=1";

            // Configurar contexto para la solicitud con User-Agent requerido
            $context = stream_context_create([
                "http" => [
                    "header" => "User-Agent: larawire/1.0\r\n",
                    "timeout" => 10
                ]
            ]);

            $response = file_get_contents($url, false, $context);
            $data = json_decode($response, true);

            if ($data && isset($data['address'])) {
                $city = $data['address']['city'] ?? $data['address']['town'] ?? $data['address']['village'] ?? 'Desconocido';
                $state = $data['address']['state'] ?? $data['address']['region'] ?? 'Desconocido';
                $country = $data['address']['country'] ?? 'Desconocido';
                $locationData['location'] = "{$city}, {$state}, {$country}";
            } else {
                $locationData['location'] = "Lat: {$lat}, Lon: {$lon}";
            }
        } catch (\Exception $e) {
            \Log::warning("Error obteniendo geolocalización para coordenadas ({$lat}, {$lon}): " . $e->getMessage());
            $locationData['location'] = "Lat: {$lat}, Lon: {$lon}";
        }

        return $locationData;
    }

    // Método para verificar si un campo tiene error
    public function hasError($field)
    {
        return isset($this->errors[$field]) && !empty($this->errors[$field]);
    }

    // Método para obtener los mensajes de error de un campo
    public function getError($field)
    {
        return $this->hasError($field) ? $this->errors[$field][0] : '';
    }

    public function render()
    {
        return view('livewire.auth.login', [
            'hasError' => $this->hasError(...),
            'getError' => $this->getError(...),
        ])->layout('components.layouts.auth-basic', ['title' => 'Login']);
    }
}
