<?php

namespace App\Livewire\Admin\Ventas;

use Livewire\Component;
use App\Models\Producto;
use App\Models\Cliente;
use App\Models\Venta;
use App\Models\DetalleVenta;
use App\Models\Caja;
use App\Models\SerieDocumento;
use Illuminate\Support\Facades\DB;
use Jantinnerezo\LivewireAlert\LivewireAlert;

class VentasConBarcode extends Component
{
    use LivewireAlert;

    // Propiedades del carrito y búsqueda
    public $cart = [];
    public $searchTerm = '';
    public $searchResults = [];
    public $barcode = '';
    public $barcodeInputFocused = true;

    // Propiedades del cliente
    public $clienteId = null;
    public $clientes = [];

    // Propiedades del documento
    public $tipoDocumento = 'boleta';
    public $serieSeleccionada = '';
    public $seriesDisponibles = [];

    // Propiedades de pago
    public $metodoPago = 'efectivo';
    public $montoPagado = 0;
    public $montoCambio = 0;

    // Propiedades de hardware
    public $printerEnabled = true;
    public $cashDrawerEnabled = true;
    public $customerDisplayEnabled = false;

    // Tipo de operación (venta, nota_venta, cotizacion)
    public $tipoOperacion = 'venta';

    protected $listeners = [
        'productSelected' => 'selectProduct',
        'barcodeScanned' => 'handleBarcodeScan',
        'focusBarcodeInput' => 'focusBarcodeInput',
        'printTicket' => 'printTicket',
        'openCashDrawer' => 'openCashDrawer',
        'updateCustomerDisplay' => 'updateCustomerDisplay'
    ];

    public function mount()
    {
        $this->cargarClientes();
        $this->cargarSeriesDisponibles();
        $this->verificarCajaAbierta();
    }

    public function cargarClientes()
    {
        $this->clientes = Cliente::where('estado', 'activo')
            ->orderBy('nombre')
            ->get();
    }

    public function cargarSeriesDisponibles()
    {
        $this->seriesDisponibles = SerieDocumento::where('tipo', $this->tipoDocumento)
            ->where('estado', 'activo')
            ->get();
    }

    public function updatedTipoDocumento()
    {
        $this->cargarSeriesDisponibles();
        $this->serieSeleccionada = '';
    }

    public function verificarCajaAbierta()
    {
        $cajaAbierta = Caja::where('estado', 'abierta')
            ->where('user_id', auth()->id())
            ->first();

        if (!$cajaAbierta) {
            $this->alert('warning', 'Debe abrir una caja antes de realizar ventas');
        }
    }

    public function handleBarcodeScan($barcode)
    {
        $this->barcode = $barcode;

        // Buscar producto por código de barras
        $producto = Producto::where('codigo', $barcode)
            ->where('estado', 'activo')
            ->first();

        if ($producto) {
            $this->selectProduct($producto->id);
            $this->searchResults = [];
            $this->barcode = '';
        } else {
            $this->alert('error', 'Producto no encontrado con el código: ' . $barcode);
        }
    }

    public function focusBarcodeInput()
    {
        $this->dispatch('focus-barcode-input');
    }

    public function searchProducts()
    {
        if (empty($this->searchTerm) && empty($this->barcode)) {
            $this->searchResults = [];
            return;
        }

        $query = Producto::where('estado', 'activo');

        // Si hay un código de barras escaneado, buscar por código exacto primero
        if (!empty($this->barcode)) {
            $productoPorBarcode = $query->where('codigo', $this->barcode)->first();
            if ($productoPorBarcode) {
                $this->selectProduct($productoPorBarcode->id);
                $this->barcode = '';
                $this->searchResults = [];
                return;
            }
        }

        // Búsqueda por término
        if (!empty($this->searchTerm)) {
            $query->where(function($q) {
                $q->where('nombre', 'like', '%' . $this->searchTerm . '%')
                  ->orWhere('codigo', 'like', '%' . $this->searchTerm . '%')
                  ->orWhere('descripcion', 'like', '%' . $this->searchTerm . '%');
            });
        }

        $this->searchResults = $query->limit(10)->get();
    }

    public function selectProduct($productoId)
    {
        $producto = Producto::find($productoId);

        if (!$producto) {
            $this->alert('error', 'Producto no encontrado');
            return;
        }

        // Verificar stock
        if ($producto->stock <= 0) {
            $this->alert('error', 'Producto sin stock');
            return;
        }

        // Buscar si ya existe en el carrito
        $existingIndex = null;
        foreach ($this->cart as $index => $item) {
            if ($item['producto_id'] == $producto->id &&
                $item['precio_unitario'] == $producto->precio_venta) {
                $existingIndex = $index;
                break;
            }
        }

        if ($existingIndex !== null) {
            // Incrementar cantidad
            if ($this->cart[$existingIndex]['cantidad'] < $producto->stock) {
                $this->cart[$existingIndex]['cantidad']++;
                $this->cart[$existingIndex]['subtotal'] =
                    $this->cart[$existingIndex]['cantidad'] *
                    $this->cart[$existingIndex]['precio_unitario'];
            } else {
                $this->alert('warning', 'No hay suficiente stock');
            }
        } else {
            // Agregar nuevo item
            $this->cart[] = [
                'producto_id' => $producto->id,
                'nombre' => $producto->nombre,
                'codigo' => $producto->codigo,
                'cantidad' => 1,
                'precio_unitario' => $producto->precio_venta,
                'subtotal' => $producto->precio_venta
            ];
        }

        $this->searchTerm = '';
        $this->searchResults = [];
        $this->calcularCambio();

        $this->alert('success', 'Producto agregado al carrito');
    }

    public function removeFromCart($index)
    {
        unset($this->cart[$index]);
        $this->cart = array_values($this->cart);
        $this->calcularCambio();
    }

    public function incrementCartItem($index)
    {
        if (isset($this->cart[$index])) {
            $item = $this->cart[$index];
            $producto = Producto::find($item['producto_id']);

            if ($item['cantidad'] < $producto->stock) {
                $this->cart[$index]['cantidad']++;
                $this->cart[$index]['subtotal'] =
                    $this->cart[$index]['cantidad'] *
                    $this->cart[$index]['precio_unitario'];
                $this->calcularCambio();
            } else {
                $this->alert('warning', 'No hay suficiente stock');
            }
        }
    }

    public function decrementCartItem($index)
    {
        if (isset($this->cart[$index])) {
            if ($this->cart[$index]['cantidad'] > 1) {
                $this->cart[$index]['cantidad']--;
                $this->cart[$index]['subtotal'] =
                    $this->cart[$index]['cantidad'] *
                    $this->cart[$index]['precio_unitario'];
            } else {
                $this->removeFromCart($index);
            }
            $this->calcularCambio();
        }
    }

    public function updatedMontoPagado()
    {
        $this->calcularCambio();
    }

    public function calcularCambio()
    {
        $total = $this->getTotalProperty();
        if ($this->metodoPago === 'efectivo' && $this->montoPagado > 0) {
            $this->montoCambio = max(0, $this->montoPagado - $total);
        } else {
            $this->montoCambio = 0;
        }
    }

    public function getTotalProperty()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function cancelSale()
    {
        $this->cart = [];
        $this->clienteId = null;
        $this->metodoPago = 'efectivo';
        $this->montoPagado = 0;
        $this->montoCambio = 0;
        $this->searchTerm = '';
        $this->searchResults = [];
        $this->barcode = '';
    }

    public function procesarVenta()
    {
        if (count($this->cart) === 0) {
            $this->alert('error', 'El carrito está vacío');
            return;
        }

        if (empty($this->serieSeleccionada)) {
            $this->alert('error', 'Debe seleccionar una serie de documento');
            return;
        }

        // Validar monto pagado para efectivo
        if ($this->metodoPago === 'efectivo' && $this->montoPagado < $this->getTotalProperty()) {
            $this->alert('error', 'El monto pagado debe ser mayor o igual al total');
            return;
        }

        $caja = Caja::where('estado', 'abierta')
            ->where('user_id', auth()->id())
            ->first();

        if (!$caja) {
            $this->alert('error', 'Debe abrir una caja antes de realizar ventas');
            return;
        }

        $serie = SerieDocumento::find($this->serieSeleccionada);

        DB::beginTransaction();

        try {
            // Crear venta
            $venta = new Venta();
            $venta->empresa_id = session('empresa_id');
            $venta->sucursal_id = session('sucursal_id');
            $venta->caja_id = $caja->id;
            $venta->cliente_id = $this->clienteId;
            $venta->user_id = auth()->id();
            $venta->tipo_documento = $this->tipoDocumento;
            $venta->serie = $serie->serie;
            $venta->correlativo = $serie->correlativo_actual + 1;
            $venta->fecha_emision = now();
            $venta->subtotal = $this->getTotalProperty();
            $venta->igv = 0;
            $venta->total = $this->getTotalProperty();
            $venta->estado = 'completado';
            $venta->metodo_pago = $this->metodoPago;
            $venta->monto_pagado = $this->montoPagado;
            $venta->cambio = $this->montoCambio;
            $venta->save();

            // Crear detalles de venta
            foreach ($this->cart as $item) {
                $detalle = new DetalleVenta();
                $detalle->venta_id = $venta->id;
                $detalle->producto_id = $item['producto_id'];
                $detalle->cantidad = $item['cantidad'];
                $detalle->precio_unitario = $item['precio_unitario'];
                $detalle->subtotal = $item['subtotal'];
                $detalle->save();

                // Actualizar stock del producto
                $producto = Producto::find($item['producto_id']);
                $producto->stock -= $item['cantidad'];
                $producto->save();
            }

            // Incrementar correlativo de la serie
            $serie->increment('correlativo_actual');

            DB::commit();

            // Abrir cajón de dinero si está habilitado y el pago es en efectivo
            if ($this->cashDrawerEnabled && $this->metodoPago === 'efectivo') {
                $this->openCashDrawer();
            }

            // Imprimir ticket si está habilitado
            if ($this->printerEnabled) {
                $this->printTicket($venta->id);
            }

            // Actualizar pantalla del cliente
            $this->updateCustomerDisplay('payment');

            // Limpiar carrito
            $this->cancelSale();

            $this->alert('success', 'Venta procesada exitosamente. Número: ' . $venta->serie . '-' . str_pad($venta->correlativo, 8, '0', STR_PAD_LEFT));

        } catch (\Exception $e) {
            DB::rollBack();
            $this->alert('error', 'Error al procesar la venta: ' . $e->getMessage());
        }
    }

    public function printTicket($ventaId = null)
    {
        if ($this->printerEnabled) {
            $this->dispatch('print-ticket', [
                'ventaId' => $ventaId,
                'cart' => $this->cart,
                'total' => $this->getTotalProperty(),
                'metodoPago' => $this->metodoPago,
                'montoPagado' => $this->montoPagado,
                'cambio' => $this->montoCambio
            ]);
        }
    }

    public function openCashDrawer()
    {
        if ($this->cashDrawerEnabled) {
            $this->dispatch('open-cash-drawer');
        }
    }

    public function updateCustomerDisplay($type = 'cart', $data = [])
    {
        if ($this->customerDisplayEnabled) {
            $defaultData = [
                'type' => $type,
                'items' => $this->cart,
                'total' => $this->getTotalProperty(),
                'paymentMethod' => $this->metodoPago,
                'montoPagado' => $this->montoPagado,
                'cambio' => $this->montoCambio
            ];

            $mergedData = array_merge($defaultData, $data);
            $this->dispatch('update-customer-display', $mergedData);
        }
    }

    public function render()
    {
        return view('livewire.admin.ventas.ventas-con-barcode', [
            'total' => $this->getTotalProperty()
        ]);
    }
}
