<?php

namespace App\Livewire\Admin\Ventas;

use App\Models\Venta;
use App\Models\Producto;
use App\Models\User;
use App\Models\Serie;
use Livewire\Component;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;

class Edit extends Component
{
    use HasDynamicLayout, HasRegionalFormatting;

    public Venta $venta;
    public $cliente_id;
    public $metodo_pago;
    public $tipo_documento;
    public $serie_id;
    public $observaciones;
    
    public $cart = [];
    public $search = '';
    public $searchResults = [];
    
    // Modal de variantes
    public $showVariantsModal = false;
    public $selectedProductForVariants = null;
    public $selectedVariant = null;
    public $variantQuantity = 1;
    
    public $clientes = [];
    public $series = [];

    protected $rules = [
        'metodo_pago' => 'required|in:efectivo,transferencia,tarjeta,pago_movil,punto_de_venta',
        'tipo_documento' => 'required|in:boleta,factura,nota_credito,recibo',
        'serie_id' => 'required|exists:series,id',
        'cart' => 'required|array|min:1',
    ];

    public function mount(Venta $venta)
    {
        $this->venta = $venta->load(['detalles.producto', 'cliente', 'serie']);
        
        $this->cliente_id = $venta->cliente_id;
        $this->metodo_pago = $venta->metodo_pago;
        $this->tipo_documento = $venta->tipo_documento;
        $this->serie_id = $venta->serie_id;
        $this->observaciones = $venta->observaciones;
        
        // Cargar carrito desde detalles existentes
        foreach ($venta->detalles as $detalle) {
            $this->cart[] = [
                'id' => $detalle->id,
                'producto_id' => $detalle->producto_id,
                'variante_id' => $detalle->variante_id,
                'nombre' => $detalle->nombre_producto,
                'codigo' => $detalle->codigo_producto,
                'cantidad' => $detalle->cantidad,
                'precio_unitario' => (float) $detalle->precio_unitario,
                'subtotal' => (float) $detalle->subtotal,
            ];
        }
        
        $this->loadData();
    }

    public function loadData()
    {
        $this->clientes = User::role('Cliente')
            ->where('empresa_id', auth()->user()->empresa_id)
            ->get();
            
        $this->series = Serie::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('tipo_documento', $this->tipo_documento)
            ->where('activo', true)
            ->get();
    }

    public function updatedTipoDocumento()
    {
        $this->loadData();
    }

    public function searchProducts()
    {
        if (strlen($this->search) >= 2) {
            $query = Producto::where('empresa_id', auth()->user()->empresa_id)
                ->where('status', true)
                ->with(['variants', 'featuredImage']);

            // Buscar por código exacto primero
            $exactMatch = clone $query;
            $productos = $exactMatch->where('code', $this->search)->limit(5)->get();

            // Si no hay coincidencia exacta, buscar por coincidencias parciales
            if ($productos->isEmpty()) {
                $productos = $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%')
                      ->orWhere('code', 'like', '%' . $this->search . '%');
                })
                ->orderBy('name', 'asc')
                ->limit(10)
                ->get();
            }

            $this->searchResults = [];
            foreach ($productos as $producto) {
                $this->searchResults[] = [
                    'id' => $producto->id,
                    'name' => $producto->name,
                    'code' => $producto->code,
                    'description' => $producto->description,
                    'price' => (float) $producto->precio,
                    'quantity' => $producto->quantity,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'has_variants' => $producto->variants && $producto->variants->count() > 0,
                ];
            }
        } else {
            $this->searchResults = [];
        }
    }

    public function openVariantsModal($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        $this->selectedProductForVariants = $producto;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
        $this->showVariantsModal = true;
    }

    public function closeVariantsModal()
    {
        $this->showVariantsModal = false;
        $this->selectedProductForVariants = null;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
    }

    public function selectVariant($variantId)
    {
        $this->selectedVariant = $variantId;
    }

    public function addVariantToCart()
    {
        if (!$this->selectedProductForVariants || !$this->selectedVariant) {
            session()->flash('error', 'Seleccione una opción del producto');
            return;
        }

        $producto = $this->selectedProductForVariants;

        if ($this->selectedVariant === 'main') {
            // Agregar producto principal
            $this->addProductToCart($producto, $this->variantQuantity);
        } else {
            // Agregar variante
            $variante = $producto->variants()->find($this->selectedVariant);
            if (!$variante) return;

            $precioVariante = $producto->precio + $variante->price_adjustment;
            $values = $variante->formatted_values;
            
            $existingIndex = collect($this->cart)->search(function ($item) use ($variante) {
                return isset($item['variante_id']) && $item['variante_id'] == $variante->id;
            });

            if ($existingIndex !== false) {
                $this->cart[$existingIndex]['cantidad'] += $this->variantQuantity;
                $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
            } else {
                $this->cart[] = [
                    'id' => null,
                    'producto_id' => $producto->id,
                    'variante_id' => $variante->id,
                    'nombre' => $producto->name . ' - ' . $variante->name . ': ' . $values,
                    'codigo' => $variante->sku ?? $producto->code,
                    'cantidad' => $this->variantQuantity,
                    'precio_unitario' => (float) $precioVariante,
                    'subtotal' => $this->variantQuantity * (float) $precioVariante,
                ];
            }
        }

        $this->closeVariantsModal();
    }

    public function addToCart($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        // Siempre abrir el modal para mostrar el producto principal y sus variantes
        $this->openVariantsModal($productoId);
    }

    private function addProductToCart($producto, $cantidad = 1)
    {
        $existingIndex = collect($this->cart)->search(function ($item) use ($producto) {
            return $item['producto_id'] == $producto->id && !isset($item['variante_id']);
        });

        if ($existingIndex !== false) {
            $this->cart[$existingIndex]['cantidad'] += $cantidad;
            $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
        } else {
            $this->cart[] = [
                'id' => null,
                'producto_id' => $producto->id,
                'nombre' => $producto->name,
                'codigo' => $producto->code,
                'cantidad' => $cantidad,
                'precio_unitario' => (float) $producto->precio,
                'subtotal' => $cantidad * (float) $producto->precio,
            ];
        }

        $this->search = '';
        $this->searchResults = [];
    }

    public function removeFromCart($index)
    {
        unset($this->cart[$index]);
        $this->cart = array_values($this->cart);
    }

    public function updateQuantity($index, $quantity)
    {
        if ($quantity < 1) {
            $this->removeFromCart($index);
            return;
        }

        $this->cart[$index]['cantidad'] = $quantity;
        $this->cart[$index]['subtotal'] = $quantity * $this->cart[$index]['precio_unitario'];
    }

    public function getTotal()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function save()
    {
        $this->validate();

        if (empty($this->cart)) {
            session()->flash('error', 'Debe agregar al menos un producto');
            return;
        }

        try {
            \DB::beginTransaction();

            // Actualizar venta
            $this->venta->update([
                'cliente_id' => $this->cliente_id,
                'metodo_pago' => $this->metodo_pago,
                'tipo_documento' => $this->tipo_documento,
                'serie_id' => $this->serie_id,
                'observaciones' => $this->observaciones,
                'total' => $this->getTotal(),
                'total_usd' => $this->getTotal(),
                'total_bs' => $this->getTotal(),
            ]);

            // Eliminar detalles existentes
            $this->venta->detalles()->delete();

            // Crear nuevos detalles
            foreach ($this->cart as $item) {
                $this->venta->detalles()->create([
                    'producto_id' => $item['producto_id'],
                    'variante_id' => $item['variante_id'] ?? null,
                    'cantidad' => $item['cantidad'],
                    'precio_unitario' => $item['precio_unitario'],
                    'subtotal' => $item['subtotal'],
                    'nombre_producto' => $item['nombre'],
                    'codigo_producto' => $item['codigo'],
                ]);
            }

            // Actualizar pago
            $this->venta->pagos()->update([
                'metodo_pago' => $this->metodo_pago,
                'monto' => $this->getTotal(),
                'monto_recibido' => $this->getTotal(),
            ]);

            \DB::commit();

            session()->flash('message', 'Venta actualizada exitosamente');
            return redirect()->route('admin.ventas.show', $this->venta->id);

        } catch (\Exception $e) {
            \DB::rollback();
            session()->flash('error', 'Error al actualizar la venta: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $this->searchProducts();
        
        return view('livewire.admin.ventas.edit')->layout($this->getLayout());
    }
}