<?php

namespace App\Livewire\Admin\Ventas;

use App\Models\Venta;
use App\Models\Producto;
use App\Models\User;
use App\Models\Caja;
use App\Models\Serie;
use Livewire\Component;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;

class Create extends Component
{
    use HasDynamicLayout, HasRegionalFormatting;

    public $codigo = '';
    public $cliente_id = null;
    public $metodo_pago = 'efectivo';
    public $tipo_documento = 'boleta';
    public $serie_id = null;
    public $observaciones = '';
    
    public $cart = [];
    public $search = '';
    public $searchResults = [];
    
    // Modal de variantes
    public $showVariantsModal = false;
    public $selectedProductForVariants = null;
    public $selectedVariant = null;
    public $variantQuantity = 1;
    
    public $clientes = [];
    public $series = [];
    public $productos = [];

    protected $rules = [
        'metodo_pago' => 'required|in:efectivo,transferencia,tarjeta,pago_movil,punto_de_venta',
        'tipo_documento' => 'required|in:boleta,factura,nota_credito,recibo',
        'serie_id' => 'required|exists:series,id',
        'cart' => 'required|array|min:1',
        'cart.*.producto_id' => 'required|exists:productos,id',
        'cart.*.cantidad' => 'required|integer|min:1',
        'cart.*.precio_unitario' => 'required|numeric|min:0',
    ];

    public function mount()
    {
        // Verificar si hay caja abierta
        $caja = Caja::obtenerCajaAbierta(auth()->user()->empresa_id, auth()->user()->sucursal_id);
        if (!$caja) {
            session()->flash('error', 'Debe abrir una caja antes de crear una venta.');
            return redirect()->route('admin.cajas.index');
        }
        
        // Inicializar configuración regional
        $this->bootHasRegionalFormatting();

        $this->codigo = 'VEN-' . time();
        $this->loadData();
    }

    public function loadData()
    {
        $this->clientes = User::role('Cliente')
            ->where('empresa_id', auth()->user()->empresa_id)
            ->get();
            
        $this->series = Serie::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('tipo_documento', $this->tipo_documento)
            ->where('activo', true)
            ->get();
            
        if ($this->series->count() > 0) {
            $this->serie_id = $this->series->first()->id;
        }
    }

    public function updatedTipoDocumento()
    {
        $this->loadData();
    }

    public function searchProducts()
    {
        if (strlen($this->search) >= 2) {
            $query = Producto::where('empresa_id', auth()->user()->empresa_id)
                ->where('status', true)
                ->with(['variants', 'featuredImage']);

            // Buscar por código exacto primero
            $exactMatch = clone $query;
            $productos = $exactMatch->where('code', $this->search)->limit(5)->get();

            // Si no hay coincidencia exacta, buscar por coincidencias parciales
            if ($productos->isEmpty()) {
                $productos = $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->search . '%')
                      ->orWhere('description', 'like', '%' . $this->search . '%')
                      ->orWhere('code', 'like', '%' . $this->search . '%');
                })
                ->orderBy('name', 'asc')
                ->limit(10)
                ->get();
            }

            $this->searchResults = [];
            foreach ($productos as $producto) {
                $this->searchResults[] = [
                    'id' => $producto->id,
                    'name' => $producto->name,
                    'code' => $producto->code,
                    'description' => $producto->description,
                    'price' => (float) $producto->precio,
                    'quantity' => $producto->quantity,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'has_variants' => $producto->variants && $producto->variants->count() > 0,
                ];
            }
        } else {
            $this->searchResults = [];
        }
    }

    public function openVariantsModal($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        $this->selectedProductForVariants = $producto;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
        $this->showVariantsModal = true;
    }

    public function closeVariantsModal()
    {
        $this->showVariantsModal = false;
        $this->selectedProductForVariants = null;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
    }

    public function selectVariant($variantId)
    {
        $this->selectedVariant = $variantId;
    }

    public function addVariantToCart()
    {
        if (!$this->selectedProductForVariants || !$this->selectedVariant) {
            session()->flash('error', 'Seleccione una opción del producto');
            return;
        }

        $producto = $this->selectedProductForVariants;

        if ($this->selectedVariant === 'main') {
            // Agregar producto principal
            $this->addProductToCart($producto, $this->variantQuantity);
        } else {
            // Agregar variante
            $variante = $producto->variants()->find($this->selectedVariant);
            if (!$variante) return;

            $precioVariante = $producto->precio + $variante->price_adjustment;
            $values = $variante->formatted_values;
            
            $existingIndex = collect($this->cart)->search(function ($item) use ($variante) {
                return isset($item['variante_id']) && $item['variante_id'] == $variante->id;
            });

            if ($existingIndex !== false) {
                $this->cart[$existingIndex]['cantidad'] += $this->variantQuantity;
                $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
            } else {
                $this->cart[] = [
                    'producto_id' => $producto->id,
                    'variante_id' => $variante->id,
                    'nombre' => $producto->name . ' - ' . $variante->name . ': ' . $values,
                    'codigo' => $variante->sku ?? $producto->code,
                    'cantidad' => $this->variantQuantity,
                    'precio_unitario' => (float) $precioVariante,
                    'subtotal' => $this->variantQuantity * (float) $precioVariante,
                ];
            }
        }

        $this->closeVariantsModal();
    }

    public function addToCart($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        // Siempre abrir el modal para mostrar el producto principal y sus variantes
        $this->openVariantsModal($productoId);
    }

    private function addProductToCart($producto, $cantidad = 1)
    {
        $existingIndex = collect($this->cart)->search(function ($item) use ($producto) {
            return $item['producto_id'] == $producto->id && !isset($item['variante_id']);
        });

        if ($existingIndex !== false) {
            $this->cart[$existingIndex]['cantidad'] += $cantidad;
            $this->cart[$existingIndex]['subtotal'] = $this->cart[$existingIndex]['cantidad'] * $this->cart[$existingIndex]['precio_unitario'];
        } else {
            $this->cart[] = [
                'producto_id' => $producto->id,
                'nombre' => $producto->name,
                'codigo' => $producto->code,
                'cantidad' => $cantidad,
                'precio_unitario' => (float) $producto->precio,
                'subtotal' => $cantidad * (float) $producto->precio,
            ];
        }

        $this->search = '';
        $this->searchResults = [];
    }

    public function removeFromCart($index)
    {
        unset($this->cart[$index]);
        $this->cart = array_values($this->cart);
    }

    public function updateQuantity($index, $quantity)
    {
        if ($quantity < 1) {
            $this->removeFromCart($index);
            return;
        }

        $this->cart[$index]['cantidad'] = $quantity;
        $this->cart[$index]['subtotal'] = $quantity * $this->cart[$index]['precio_unitario'];
    }

    public function getTotal()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function save()
    {
        $this->validate();

        if (empty($this->cart)) {
            session()->flash('error', 'Debe agregar al menos un producto');
            return;
        }

        $caja = Caja::obtenerCajaAbierta(auth()->user()->empresa_id, auth()->user()->sucursal_id);
        if (!$caja) {
            session()->flash('error', 'No hay una caja abierta');
            return;
        }

        try {
            \DB::beginTransaction();

            $serie = Serie::find($this->serie_id);
            
            // Buscar el próximo número disponible
            $nextNumber = $serie->correlativo_actual + 1;
            
            while (Venta::where('serie_id', $this->serie_id)
                ->where('numero_documento', $nextNumber)
                ->where('empresa_id', auth()->user()->empresa_id)
                ->exists()) {
                $nextNumber++;
            }
            
            $venta = Venta::create([
                'codigo' => $this->codigo,
                'total' => $this->getTotal(),
                'total_usd' => $this->getTotal(),
                'total_bs' => $this->getTotal(),
                'metodo_pago' => $this->metodo_pago,
                'estado' => 'pagado',
                'tipo_documento' => $this->tipo_documento,
                'numero_documento' => $nextNumber,
                'serie_id' => $this->serie_id,
                'caja_id' => $caja->id,
                'empresa_id' => auth()->user()->empresa_id,
                'sucursal_id' => auth()->user()->sucursal_id,
                'user_id' => auth()->id(),
                'empleado_id' => auth()->id(),
                'cliente_id' => $this->cliente_id,
                'observaciones' => $this->observaciones,
            ]);

            foreach ($this->cart as $item) {
                // Verificar stock antes de crear el detalle
                if (isset($item['variante_id'])) {
                    $variante = \App\Models\ProductoVariant::find($item['variante_id']);
                    if (!$variante || $variante->quantity < $item['cantidad']) {
                        throw new \Exception("Stock insuficiente para la variante: {$item['nombre']}. Stock disponible: " . ($variante ? $variante->quantity : 0));
                    }
                } else {
                    $producto = Producto::find($item['producto_id']);
                    if (!$producto || $producto->quantity < $item['cantidad']) {
                        throw new \Exception("Stock insuficiente para el producto: {$item['nombre']}. Stock disponible: " . ($producto ? $producto->quantity : 0));
                    }
                }

                $venta->detalles()->create([
                    'producto_id' => $item['producto_id'],
                    'variante_id' => $item['variante_id'] ?? null,
                    'cantidad' => $item['cantidad'],
                    'precio_unitario' => $item['precio_unitario'],
                    'subtotal' => $item['subtotal'],
                    'nombre_producto' => $item['nombre'],
                    'codigo_producto' => $item['codigo'],
                ]);

                // Actualizar stock y registrar movimiento
                if (isset($item['variante_id'])) {
                    $variante = \App\Models\ProductoVariant::find($item['variante_id']);
                    if ($variante) {
                        $stockAnterior = $variante->quantity;
                        $variante->decrement('quantity', $item['cantidad']);
                        
                        \App\Models\InventoryMovement::create([
                            'producto_id' => $item['producto_id'],
                            'type' => 'salida',
                            'quantity' => $item['cantidad'],
                            'quantity_before' => $stockAnterior,
                            'quantity_after' => $variante->quantity,
                            'reference' => 'Venta directa variante #' . $venta->codigo,
                            'notes' => 'Venta ID: ' . $venta->id . ' - Variante: ' . $item['variante_id'],
                            'user_id' => auth()->id(),
                            'empresa_id' => auth()->user()->empresa_id,
                            'sucursal_id' => auth()->user()->sucursal_id
                        ]);
                    }
                } else {
                    $producto = Producto::find($item['producto_id']);
                    if ($producto) {
                        $stockAnterior = $producto->quantity;
                        $producto->decrement('quantity', $item['cantidad']);
                        
                        \App\Models\InventoryMovement::create([
                            'producto_id' => $item['producto_id'],
                            'type' => 'salida',
                            'quantity' => $item['cantidad'],
                            'quantity_before' => $stockAnterior,
                            'quantity_after' => $producto->quantity,
                            'reference' => 'Venta directa #' . $venta->codigo,
                            'notes' => 'Venta ID: ' . $venta->id,
                            'user_id' => auth()->id(),
                            'empresa_id' => auth()->user()->empresa_id,
                            'sucursal_id' => auth()->user()->sucursal_id
                        ]);
                    }
                }
            }

            // Crear pago
            $venta->pagos()->create([
                'metodo_pago' => $this->metodo_pago,
                'monto' => $this->getTotal(),
                'monto_recibido' => $this->getTotal(),
                'cambio' => 0,
                'caja_id' => $caja->id,
                'confirmado' => true,
            ]);

            // Actualizar serie con el número usado
            if ($nextNumber > $serie->correlativo_actual) {
                $serie->update(['correlativo_actual' => $nextNumber]);
            }

            \DB::commit();

            session()->flash('message', 'Venta creada exitosamente');
            return redirect()->route('admin.ventas.show', $venta->id);

        } catch (\Exception $e) {
            \DB::rollback();
            session()->flash('error', 'Error al crear la venta: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $this->searchProducts();
        
        return view('livewire.admin.ventas.create')->layout($this->getLayout());
    }
}