<?php

namespace App\Livewire\Admin\Users\Profile;

use App\Traits\HasDynamicLayout;
use Livewire\Component;
use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;

class SecuritySettings extends Component
{
    use HasDynamicLayout;

    public $user;
    public $current_password;
    public $new_password;
    public $new_password_confirmation;
    public $logout_other_devices = false;

    public function mount(User $user)
    {
        $this->user = $user;
    }

    public function updatePassword()
    {
        $this->validate([
            'current_password' => 'required',
            'new_password' => 'required|min:8|confirmed',
        ], [
            'current_password.required' => 'La contraseña actual es requerida',
            'new_password.required' => 'La nueva contraseña es requerida',
            'new_password.min' => 'La nueva contraseña debe tener al menos 8 caracteres',
            'new_password.confirmed' => 'La confirmación de contraseña no coincide',
        ]);

        // Verificar contraseña actual
        if (!Hash::check($this->current_password, $this->user->password)) {
            $this->addError('current_password', 'La contraseña actual es incorrecta');
            return;
        }

        // Actualizar contraseña
        $this->user->update([
            'password' => Hash::make($this->new_password)
        ]);

        // Cerrar otras sesiones si se solicita
        if ($this->logout_other_devices) {
            Auth::logoutOtherDevices($this->new_password);
        }

        // Limpiar campos
        $this->reset(['current_password', 'new_password', 'new_password_confirmation', 'logout_other_devices']);

        $this->dispatch('alert', [
            'title' => '¡Actualizado!',
            'text' => 'Tu contraseña ha sido actualizada correctamente',
            'icon' => 'success'
        ]);
    }

    public function getSecurityScore()
    {
        $score = 0;
        $factors = [];

        // Verificar si tiene 2FA activado
        if ($this->user->two_factor_enabled) {
            $score += 30;
            $factors[] = 'Autenticación de dos factores activada';
        }

        // Verificar si tiene avatar (indica cuenta activa)
        if ($this->user->avatar) {
            $score += 10;
            $factors[] = 'Foto de perfil configurada';
        }

        // Verificar si el email está verificado
        if ($this->user->email_verified_at) {
            $score += 20;
            $factors[] = 'Email verificado';
        }

        // Verificar información de perfil completa
        if ($this->user->phone && $this->user->address) {
            $score += 15;
            $factors[] = 'Información de contacto completa';
        }

        // Verificar actividad reciente (últimos 30 días)
        if ($this->user->updated_at->diffInDays(now()) <= 30) {
            $score += 15;
            $factors[] = 'Cuenta activa recientemente';
        }

        // Verificar si tiene roles asignados
        if ($this->user->roles->count() > 0) {
            $score += 10;
            $factors[] = 'Roles de seguridad asignados';
        }

        return [
            'score' => min($score, 100),
            'factors' => $factors,
            'level' => $this->getSecurityLevel($score)
        ];
    }

    private function getSecurityLevel($score)
    {
        if ($score >= 80) return ['name' => 'Excelente', 'color' => 'success'];
        if ($score >= 60) return ['name' => 'Bueno', 'color' => 'primary'];
        if ($score >= 40) return ['name' => 'Regular', 'color' => 'warning'];
        return ['name' => 'Bajo', 'color' => 'danger'];
    }

    public function render()
    {
        return view('livewire.admin.users.profile.security-settings', [
            'securityInfo' => $this->getSecurityScore()
        ]);
    }
}