<?php

namespace App\Livewire\Admin\Users\Profile;

use App\Traits\HasDynamicLayout;
use Livewire\Component;
use App\Models\User;
use Illuminate\Support\Facades\Auth;

class QuickActions extends Component
{
    use HasDynamicLayout;

    public $user;

    public function mount(User $user)
    {
        $this->user = $user;
    }

    public function getQuickActions()
    {
        return [
            [
                'title' => 'Mis Pedidos',
                'description' => 'Ver todos mis pedidos',
                'icon' => 'ri-shopping-cart-line',
                'color' => 'primary',
                'route' => route('admin.pedidos.index', ['userId' => $this->user->id]),
                'count' => $this->getPedidosCount()
            ],
            [
                'title' => 'Cambiar Contraseña',
                'description' => 'Actualizar mi contraseña',
                'icon' => 'ri-lock-line',
                'color' => 'warning',
                'route' => route('admin.users.password', $this->user->id),
                'count' => null
            ],
            [
                'title' => 'Historial de Actividad',
                'description' => 'Ver mi historial completo',
                'icon' => 'ri-history-line',
                'color' => 'info',
                'route' => route('admin.users.history', $this->user->id),
                'count' => null
            ],
            [
                'title' => 'Configuración 2FA',
                'description' => $this->user->two_factor_enabled ? 'Gestionar 2FA' : 'Activar 2FA',
                'icon' => 'ri-shield-line',
                'color' => $this->user->two_factor_enabled ? 'success' : 'secondary',
                'route' => '#',
                'count' => null,
                'action' => 'toggle2FA'
            ]
        ];
    }

    public function getSystemInfo()
    {
        return [
            'php_version' => PHP_VERSION,
            'laravel_version' => app()->version(),
            'timezone' => config('app.timezone'),
            'locale' => config('app.locale'),
            'last_login' => $this->user->updated_at->format('d/m/Y H:i'),
            'account_created' => $this->user->created_at->format('d/m/Y'),
            'days_active' => $this->user->created_at->diffInDays(now())
        ];
    }

    private function getPedidosCount()
    {
        $empleado = \App\Models\Empleado::where('user_id', $this->user->id)->first();
        
        if ($empleado) {
            return \App\Models\Pedido::whereHas('empleados', function($q) use ($empleado) {
                $q->where('empleadoId', $empleado->id);
            })->count();
        }
        
        // Si no es empleado, contar pedidos de la empresa
        if ($this->user->empresa_id) {
            return \App\Models\Pedido::where('empresaId', $this->user->empresa_id)->count();
        }
        
        return 0;
    }

    public function toggle2FA()
    {
        $this->dispatch('alert', [
            'title' => 'Configuración 2FA',
            'text' => 'La configuración de autenticación de dos factores estará disponible próximamente',
            'icon' => 'info'
        ]);
    }

    public function exportUserData()
    {
        $this->dispatch('alert', [
            'title' => 'Exportación iniciada',
            'text' => 'Te enviaremos un email con tus datos cuando esté listo',
            'icon' => 'info'
        ]);
    }

    public function render()
    {
        return view('livewire.admin.users.profile.quick-actions', [
            'quickActions' => $this->getQuickActions(),
            'systemInfo' => $this->getSystemInfo()
        ]);
    }
}
