<?php

namespace App\Livewire\Admin\Users;

use App\Traits\HasDynamicLayout;
use Livewire\Component;
use App\Models\User;
use App\Models\Empresa;
use App\Models\Sucursal;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use App\Mail\UserWelcomeMail;
use Illuminate\Validation\Rules;
use Illuminate\Support\Str;

class Create extends Component
{
    use HasDynamicLayout;


    public $name;
    public $username;
    public $email;
    public $telefono;
    public $password;
    public $password_confirmation;
    public $empresa_id;
    public $sucursal_id;
    public $status = true;
    public $role;
    public $sucursales = [];

    protected function rules()
    {
        return [
            'name' => ['required', 'string', 'max:255'],
            'username' => ['required', 'string', 'max:255', 'unique:users', 'alpha_dash'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'telefono' => ['nullable', 'string', 'max:20'],
            'password' => ['required', 'confirmed', Rules\Password::defaults()],
            'empresa_id' => ['required', 'exists:empresas,id'],
            'sucursal_id' => ['required', 'exists:sucursales,id'],
            'status' => ['boolean'],
            'role' => ['required', 'exists:roles,name']
        ];
    }

    public function updatedName($value)
    {
        if (empty($this->username)) {
            $this->username = $this->generarUsername($value);
        }
    }

    public function updatedEmpresaId($value)
    {
        $this->loadSucursales();
    }

    public function loadSucursales()
    {
        if ($this->empresa_id) {
            $this->sucursales = Sucursal::forUser()
                ->where('empresa_id', $this->empresa_id)
                ->where('status', true)
                ->get();
        } else {
            $this->sucursales = [];
        }
        $this->sucursal_id = null;
    }

    public function save()
    {
        $this->validate();

        $plainPassword = $this->password;

        //$username = $this->generarUsername($this->name);

        $user = new User();
        $user->name = $this->name;
        $user->username = $this->username;
        $user->email = $this->email;
        $user->telefono = $this->telefono;
        $user->password = Hash::make($plainPassword);
        $user->empresa_id = $this->empresa_id;
        $user->sucursal_id = $this->sucursal_id;
        $user->status = $this->status;
        $user->save();

        $user->assignRole($this->role);

        // Enviar correo de bienvenida
        try {
            Mail::to($user->email)->send(new UserWelcomeMail($user, $plainPassword));
        } catch (\Exception $e) {
            \Log::error('Error enviando correo de bienvenida: ' . $e->getMessage());
        }

        session()->flash('message', 'Usuario creado correctamente. Se ha enviado un correo con las credenciales.');

        return redirect()->route('admin.users.index');
    }

    public function render()
    {
         \Gate::authorize('create users');

        $empresas = Empresa::forUser()->get();
        $sucursales = Sucursal::forUser()->where('status', 'active')
            ->when($this->empresa_id, function ($query) {
                $query->where('empresa_id', $this->empresa_id);
            })
            ->get();

        $roles = Role::all();

        // Calcular estadísticas
        $totalUsers = User::forUser()->count();
        $activeUsers = User::forUser()->where('status', 1)->count();
        $pendingUsers = 0;
        $inactiveUsers = User::forUser()->where('status', 0)->count();

        return $this->renderWithLayout('livewire.admin.users.create', compact('empresas', 'sucursales', 'roles', 'totalUsers', 'activeUsers', 'pendingUsers', 'inactiveUsers'), [
            'title' => 'Lista de Usuarios',
            'breadcrumb' => [
                'admin.dashboard' => 'Dashboard',
                'admin.users.index' => 'Usuarios'
            ]
        ]);
    }

        public function generarUsername($nombreCompleto)
       {
            // 1. Limpiar espacios extras y dividir por espacios
            // Ejemplo: " Theizer Gabriel  Gonzalez Lugo " -> ["Theizer", "Gabriel", "Gonzalez", "Lugo"]
            $partes = preg_split('/\s+/', trim($nombreCompleto));

            if (count($partes) < 2) {
                return Str::lower($partes[0]);
            }

            $primerNombre = $partes[0];
            $primerApellido = $partes[1];

            // Si el segundo elemento es un conector (de, del, la), el apellido real es el tercero
            $conectores = ['de', 'del', 'la', 'las', 'los'];
            if (in_array(strtolower($partes[1]), $conectores) && isset($partes[2])) {
                $primerApellido = $partes[2];
            }

            // 2. Obtener primera letra del nombre y el primer apellido
            $inicial = Str::substr($primerNombre, 0, 1);

            // 3. Formatear: tgonzalez (en minúsculas y sin acentos)
            return Str::slug($inicial . $primerApellido, '');

     }
}
