<?php

namespace App\Livewire\Admin\Promociones;

use App\Models\Promotion;
use Livewire\Component;
use Livewire\WithPagination;
use App\Traits\HasDynamicLayout;

class Index extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $status = '';
    public $discountType = '';
    public $perPage = 10;
    public $sortField = 'created_at';
    public $sortDirection = 'desc';

    protected $queryString = [
        'search' => ['except' => ''],
        'status' => ['except' => ''],
        'discountType' => ['except' => ''],
        'perPage',
        'sortField',
        'sortDirection'
    ];

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedStatus()
    {
        $this->resetPage();
    }

    public function updatedDiscountType()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function getStats()
    {
        $empresaId = auth()->user()->empresa_id;
        
        return [
            'total' => Promotion::where('empresa_id', $empresaId)->count(),
            'activas' => Promotion::where('empresa_id', $empresaId)->where('is_active', true)->count(),
            'automaticas' => Promotion::where('empresa_id', $empresaId)->where('is_automatic', true)->count(),
            'con_cupones' => Promotion::where('empresa_id', $empresaId)->whereHas('coupons')->count(),
        ];
    }

    public function toggleEstado($id)
    {
        $promotion = Promotion::where('empresa_id', auth()->user()->empresa_id)->findOrFail($id);
        $promotion->is_active = !$promotion->is_active;
        $promotion->save();

        $this->dispatch('notify', ['message' => 'Estado de la promoción actualizado.', 'type' => 'success']);
    }

    public function delete($id)
    {
        $promotion = Promotion::where('empresa_id', auth()->user()->empresa_id)->findOrFail($id);
        
        if ($promotion->pedidoDiscounts()->count() > 0) {
            $this->dispatch('notify', ['message' => 'No se puede eliminar porque tiene descuentos aplicados.', 'type' => 'error']);
            return;
        }

        $promotion->delete();
        $this->dispatch('notify', ['message' => 'Promoción eliminada correctamente.', 'type' => 'success']);
    }

    public function render()
    {
        $empresaId = auth()->user()->empresa_id;

        $promotions = Promotion::where('empresa_id', $empresaId)
            ->withCount('coupons')
            ->when($this->search, function ($query) {
                $query->where('name', 'like', '%' . $this->search . '%');
            })
            ->when($this->status !== '', function ($query) {
                $query->where('is_active', $this->status == '1');
            })
            ->when($this->discountType, function ($query) {
                $query->where('discount_type', $this->discountType);
            })
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        return view('livewire.admin.promociones.index', [
            'promotions' => $promotions,
            'stats' => $this->getStats()
        ])->layout($this->getLayout(), [
            'title' => 'Promociones',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Promociones', 'active' => true]
            ]
        ]);
    }
}
