<?php

namespace App\Livewire\Admin\Productos;

use App\Models\Producto;
use App\Models\InventoryMovement;
use App\Services\InventoryService;
use Livewire\Component;
use Livewire\WithPagination;
use App\Traits\HasDynamicLayout;
use Carbon\Carbon;

class InventoryHistory extends Component
{
    use WithPagination, HasDynamicLayout;

    public Producto $producto;
    public $showAdjustModal = false;
    public $showBulkModal = false;
    public $adjustQuantity = 0;
    public $adjustReason = '';
    public $bulkType = 'ingreso';
    public $bulkQuantity = 0;
    public $bulkCost = 0;
    public $bulkReference = '';
    public $bulkNotes = '';
    
    // Filtros
    public $filterType = '';
    public $filterDateFrom = '';
    public $filterDateTo = '';
    public $filterUser = '';
    public $perPage = 15;
    public $sortField = 'created_at';
    public $sortDirection = 'desc';
    
    // Estadísticas
    public $stats = [];
    public $chartData = [];

    protected $rules = [
        'adjustQuantity' => 'required|integer|min:0',
        'adjustReason' => 'required|string|max:255',
        'bulkQuantity' => 'required|integer|min:1',
        'bulkCost' => 'nullable|numeric|min:0',
        'bulkReference' => 'nullable|string|max:100',
        'bulkNotes' => 'nullable|string|max:500'
    ];

    public function mount(Producto $producto)
    {
        $this->producto = $producto;
        $this->adjustQuantity = $producto->quantity;
        $this->filterDateFrom = Carbon::now()->subDays(30)->format('Y-m-d');
        $this->filterDateTo = Carbon::now()->format('Y-m-d');
        $this->loadStats();
        $this->loadChartData();
    }

    public function openAdjustModal()
    {
        $this->adjustQuantity = $this->producto->quantity;
        $this->adjustReason = '';
        $this->showAdjustModal = true;
    }

    public function closeAdjustModal()
    {
        $this->showAdjustModal = false;
        $this->reset(['adjustQuantity', 'adjustReason']);
    }

    public function adjustInventory()
    {
        $this->validate();

        try {
            $inventoryService = new InventoryService();
            $inventoryService->adjustInventory(
                $this->producto,
                $this->adjustQuantity,
                $this->adjustReason
            );

            $this->producto->refresh();
            $this->closeAdjustModal();

            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => 'Inventario ajustado correctamente'
            ]);

        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al ajustar inventario: ' . $e->getMessage()
            ]);
        }
    }

    public function loadStats()
    {
        $movements = $this->producto->inventoryMovements()
            ->when($this->filterDateFrom, fn($q) => $q->whereDate('created_at', '>=', $this->filterDateFrom))
            ->when($this->filterDateTo, fn($q) => $q->whereDate('created_at', '<=', $this->filterDateTo));

        $this->stats = [
            'total_movements' => $movements->count(),
            'total_ingresos' => $movements->clone()->whereIn('type', ['ingreso', 'compra', 'devolucion'])->sum('quantity'),
            'total_salidas' => abs($movements->clone()->whereIn('type', ['salida', 'venta'])->sum('quantity')),
            'total_ajustes' => $movements->clone()->where('type', 'ajuste')->count(),
            'valor_inventario' => $this->producto->quantity * $this->producto->price,
            'rotacion' => $this->calculateRotation(),
            'stock_promedio' => $this->calculateAverageStock(),
            'ultimo_movimiento' => $this->producto->inventoryMovements()->latest()->first()?->created_at
        ];
    }

    public function loadChartData()
    {
        $movements = $this->producto->inventoryMovements()
            ->selectRaw('DATE(created_at) as date, SUM(CASE WHEN quantity > 0 THEN quantity ELSE 0 END) as ingresos, SUM(CASE WHEN quantity < 0 THEN ABS(quantity) ELSE 0 END) as salidas')
            ->when($this->filterDateFrom, fn($q) => $q->whereDate('created_at', '>=', $this->filterDateFrom))
            ->when($this->filterDateTo, fn($q) => $q->whereDate('created_at', '<=', $this->filterDateTo))
            ->groupBy('date')
            ->orderBy('date')
            ->get();

        $this->chartData = $movements->map(function($item) {
            return [
                'date' => Carbon::parse($item->date)->format('d/m'),
                'ingresos' => $item->ingresos,
                'salidas' => $item->salidas
            ];
        })->toArray();
    }

    private function calculateRotation()
    {
        $totalSalidas = abs($this->producto->inventoryMovements()
            ->whereIn('type', ['salida', 'venta'])
            ->whereDate('created_at', '>=', Carbon::now()->subDays(30))
            ->sum('quantity'));
        
        return $this->producto->quantity > 0 ? round($totalSalidas / $this->producto->quantity, 2) : 0;
    }

    private function calculateAverageStock()
    {
        return $this->producto->inventoryMovements()
            ->whereDate('created_at', '>=', Carbon::now()->subDays(30))
            ->avg('quantity_after') ?? $this->producto->quantity;
    }

    public function openBulkModal()
    {
        $this->reset(['bulkQuantity', 'bulkCost', 'bulkReference', 'bulkNotes']);
        $this->showBulkModal = true;
    }

    public function closeBulkModal()
    {
        $this->showBulkModal = false;
    }

    public function processBulkMovement()
    {
        $this->validate([
            'bulkQuantity' => 'required|integer|min:1',
            'bulkCost' => 'nullable|numeric|min:0',
            'bulkReference' => 'nullable|string|max:100',
            'bulkNotes' => 'nullable|string|max:500'
        ]);

        try {
            $inventoryService = new InventoryService();
            $inventoryService->recordMovement(
                $this->producto,
                $this->bulkType,
                $this->bulkQuantity,
                $this->bulkCost ?: null,
                $this->bulkReference ?: null,
                $this->bulkNotes ?: null
            );

            $this->producto->refresh();
            $this->loadStats();
            $this->loadChartData();
            $this->closeBulkModal();

            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => 'Movimiento registrado correctamente'
            ]);

        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al registrar movimiento: ' . $e->getMessage()
            ]);
        }
    }

    public function updatedFilterDateFrom()
    {
        $this->loadStats();
        $this->loadChartData();
        $this->resetPage();
    }

    public function updatedFilterDateTo()
    {
        $this->loadStats();
        $this->loadChartData();
        $this->resetPage();
    }

    public function updatedFilterType()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortField === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortDirection = 'asc';
        }
        $this->sortField = $field;
    }

    public function render()
    {
        $movements = $this->producto->inventoryMovements()
            ->with('user')
            ->when($this->filterType, fn($q) => $q->where('type', $this->filterType))
            ->when($this->filterDateFrom, fn($q) => $q->whereDate('created_at', '>=', $this->filterDateFrom))
            ->when($this->filterDateTo, fn($q) => $q->whereDate('created_at', '<=', $this->filterDateTo))
            ->when($this->filterUser, fn($q) => $q->where('user_id', $this->filterUser))
            ->orderBy($this->sortField, $this->sortDirection)
            ->paginate($this->perPage);

        $users = \App\Models\User::whereHas('inventoryMovements', function($q) {
            $q->where('producto_id', $this->producto->id);
        })->get();

        return $this->renderWithLayout('livewire.admin.productos.inventory-history', [
            'movements' => $movements,
            'users' => $users
        ], [
            'title' => 'Historial de Inventario - ' . $this->producto->name,
            'description' => 'Historial completo de movimientos de inventario del producto',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Productos', 'route' => 'admin.productos.index'],
                ['name' => $this->producto->name, 'active' => true]
            ]
        ]);
    }
}