<?php

namespace App\Livewire\Admin\Productos;

use App\Models\Producto;
use App\Models\Categoria;
use App\Models\Marca;
use Livewire\Component;
use Livewire\WithFileUploads;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;

class EditMejorado extends Component
{
    use WithFileUploads, HasDynamicLayout;

    // Propiedades del producto
    public $producto;
    public $code;
    public $name;
    public $price;
    public $precio_bs;
    public $description;
    public $quantity;
    public $quantity_alert;
    public $status;
    public $categoria_id;
    public $marca_id;

    // Propiedades adicionales
    public $sku;
    public $barcode;
    public $weight;
    public $dimensions;
    public $tags;
    public $meta_title;
    public $meta_description;
    public $featured;

    // Propiedades de imágenes
    public $images = [];
    public $existingImages = [];
    public $removedImages = [];

    // Propiedades de imágenes de variantes
    public $variantImages = []; // Array de imágenes para cada variante

    // Propiedades de variaciones
    public $has_variants;
    public $variants = [];
    public $variantOptions = [];

    // Propiedades de país
    public $empresaPais;

    // Propiedades de inventario
    public $track_inventory;
    public $allow_backorder;
    public $inventory_policy;

    // UI
    public $activeTab = 'basic';

    public function changeTab($tab)
    {
        \Log::info('Cambiando pestaña a: ' . $tab);
        $this->activeTab = $tab;

        // Debug: Verificar qué pestaña está activa y qué datos hay
        \Log::info('Pestaña activa después de cambio:', [
            'activeTab' => $this->activeTab,
            'sku' => $this->sku,
            'barcode' => $this->barcode,
            'tags' => $this->tags,
            'has_variants' => $this->has_variants,
            'track_inventory' => $this->track_inventory,
        ]);

        // Emitir evento para que JavaScript pueda reaccionar
        $this->dispatch('tab-changed', $tab);
    }

    public function removeExistingImage($imageId)
    {
        $this->removedImages[] = $imageId;
        $this->existingImages = array_filter($this->existingImages, function($img) use ($imageId) {
            return $img['id'] != $imageId;
        });
    }
    public $uploading = false;
    public $uploadProgress = 0;

    protected function rules()
    {
        return [
            'code' => 'required|string|max:50',
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'precio_bs' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'quantity' => 'required|integer|min:0',
            'quantity_alert' => 'required|integer|min:0',
            'status' => 'boolean',
            'categoria_id' => 'required|exists:categorias,id',
            'marca_id' => 'required|exists:marcas,id',
            'sku' => 'nullable|string|max:100',
            'barcode' => 'nullable|string|max:100',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|string|max:255',
            'tags' => 'nullable|string|max:500',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'featured' => 'boolean',
            'images.*' => 'image|max:2048',
            'variantImages.*' => 'nullable|image|max:2048', // Agregamos validación para imágenes de variantes
            'has_variants' => 'boolean',
            'track_inventory' => 'boolean',
            'allow_backorder' => 'boolean',
            'inventory_policy' => 'in:deny,allow',
            // Validaciones para variantes
            'variants.*.name' => 'nullable|string|max:255',
            'variants.*.values' => 'nullable|array|min:1',
            'variants.*.values.*' => 'nullable|string|max:255',
            'variants.*.price_adjustment' => 'nullable|numeric|min:0',
            'variants.*.precio_bs' => 'nullable|numeric|min:0',
            'variants.*.quantity' => 'nullable|integer|min:0',
            'variants.*.sku' => 'nullable|string|max:100',
            'variants.*.alt_text' => 'nullable|string|max:255'
        ];
    }

    protected $messages = [
        'code.required' => 'El código es requerido.',
        'code.unique' => 'Este código ya existe.',
        'name.required' => 'El nombre es requerido.',
        'price.required' => 'El precio es requerido.',
        'price.numeric' => 'El precio debe ser un número.',
        'price.min' => 'El precio debe ser mayor o igual a 0.',
        'quantity.required' => 'La cantidad es requerida.',
        'quantity.integer' => 'La cantidad debe ser un número entero.',
        'quantity.min' => 'La cantidad debe ser mayor o igual a 0.',
        'quantity_alert.required' => 'La alerta de cantidad es requerida.',
        'categoria_id.required' => 'La categoría es requerida.',
        'marca_id.required' => 'La marca es requerida.',
        'images.*.image' => 'El archivo debe ser una imagen.',
        'images.*.max' => 'La imagen no puede pesar más de 2MB.',
        // Mensajes para validaciones de variantes
        'variants.*.name.required' => 'El nombre de la variante es requerido.',
        'variants.*.name.string' => 'El nombre de la variante debe ser texto.',
        'variants.*.name.max' => 'El nombre de la variante no puede exceder 255 caracteres.',
        'variants.*.values.required' => 'Los valores de la variante son requeridos.',
        'variants.*.values.array' => 'Los valores deben ser un array.',
        'variants.*.values.min' => 'Debe haber al menos un valor.',
        'variants.*.values.*.required' => 'Cada valor es requerido.',
        'variants.*.values.*.string' => 'Cada valor debe ser texto.',
        'variants.*.values.*.max' => 'Cada valor no puede exceder 255 caracteres.',
        'variants.*.price_adjustment.required' => 'El ajuste de precio es requerido.',
        'variants.*.price_adjustment.numeric' => 'El ajuste de precio debe ser un número.',
        'variants.*.price_adjustment.min' => 'El ajuste de precio debe ser mayor o igual a 0.',
        'variants.*.quantity.required' => 'La cantidad es requerida.',
        'variants.*.quantity.integer' => 'La cantidad debe ser un número entero.',
        'variants.*.quantity.min' => 'La cantidad debe ser mayor o igual a 0.',
        'variants.*.sku.required' => 'El SKU de la variante es requerido.',
        'variants.*.sku.string' => 'El SKU debe ser texto.',
        'variants.*.sku.max' => 'El SKU no puede exceder 100 caracteres.',
        'variants.*.alt_text.string' => 'El texto alternativo debe ser texto.',
        'variants.*.alt_text.max' => 'El texto alternativo no puede exceder 255 caracteres.'
    ];

    public function mount(Producto $producto)
    {
        $this->producto = $producto->load('seo', 'inventorySetting', 'variants', 'images');
        $this->loadProductData();
        $this->initializeVariants();
        $this->loadEmpresaPais();

        // Inicializar array de imágenes de variantes
        $this->variantImages = [];

        // Sincronizar SKUs si hay variantes al cargar
        if ($this->has_variants && !empty($this->variants)) {
            $this->syncVariantSKUs();
        }

        // Debug: Verificar que los datos se cargaron correctamente
        \Log::info('Producto cargado:', [
            'id' => $this->producto->id,
            'name' => $this->name,
            'code' => $this->code,
            'sku' => $this->sku,
            'barcode' => $this->barcode,
            'tags' => $this->tags,
            'meta_title' => $this->meta_title,
            'meta_description' => $this->meta_description,
            'has_variants' => $this->has_variants,
            'track_inventory' => $this->track_inventory,
            'activeTab' => $this->activeTab
        ]);
    }

    public function loadProductData()
    {
        // Cargar datos básicos del producto
        $this->code = $this->producto->code;
        $this->name = $this->producto->name;
        $this->price = $this->producto->price;
        $this->precio_bs = $this->producto->precio_bs;
        $this->description = $this->producto->description;
        $this->quantity = $this->producto->quantity;
        $this->quantity_alert = $this->producto->quantity_alert;
        $this->status = $this->producto->status ?? true;
        $this->categoria_id = $this->producto->categoria_id;
        $this->marca_id = $this->producto->marca_id;
        $this->weight = $this->producto->weight;
        $this->dimensions = $this->producto->dimensions;

        // Cargar datos de SEO (sku, barcode, tags, meta_title, meta_description, featured)
        $seo = $this->producto->seo;
        if ($seo) {
            $this->sku = $seo->sku ?? '';
            $this->barcode = $seo->barcode ?? '';
            $this->tags = $seo->tags ?? '';
            $this->meta_title = $seo->meta_title ?? '';
            $this->meta_description = $seo->meta_description ?? '';
            $this->featured = $seo->featured ?? false;
        } else {
            $this->sku = '';
            $this->barcode = '';
            $this->tags = '';
            $this->meta_title = '';
            $this->meta_description = '';
            $this->featured = false;
        }

        // Cargar configuración de inventario
        if ($this->producto->inventorySetting) {
            $this->track_inventory = $this->producto->inventorySetting->track_inventory;
            $this->allow_backorder = $this->producto->inventorySetting->allow_backorder;
            $this->inventory_policy = $this->producto->inventorySetting->inventory_policy;
        } else {
            $this->track_inventory = true;
            $this->allow_backorder = false;
            $this->inventory_policy = 'deny';
        }

        // Cargar variantes
        if ($this->producto->variants && $this->producto->variants->count() > 0) {
            $this->has_variants = true;
            $this->variants = $this->producto->variants->map(function ($variant) {
                // Decodificar valores y convertir al formato esperado por el formulario
                $decodedValues = is_string($variant->values) ? json_decode($variant->values, true) : $variant->values;
                $formValues = [];
                
                if (is_array($decodedValues)) {
                    // Si es el formato nuevo {"Volumen": "900ml"}, extraer los valores
                    if (isset($decodedValues[ucfirst($variant->name)])) {
                        $formValues = explode(', ', $decodedValues[ucfirst($variant->name)]);
                    } else {
                        // Si es el formato antiguo ["900 ML"], usar directamente
                        $formValues = array_values($decodedValues);
                    }
                }
                
                return [
                    'id' => $variant->id,
                    'name' => $variant->name,
                    'values' => $formValues,
                    'price_adjustment' => $variant->price_adjustment ?? 0,
                    'precio_bs' => $variant->precio_bs ?? 0,
                    'quantity' => $variant->quantity ?? 0,
                    'sku' => $variant->sku ?? $this->sku,
                    'image_path' => $variant->image_path,
                    'image_thumbnail_path' => $variant->image_thumbnail_path,
                    'alt_text' => $variant->alt_text ?? ''
                ];
            })->toArray();
        } else {
            $this->has_variants = false;
            $this->variants = [];
        }

        // Inicializar array de imágenes de variantes con el mismo tamaño que las variantes
        $this->variantImages = array_fill(0, count($this->variants), null);

        // Cargar imágenes existentes
        $this->existingImages = $this->producto->images->map(function ($image) {
            return [
                'id' => $image->id,
                'path' => $image->path,
                'thumbnail_path' => $image->thumbnail_path,
                'alt_text' => $image->alt_text,
                'is_primary' => $image->is_primary,
                'order' => $image->order
            ];
        })->toArray();
    }

    public function loadEmpresaPais()
    {
        $this->empresaPais = auth()->user()->empresa->pais;
    }

    public function getEsVenezuelaProperty()
    {
        return $this->empresaPais && $this->empresaPais->codigo_iso2 === 'VE';
    }

    public function generateSKU()
    {
        $this->sku = strtoupper(Str::slug($this->name)) . '-' . strtoupper(Str::random(4));

        // Actualizar SKU de las variantes que usan el SKU del producto
        $this->syncVariantSKUs();
    }

    public function updatedSku($value)
    {
        // Sincronizar SKU de variantes cuando se actualiza el SKU del producto
        $this->syncVariantSKUs();
    }

    public function updatingSku($value)
    {
        // También sincronizar mientras se está actualizando el valor
        $this->syncVariantSKUs();
    }

    public function updatedHas_variants($value)
    {
        // Sincronizar SKUs cuando se activa/desactiva variantes
        if ($value && !empty($this->variants)) {
            $this->syncVariantSKUs();
        }
    }

    public function syncVariantSKUs()
    {
        if (!empty($this->variants)) {
            foreach ($this->variants as $index => $variant) {
                // Si el SKU de la variante es igual al SKU del producto o está vacío, actualizarlo
                if (!isset($variant['sku']) || empty($variant['sku']) || $variant['sku'] === $this->sku) {
                    $this->variants[$index]['sku'] = $this->sku;
                    $this->dispatch('skus-updated');
                }
                $this->variants[$index]['sku'] = $this->sku;
                    $this->dispatch('skus-updated');
            }

            // Emitir evento para actualizar la vista
        } elseif ($this->has_variants) {
            // Si no hay variantes pero has_variants es true, crear una variante inicial
            $this->variants = [
                [
                    'name' => '',
                    'values' => [''],
                    'price_adjustment' => 0,
                    'quantity' => 0,
                    'sku' => $this->sku,
                    'image_path' => null,
                    'image_thumbnail_path' => null,
                    'alt_text' => ''
                ]
            ];
            $this->dispatch('skus-updated');
        }
    }

    public function generateCode()
    {
        $this->code = strtoupper(Str::random(8));
    }

    public function initializeVariants()
    {
        if (empty($this->variants)) {
            $this->variants = [
                [
                    'name' => '',
                    'values' => [''],
                    'price_adjustment' => 0,
                    'precio_bs' => 0,
                    'quantity' => 0,
                    'sku' => $this->sku, // Usar el SKU del producto por defecto
                    'image_path' => null,
                    'image_thumbnail_path' => null,
                    'alt_text' => ''
                ]
            ];
        }

        $this->variantOptions = [
            'size' => ['XS', 'S', 'M', 'L', 'XL', 'XXL', 'XXXL'],
            'color' => ['Rojo', 'Azul', 'Verde', 'Negro', 'Blanco', 'Amarillo', 'Naranja', 'Rosa', 'Morado', 'Gris', 'Marrón', 'Beige'],
            'material' => ['Algodón', 'Poliéster', 'Nylon', 'Cuero', 'Lana', 'Seda', 'Lino', 'Denim', 'Spandex', 'Viscosa'],
            'sabor' => ['Chocolate', 'Vainilla', 'Fresa', 'Limón', 'Naranja', 'Menta', 'Coco', 'Caramelo', 'Café', 'Natural'],
            'peso' => ['100g', '250g', '500g', '1kg', '2kg', '5kg', '10kg'],
            'volumen' => ['50ml', '100ml', '250ml', '500ml', '750ml', '1L', '2L', '5L'],
            'capacidad' => ['8GB', '16GB', '32GB', '64GB', '128GB', '256GB', '512GB', '1TB'],
            'potencia' => ['5W', '10W', '15W', '20W', '25W', '50W', '100W'],
            'voltaje' => ['12V', '24V', '110V', '220V'],
            'frecuencia' => ['50Hz', '60Hz'],
            'edad' => ['0-6 meses', '6-12 meses', '1-2 años', '2-4 años', '4-6 años', '6+ años'],
            'genero' => ['Masculino', 'Femenino', 'Unisex'],
            'temporada' => ['Primavera', 'Verano', 'Otoño', 'Invierno', 'Todo el año'],
            'tipo' => ['Básico', 'Premium', 'Deluxe', 'Estándar', 'Profesional'],
            'formato' => ['Líquido', 'Polvo', 'Gel', 'Crema', 'Spray', 'Tableta', 'Cápsula'],
            'concentracion' => ['5%', '10%', '15%', '20%', '25%', '30%', '50%'],
            'dureza' => ['Suave', 'Medio', 'Duro', 'Extra Duro'],
            'textura' => ['Liso', 'Rugoso', 'Mate', 'Brillante', 'Satinado'],
            'resistencia' => ['Baja', 'Media', 'Alta', 'Muy Alta'],
            'temperatura' => ['Frío', 'Ambiente', 'Tibio', 'Caliente'],
            'ph' => ['Ácido', 'Neutro', 'Alcalino'],
            'origen' => ['Nacional', 'Importado', 'Artesanal', 'Industrial'],
            'certificacion' => ['Orgánico', 'Fair Trade', 'Eco-friendly', 'Sin Gluten', 'Vegano'],
            'conectividad' => ['USB', 'Bluetooth', 'WiFi', 'Ethernet', 'Inalámbrico'],
            'compatibilidad' => ['Android', 'iOS', 'Windows', 'Mac', 'Universal'],
            'modelo' => ['2024', '2023', '2022', 'Clásico', 'Nuevo', 'Edición Limitada'],
            'marca_procesador' => ['Intel', 'AMD', 'Apple M1', 'Apple M2', 'Snapdragon', 'MediaTek'],
            'resolucion' => ['HD', 'Full HD', '2K', '4K', '8K', 'Retina'],
            'almacenamiento' => ['32GB', '64GB', '128GB', '256GB', '512GB', '1TB', '2TB'],
            'ram' => ['4GB', '8GB', '16GB', '32GB', '64GB'],
            'pantalla' => ['5"', '6"', '6.5"', '7"', '10"', '13"', '15"', '17"'],
            'bateria' => ['2000mAh', '3000mAh', '4000mAh', '5000mAh', '6000mAh'],
            'camara' => ['8MP', '12MP', '16MP', '24MP', '48MP', '64MP', '108MP'],
            'sistema_operativo' => ['Android 13', 'Android 14', 'iOS 16', 'iOS 17', 'Windows 11', 'macOS'],
            'procesador' => ['Dual Core', 'Quad Core', 'Hexa Core', 'Octa Core'],
            'velocidad' => ['1.5GHz', '2.0GHz', '2.5GHz', '3.0GHz', '3.5GHz'],
            'memoria_grafica' => ['2GB', '4GB', '6GB', '8GB', '12GB', '16GB'],
            'tipo_memoria' => ['DDR3', 'DDR4', 'DDR5', 'LPDDR4', 'LPDDR5'],
            'interfaz' => ['HDMI', 'USB-C', 'Lightning', 'Micro USB', 'Jack 3.5mm'],
            'red' => ['3G', '4G', '5G', 'WiFi 5', 'WiFi 6', 'WiFi 6E'],
            'sensor' => ['Huella', 'Facial', 'Iris', 'Proximidad', 'Giroscopio'],
            'carga' => ['5W', '10W', '15W', '20W', '25W', '30W', '45W', '65W', '100W'],
            'tipo_carga' => ['Inalámbrica', 'Rápida', 'Super Rápida', 'USB-C', 'Lightning'],
            'resistencia_agua' => ['IPX4', 'IPX5', 'IPX7', 'IP67', 'IP68'],
            'acabado' => ['Mate', 'Brillante', 'Metalizado', 'Texturizado', 'Transparente'],
            'forma' => ['Rectangular', 'Cuadrado', 'Circular', 'Ovalado', 'Irregular'],
            'estilo' => ['Moderno', 'Clásico', 'Vintage', 'Minimalista', 'Elegante'],
            'uso' => ['Casual', 'Formal', 'Deportivo', 'Trabajo', 'Fiesta'],
            'ocasion' => ['Diario', 'Especial', 'Navidad', 'Cumpleaños', 'Aniversario'],
            'publico' => ['Niños', 'Adolescentes', 'Adultos', 'Tercera Edad', 'Familiar'],
            'nivel' => ['Principiante', 'Intermedio', 'Avanzado', 'Profesional', 'Experto'],
            'dificultad' => ['Fácil', 'Medio', 'Difícil', 'Muy Difícil', 'Extremo'],
            'categoria_edad' => ['0+', '3+', '6+', '12+', '16+', '18+'],
            'idioma' => ['Español', 'Inglés', 'Francés', 'Portugués', 'Italiano'],
            'region' => ['América', 'Europa', 'Asia', 'África', 'Oceanía'],
            'clima' => ['Tropical', 'Templado', 'Frío', 'Seco', 'Húmedo'],
            'aplicacion' => ['Interior', 'Exterior', 'Baño', 'Cocina', 'Jardín'],
            'instalacion' => ['Fácil', 'Moderada', 'Compleja', 'Profesional'],
            'mantenimiento' => ['Bajo', 'Medio', 'Alto', 'Sin Mantenimiento'],
            'garantia' => ['6 meses', '1 año', '2 años', '3 años', '5 años', 'De por vida'],
            'eficiencia' => ['A+++', 'A++', 'A+', 'A', 'B', 'C'],
            'consumo' => ['Bajo', 'Medio', 'Alto', 'Muy Alto'],
            'ruido' => ['Silencioso', 'Bajo', 'Moderado', 'Alto'],
            'velocidad_maxima' => ['30 km/h', '50 km/h', '80 km/h', '120 km/h'],
            'autonomia' => ['50km', '100km', '200km', '300km', '500km'],
            'tiempo_carga' => ['30min', '1h', '2h', '4h', '8h'],
            'numero_puertos' => ['1', '2', '4', '6', '8', '12'],
            'longitud' => ['1m', '2m', '3m', '5m', '10m', '15m'],
            'ancho' => ['10cm', '20cm', '30cm', '50cm', '100cm'],
            'grosor' => ['1mm', '2mm', '5mm', '10mm', '20mm'],
            'densidad' => ['Baja', 'Media', 'Alta', 'Muy Alta'],
            'flexibilidad' => ['Rígido', 'Semi-rígido', 'Flexible', 'Muy Flexible'],
            'transparencia' => ['Opaco', 'Translúcido', 'Transparente', 'Cristalino'],
            'aroma' => ['Sin aroma', 'Suave', 'Intenso', 'Cítrico', 'Floral', 'Frutal'],
            'presentacion' => ['Individual', 'Pack 2', 'Pack 3', 'Pack 5', 'Pack 10'],
            'unidades' => ['1 unidad', '2 unidades', '5 unidades', '10 unidades', '25 unidades', '50 unidades'],
            'medida' => ['Pequeño', 'Mediano', 'Grande', 'Extra Grande', 'Jumbo']
        ];
    }

    public function addVariant()
    {
        $this->variants[] = [
            'name' => '',
            'values' => [''],
            'price_adjustment' => 0,
            'precio_bs' => 0,
            'quantity' => 0,
            'sku' => $this->sku, // Usar el SKU del producto por defecto
            'image_path' => null,
            'image_thumbnail_path' => null,
            'alt_text' => ''
        ];

        // Agregar un espacio para la imagen de la nueva variante
        $this->variantImages[] = null;

        // Sincronizar SKUs después de agregar una nueva variante
        $this->syncVariantSKUs();
    }

    public function removeVariant($index)
    {
        // Eliminar imagen de variante si existe
        if (isset($this->variantImages[$index])) {
            unset($this->variantImages[$index]);
            $this->variantImages = array_values($this->variantImages);
        }

        unset($this->variants[$index]);
        $this->variants = array_values($this->variants);
    }

    public function addVariantValue($variantIndex)
    {
        $this->variants[$variantIndex]['values'][] = '';
    }

    public function removeVariantValue($variantIndex, $valueIndex)
    {
        unset($this->variants[$variantIndex]['values'][$valueIndex]);
        $this->variants[$variantIndex]['values'] = array_values($this->variants[$variantIndex]['values']);
    }

    public function loadVariantOptions($variantIndex, $variantType)
    {
        if (!empty($variantType) && isset($this->variantOptions[$variantType])) {
            // Solo cargar si no hay valores o solo hay un valor vacío
            $currentValues = $this->variants[$variantIndex]['values'] ?? [''];
            $hasOnlyEmptyValue = count($currentValues) === 1 && empty($currentValues[0]);
            
            if (empty($currentValues) || $hasOnlyEmptyValue) {
                $this->variants[$variantIndex]['values'] = [];
            }
        }
    }

    public function updatedImages()
    {
        $this->validate([
            'images.*' => 'image|max:2048'
        ]);
    }

    public function removeImage($index)
    {
        unset($this->images[$index]);
        $this->images = array_values($this->images);
    }

    public function updatedVariantImages($value, $key)
    {
        // Validar la imagen
        $this->validate([
            'variantImages.' . $key => 'nullable|image|max:2048'
        ]);

        // Debug para verificar qué se está recibiendo
        \Log::info('updatedVariantImages llamado', [
            'key' => $key,
            'value_type' => gettype($value),
            'has_file' => isset($this->variantImages[$key]) && $this->variantImages[$key] instanceof \Livewire\TemporaryUploadedFile
        ]);

        // Si se subió una imagen válida, actualizar la vista
        if (isset($this->variantImages[$key]) && $this->variantImages[$key] instanceof \Livewire\TemporaryUploadedFile) {
            $this->dispatch('variant-image-updated', [
                'index' => $key,
                'filename' => $this->variantImages[$key]->getClientOriginalName()
            ]);
        }
    }

    public function removeVariantImage($variantIndex)
    {
        unset($this->variantImages[$variantIndex]);
        $this->variantImages = array_values($this->variantImages);

        // También limpiar los datos de imagen en la variante
        if (isset($this->variants[$variantIndex])) {
            $this->variants[$variantIndex]['image_path'] = null;
            $this->variants[$variantIndex]['image_thumbnail_path'] = null;
        }
    }

    public function update()
    {
        // Validar sin las imágenes de variantes para evitar errores con archivos temporales
        $rules = $this->rules();
        unset($rules['variantImages.*']);
        $this->validate($rules);

        try {
            // Iniciar transacción de base de datos
            \DB::transaction(function () {
                // Guardar cantidad anterior para el movimiento de inventario
                $previousQuantity = $this->producto->quantity;

                // Actualizar el producto
                $this->producto->update([
                    'code' => $this->code,
                    'name' => $this->name,
                    'slug' => Str::slug($this->name),
                    'price' => $this->price,
                    'precio_bs' => $this->precio_bs,
                    'description' => $this->description,
                    'quantity' => $this->quantity,
                    'quantity_alert' => $this->quantity_alert,
                    'status' => $this->status,
                    'categoria_id' => $this->categoria_id,
                    'marca_id' => $this->marca_id,
                    'weight' => $this->weight,
                    'dimensions' => $this->dimensions,
                    'has_variants' => $this->has_variants,
                    'track_inventory' => $this->track_inventory,
                    'allow_backorder' => $this->allow_backorder,
                    'inventory_policy' => $this->inventory_policy,
                    'empresa_id' => auth()->user()->empresa_id,
                    'sucursal_id' => auth()->user()->sucursal_id,
                ]);

                // Guardar imágenes principales
                if (!empty($this->images)) {
                    $this->processImages($this->producto);
                    $this->cleanupTemporaryFiles();
                }

                // Actualizar o crear SEO
                $this->producto->seo()->updateOrCreate(
                    ['producto_id' => $this->producto->id],
                    [
                        'sku' => $this->sku,
                        'barcode' => $this->barcode,
                        'tags' => $this->tags,
                        'meta_title' => $this->meta_title,
                        'meta_description' => $this->meta_description,
                        'featured' => $this->featured,
                        'empresa_id' => auth()->user()->empresa_id,
                    ]
                );

                // Actualizar configuración de inventario
                $this->producto->inventorySetting()->update([
                    'track_inventory' => $this->track_inventory,
                    'allow_backorder' => $this->allow_backorder,
                    'inventory_policy' => $this->inventory_policy,
                    'empresa_id' => auth()->user()->empresa_id,
                ]);

                // Procesar variaciones SOLO si están habilitadas
                if ($this->has_variants) {
                    \Log::info('Procesando variantes en edición porque has_variants es true', [
                        'has_variants' => $this->has_variants,
                        'variants_count' => count($this->variants)
                    ]);
                    $this->processVariants($this->producto);
                } else {
                    \Log::info('NO procesando variantes en edición porque has_variants es false', [
                        'has_variants' => $this->has_variants
                    ]);
                    // Si has_variants es false, eliminar todas las variantes existentes
                    $this->producto->variants()->delete();
                }

                // Registrar movimiento de inventario si la cantidad cambió
                if ($previousQuantity != $this->quantity) {
                    $difference = $this->quantity - $previousQuantity;
                    $type = $difference > 0 ? 'ingreso' : 'salida';
                    
                    $this->producto->inventoryMovements()->create([
                        'type' => $type,
                        'quantity' => abs($difference),
                        'quantity_before' => $previousQuantity,
                        'quantity_after' => $this->quantity,
                        'reference' => 'Actualización producto: ' . $this->code,
                        'notes' => $difference > 0 ? 'Incremento de stock' : 'Reducción de stock',
                        'user_id' => auth()->id(),
                        'empresa_id' => auth()->user()->empresa_id,
                        'sucursal_id' => auth()->user()->sucursal_id,
                    ]);
                }
            });

            // Mostrar notificación de éxito
            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => 'Producto actualizado exitosamente',
                'timeout' => 5000
            ]);

            // Redirigir después de un breve retraso para que se vea la notificación
            return redirect()->route('admin.productos.index')->with([
                'toast' => [
                    'type' => 'success',
                    'message' => 'Producto actualizado exitosamente'
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error de validación: ' . collect($e->errors())->flatten()->first(),
                'timeout' => 8000
            ]);
            throw $e;
        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al actualizar el producto: ' . $e->getMessage(),
                'timeout' => 8000
            ]);
            report($e);
        }
    }

    protected function processImages($producto)
    {
        if ($this->images) {
            // Eliminar imágenes existentes si hay nuevas imágenes
            $producto->images()->delete();
            
            foreach ($this->images as $index => $image) {
                // Procesar imagen principal
                $filename = time() . '_' . $index . '.' . $image->getClientOriginalExtension();
                $path = $image->storeAs('productos', $filename);

                // Crear miniatura
                $thumbnailPath = 'productos/thumb_' . $filename;
                // Usar la nueva API de Intervention Image v3
                $imageManager = new \Intervention\Image\ImageManager(
                    new \Intervention\Image\Drivers\Gd\Driver()
                );
                $thumbnail = $imageManager->read($image->getRealPath());
                $thumbnail->cover(300, 300);
                $thumbnail->save(public_path('app/' . $thumbnailPath));

                // Guardar en la base de datos - solo campos existentes
                $producto->images()->create([
                    'path' => $path,
                    'featured' => empty($this->existingImages) && $index === 0,
                    'empresa_id' => auth()->user()->empresa_id,
                ]);
            }

            // Limpiar las imágenes después de procesarlas
            $this->images = [];
        }
    }

    protected function processVariants($producto)
    {
        // Obtener IDs de variantes actuales
        $existingVariantIds = $producto->variants()->pluck('id')->toArray();
        $processedVariantIds = [];

        \Log::info('=== INICIO processVariants EDICIÓN ===', [
            'producto_id' => $producto->id,
            'has_variants' => $this->has_variants,
            'existing_variant_ids' => $existingVariantIds,
            'total_variants_form' => count($this->variants),
            'variants_form_data' => array_map(function($v, $i) {
                return [
                    'index' => $i,
                    'id' => $v['id'] ?? 'nueva',
                    'name' => $v['name'] ?? 'empty',
                    'values_count' => count($v['values'] ?? []),
                    'values' => $v['values'] ?? []
                ];
            }, $this->variants, array_keys($this->variants))
        ]);

        // Procesar cada variante que tenga nombre
        foreach ($this->variants as $originalIndex => $variant) {
            \Log::info('--- Procesando variante ' . $originalIndex . ' ---', [
                'variant_data' => $variant
            ]);
            
            // Solo verificar que tenga nombre
            if (empty(trim($variant['name'] ?? ''))) {
                \Log::warning('SALTANDO variante sin nombre', [
                    'index' => $originalIndex,
                    'name' => $variant['name'] ?? 'null'
                ]);
                continue;
            }
            // Formatear valores
            $values = array_filter($variant['values'] ?? [], function($value) {
                return !empty(trim($value));
            });
            
            $formattedValues = [];
            if (!empty($values)) {
                $variantName = ucfirst($variant['name']);
                $formattedValues[$variantName] = implode(', ', $values);
            }
            
            $variantData = [
                'name' => $variant['name'],
                'values' => json_encode($formattedValues),
                'sku' => $variant['sku'] ?? $this->sku,
                'price_adjustment' => $variant['price_adjustment'] ?? 0,
                'precio_bs' => $variant['precio_bs'] ?? 0,
                'quantity' => $variant['quantity'] ?? 0,
                'empresa_id' => $producto->empresa_id,
                'alt_text' => $variant['alt_text'] ?? ''
            ];

            // Procesar imagen de variante si existe una nueva
            if (isset($this->variantImages[$originalIndex]) && $this->variantImages[$originalIndex] && is_object($this->variantImages[$originalIndex])) {
                try {
                    $image = $this->variantImages[$originalIndex];
                    
                    // Verificar que la imagen es válida y el archivo existe
                    if (!$image->isValid() || !file_exists($image->getRealPath())) {
                        \Log::warning('Imagen de variante no válida, continuando sin imagen', [
                            'index' => $originalIndex,
                            'isValid' => $image->isValid(),
                            'fileExists' => file_exists($image->getRealPath()),
                            'path' => $image->getRealPath()
                        ]);
                        // NO hacer continue aquí, solo no procesar la imagen
                    } else {
                        // Generar nombres de archivo únicos
                        $filename = time() . '_variant_' . $originalIndex . '.' . $image->getClientOriginalExtension();
                        $path = $image->storeAs('productos', $filename);

                        // Verificar que el archivo se guardó correctamente
                        if (!Storage::exists($path)) {
                            throw new \Exception('El archivo no se guardó correctamente');
                        }

                        // Procesar miniatura
                        $thumbnailPath = 'productos/thumb_' . $filename;
                        $imageManager = new \Intervention\Image\ImageManager(new \Intervention\Image\Drivers\Gd\Driver());
                        $thumbnail = $imageManager->read($image->getRealPath());
                        $thumbnail->resize(300, 300, function ($constraint) {
                            $constraint->aspectRatio();
                            $constraint->upsize();
                        });
                        $thumbnail->save(public_path('app/' . $thumbnailPath));

                        if (!file_exists(public_path('app/' . $thumbnailPath))) {
                            throw new \Exception('La miniatura no se generó correctamente');
                        }

                        // Asignar paths solo si todo fue exitoso
                        $variantData['image_path'] = $path;
                        $variantData['image_thumbnail_path'] = $thumbnailPath;

                        \Log::info('Imagen de variante procesada exitosamente', [
                            'index' => $originalIndex,
                            'path' => $path,
                            'thumbnailPath' => $thumbnailPath
                        ]);
                    }

                } catch (\Exception $e) {
                    \Log::error('Error procesando imagen de variante: ' . $e->getMessage());
                    $this->dispatch('show-toast', [
                        'type' => 'error',
                        'message' => 'Error al procesar imagen de variante: ' . $e->getMessage(),
                        'timeout' => 8000
                    ]);
                }
            } elseif (isset($variant['id']) && $variant['id']) {
                // Si es una variante existente y no se subió nueva imagen, mantener las rutas existentes
                $existingVariant = $producto->variants()->find($variant['id']);
                if ($existingVariant) {
                    $variantData['image_path'] = $existingVariant->image_path;
                    $variantData['image_thumbnail_path'] = $existingVariant->image_thumbnail_path;
                }
            } else {
                // Si es una nueva variante sin imagen, establecer valores null explícitamente
                $variantData['image_path'] = null;
                $variantData['image_thumbnail_path'] = null;
            }

            // Debug: Log de los datos que se van a guardar
            \Log::info('Procesando variante', [
                'index' => $originalIndex,
                'variant_id' => $variant['id'] ?? 'nueva',
                'name' => $variant['name'],
                'image_path' => $variantData['image_path'] ?? 'no establecido',
                'image_thumbnail_path' => $variantData['image_thumbnail_path'] ?? 'no establecido',
                'has_image' => isset($this->variantImages[$originalIndex]) && $this->variantImages[$originalIndex] ? 'si' : 'no'
            ]);

            \Log::info('Procesando variante individual', [
                'index' => $originalIndex,
                'name' => $variant['name'],
                'has_id' => isset($variant['id']) && $variant['id'],
                'variant_id' => $variant['id'] ?? 'nueva'
            ]);
            
            // Actualizar o crear la variante
            if (isset($variant['id']) && $variant['id']) {
                // Actualizar variante existente
                $producto->variants()->where('id', $variant['id'])->update($variantData);
                $processedVariantIds[] = $variant['id'];
                \Log::info('Variante actualizada', ['id' => $variant['id']]);
            } else {
                // Crear nueva variante
                $newVariant = $producto->variants()->create($variantData);
                $processedVariantIds[] = $newVariant->id;
                \Log::info('Nueva variante creada', ['id' => $newVariant->id]);
            }
        }

        // Solo eliminar variantes si has_variants sigue siendo true
        if ($this->has_variants) {
            $variantsToDelete = array_diff($existingVariantIds, $processedVariantIds);
            if (!empty($variantsToDelete)) {
                \Log::info('Eliminando variantes no procesadas', [
                    'existing_ids' => $existingVariantIds,
                    'processed_ids' => $processedVariantIds,
                    'to_delete' => $variantsToDelete
                ]);
                $producto->variants()->whereIn('id', $variantsToDelete)->delete();
            }
        }
        
        \Log::info('=== FIN processVariants EDICIÓN ===', [
            'total_processed' => count($processedVariantIds),
            'processed_ids' => $processedVariantIds,
            'final_variant_count' => $producto->variants()->count()
        ]);
    }

    protected function cleanupTemporaryFiles()
    {
        try {
            // Limpiar imágenes principales
            foreach ($this->images as $image) {
                if ($image && method_exists($image, 'getRealPath')) {
                    $tempPath = $image->getRealPath();
                    if (file_exists($tempPath)) {
                        unlink($tempPath);
                    }
                }
            }

            // Limpiar imágenes de variantes
            foreach ($this->variantImages as $image) {
                if ($image && method_exists($image, 'getRealPath')) {
                    $tempPath = $image->getRealPath();
                    if (file_exists($tempPath)) {
                        unlink($tempPath);
                    }
                }
            }

            // Limpiar directorio temporal de Livewire
            $tempDir = public_path('app/livewire-tmp');
            if (is_dir($tempDir)) {
                $files = glob($tempDir . '/*');
                foreach ($files as $file) {
                    if (is_file($file) && (time() - filemtime($file)) > 3600) { // Archivos más antiguos de 1 hora
                        unlink($file);
                    }
                }
            }
        } catch (\Exception $e) {
            \Log::warning('Error limpiando archivos temporales: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $categorias = Categoria::filterByEmpresaSucursal()
            ->where('activo', true)
            ->orderBy('nombre')
            ->get();

        $marcas = Marca::filterByEmpresaSucursal()
            ->where('activo', true)
            ->orderBy('nombre')
            ->get();

        return $this->renderWithLayout('livewire.admin.productos.edit-mejorado', [
            'categorias' => $categorias,
            'marcas' => $marcas
        ], [
            'title' => 'Editar Producto',
            'description' => 'Formulario mejorado de edición de productos con variantes e imágenes',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Productos', 'route' => 'admin.productos.index'],
                ['name' => 'Editar', 'active' => true]
            ]
        ]);
    }
}
