<?php

namespace App\Livewire\Admin\Productos;

use App\Models\Producto;
use App\Models\Categoria;
use App\Models\Marca;
use Livewire\Component;
use Livewire\WithFileUploads;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Str;
use Intervention\Image\Facades\Image;
use App\Models\ProductoVariant;

class CreateMejorado extends Component
{
    use WithFileUploads, HasDynamicLayout;

    // Propiedades del producto
    public $code;
    public $name;
    public $price;
    public $precio_bs;
    public $description;
    public $quantity;
    public $quantity_alert;
    public $status = true;
    public $categoria_id;
    public $marca_id;

    // Propiedades adicionales
    public $sku;
    public $barcode;
    public $weight;
    public $dimensions;
    public $tags;
    public $meta_title;
    public $meta_description;
    public $featured = false;
    
    // Propiedades de lote
    public $batch_number;
    public $expiry_date;
    public $requires_batch_tracking = false;

    // Propiedades de imágenes
    public $images = [];
    public $existingImages = [];
    public $removedImages = [];

    // Propiedades de imágenes de variantes
    public $variantImages = []; // Array de imágenes para cada variante

    // Propiedades de variaciones
    public $has_variants = false;
    public $variants = [];
    public $variantOptions = [];

    // Propiedades de país
    public $empresaPais;

    // Propiedades de inventario
    public $track_inventory = true;
    public $allow_backorder = false;
    public $inventory_policy = 'deny';

    // UI
    public $activeTab = 'basic';
    public $uploading = false;
    public $uploadProgress = 0;

    protected function rules()
    {
        return [
            'code' => 'required|string|max:50|unique:productos,code',
            'name' => 'required|string|max:255',
            'price' => 'required|numeric|min:0',
            'precio_bs' => 'nullable|numeric|min:0',
            'description' => 'nullable|string',
            'quantity' => 'required|integer|min:0',
            'quantity_alert' => 'required|integer|min:0',
            'status' => 'boolean',
            'categoria_id' => 'required|exists:categorias,id',
            'marca_id' => 'required|exists:marcas,id',
            'sku' => 'nullable|string|max:100',
            'barcode' => 'nullable|string|max:100',
            'weight' => 'nullable|numeric|min:0',
            'dimensions' => 'nullable|string|max:255',
            'tags' => 'nullable|string|max:500',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string|max:500',
            'featured' => 'boolean',
            'batch_number' => 'nullable|string|max:100',
            'expiry_date' => 'nullable|date|after:today',
            'requires_batch_tracking' => 'boolean',
            'images.*' => 'image|max:2048',
            'has_variants' => 'boolean',
            'track_inventory' => 'boolean',
            'allow_backorder' => 'boolean',
            'inventory_policy' => 'in:deny,allow',
            // Validaciones para variantes (solo si has_variants es true)
            'variants.*.name' => $this->has_variants ? 'required|string|max:255' : 'nullable|string|max:255',
            'variants.*.values' => $this->has_variants ? 'required|array|min:1' : 'nullable|array',
            'variants.*.values.*' => $this->has_variants ? 'required|string|max:255' : 'nullable|string|max:255',
            'variants.*.price_adjustment' => 'nullable|numeric|min:0',
            'variants.*.precio_bs' => 'nullable|numeric|min:0',
            'variants.*.quantity' => 'nullable|integer|min:0',
            'variants.*.sku' => 'nullable|string|max:100',
            'variants.*.alt_text' => 'nullable|string|max:255'
        ];
    }

    protected $messages = [
        'code.required' => 'El código es requerido.',
        'code.unique' => 'Este código ya existe.',
        'name.required' => 'El nombre es requerido.',
        'price.required' => 'El precio es requerido.',
        'price.numeric' => 'El precio debe ser un número.',
        'price.min' => 'El precio debe ser mayor o igual a 0.',
        'quantity.required' => 'La cantidad es requerida.',
        'quantity.integer' => 'La cantidad debe ser un número entero.',
        'quantity.min' => 'La cantidad debe ser mayor o igual a 0.',
        'quantity_alert.required' => 'La alerta de cantidad es requerida.',
        'categoria_id.required' => 'La categoría es requerida.',
        'marca_id.required' => 'La marca es requerida.',
        'images.*.image' => 'El archivo debe ser una imagen.',
        'images.*.max' => 'La imagen no puede pesar más de 2MB.',
        // Mensajes para validaciones de variantes
        'variants.*.name.required' => 'El nombre de la variante es requerido.',
        'variants.*.name.string' => 'El nombre de la variante debe ser texto.',
        'variants.*.name.max' => 'El nombre de la variante no puede exceder 255 caracteres.',
        'variants.*.values.required' => 'Los valores de la variante son requeridos.',
        'variants.*.values.array' => 'Los valores deben ser un array.',
        'variants.*.values.min' => 'Debe haber al menos un valor.',
        'variants.*.values.*.required' => 'Cada valor es requerido.',
        'variants.*.values.*.string' => 'Cada valor debe ser texto.',
        'variants.*.values.*.max' => 'Cada valor no puede exceder 255 caracteres.',
        'variants.*.price_adjustment.required' => 'El ajuste de precio es requerido.',
        'variants.*.price_adjustment.numeric' => 'El ajuste de precio debe ser un número.',
        'variants.*.price_adjustment.min' => 'El ajuste de precio debe ser mayor o igual a 0.',
        'variants.*.quantity.required' => 'La cantidad es requerida.',
        'variants.*.quantity.integer' => 'La cantidad debe ser un número entero.',
        'variants.*.quantity.min' => 'La cantidad debe ser mayor o igual a 0.',
        'variants.*.sku.required' => 'El SKU de la variante es requerido.',
        'variants.*.sku.string' => 'El SKU debe ser texto.',
        'variants.*.sku.max' => 'El SKU no puede exceder 100 caracteres.',
        'variants.*.alt_text.string' => 'El texto alternativo debe ser texto.',
        'variants.*.alt_text.max' => 'El texto alternativo no puede exceder 255 caracteres.'
    ];

    public function mount()
    {
        $this->generateCode();
        $this->initializeVariants();
        $this->loadEmpresaPais();

        // Inicializar array de imágenes de variantes
        $this->variantImages = [];
    }

    public function loadEmpresaPais()
    {
        $this->empresaPais = auth()->user()->empresa->pais;
    }

    public function getEsVenezuelaProperty()
    {
        return $this->empresaPais && $this->empresaPais->codigo_iso2 === 'VE';
    }

    public function generateCode()
    {
        $lastProduct = Producto::orderBy('id', 'desc')->first();
        $lastId = $lastProduct ? $lastProduct->id : 0;
        $this->code = 'PROD-' . str_pad($lastId + 1, 6, '0', STR_PAD_LEFT);
    }

    public function generateSKU()
    {
        $this->sku = strtoupper(Str::slug($this->name)) . '-' . strtoupper(Str::random(4));
        $this->syncVariantSKUs();
    }

    public function updatedSku($value)
    {
        $this->syncVariantSKUs();
    }

    public function syncVariantSKUs()
    {
        if ($this->has_variants && !empty($this->variants)) {
            foreach ($this->variants as $index => $variant) {
                // Si la variante no tiene SKU propio o tiene el SKU anterior del producto, actualizarlo
                if (empty($variant['sku']) || $variant['sku'] === ($this->sku ?? '')) {
                    $this->variants[$index]['sku'] = $this->sku ?? '';
                }
            }
        }
    }

    public function initializeVariants()
    {
        $this->variants = [
            [
                'name' => '',
                'values' => [''],
                'price_adjustment' => 0,
                'precio_bs' => 0,
                'quantity' => 0,
                'sku' => $this->sku ?? '',
                'alt_text' => '',
                'image' => null
            ]
        ];

        $this->variantOptions = [
            'size' => ['XS', 'S', 'M', 'L', 'XL', 'XXL', 'XXXL'],
            'color' => ['Rojo', 'Azul', 'Verde', 'Negro', 'Blanco', 'Amarillo', 'Naranja', 'Rosa', 'Morado', 'Gris', 'Marrón', 'Beige'],
            'material' => ['Algodón', 'Poliéster', 'Nylon', 'Cuero', 'Lana', 'Seda', 'Lino', 'Denim', 'Spandex', 'Viscosa'],
            'sabor' => ['Chocolate', 'Vainilla', 'Fresa', 'Limón', 'Naranja', 'Menta', 'Coco', 'Caramelo', 'Café', 'Natural'],
            'peso' => ['100g', '250g', '500g', '1kg', '2kg', '5kg', '10kg'],
            'volumen' => ['50ml', '100ml', '250ml', '500ml', '750ml', '1L', '2L', '5L'],
            'capacidad' => ['8GB', '16GB', '32GB', '64GB', '128GB', '256GB', '512GB', '1TB'],
            'potencia' => ['5W', '10W', '15W', '20W', '25W', '50W', '100W'],
            'voltaje' => ['12V', '24V', '110V', '220V'],
            'frecuencia' => ['50Hz', '60Hz'],
            'edad' => ['0-6 meses', '6-12 meses', '1-2 años', '2-4 años', '4-6 años', '6+ años'],
            'genero' => ['Masculino', 'Femenino', 'Unisex'],
            'temporada' => ['Primavera', 'Verano', 'Otoño', 'Invierno', 'Todo el año'],
            'tipo' => ['Básico', 'Premium', 'Deluxe', 'Estándar', 'Profesional'],
            'formato' => ['Líquido', 'Polvo', 'Gel', 'Crema', 'Spray', 'Tableta', 'Cápsula'],
            'concentracion' => ['5%', '10%', '15%', '20%', '25%', '30%', '50%'],
            'dureza' => ['Suave', 'Medio', 'Duro', 'Extra Duro'],
            'textura' => ['Liso', 'Rugoso', 'Mate', 'Brillante', 'Satinado'],
            'resistencia' => ['Baja', 'Media', 'Alta', 'Muy Alta'],
            'temperatura' => ['Frío', 'Ambiente', 'Tibio', 'Caliente'],
            'ph' => ['Ácido', 'Neutro', 'Alcalino'],
            'origen' => ['Nacional', 'Importado', 'Artesanal', 'Industrial'],
            'certificacion' => ['Orgánico', 'Fair Trade', 'Eco-friendly', 'Sin Gluten', 'Vegano'],
            'conectividad' => ['USB', 'Bluetooth', 'WiFi', 'Ethernet', 'Inalámbrico'],
            'compatibilidad' => ['Android', 'iOS', 'Windows', 'Mac', 'Universal'],
            'modelo' => ['2024', '2023', '2022', 'Clásico', 'Nuevo', 'Edición Limitada'],
            'marca_procesador' => ['Intel', 'AMD', 'Apple M1', 'Apple M2', 'Snapdragon', 'MediaTek'],
            'resolucion' => ['HD', 'Full HD', '2K', '4K', '8K', 'Retina'],
            'almacenamiento' => ['32GB', '64GB', '128GB', '256GB', '512GB', '1TB', '2TB'],
            'ram' => ['4GB', '8GB', '16GB', '32GB', '64GB'],
            'pantalla' => ['5"', '6"', '6.5"', '7"', '10"', '13"', '15"', '17"'],
            'bateria' => ['2000mAh', '3000mAh', '4000mAh', '5000mAh', '6000mAh'],
            'camara' => ['8MP', '12MP', '16MP', '24MP', '48MP', '64MP', '108MP'],
            'sistema_operativo' => ['Android 13', 'Android 14', 'iOS 16', 'iOS 17', 'Windows 11', 'macOS'],
            'procesador' => ['Dual Core', 'Quad Core', 'Hexa Core', 'Octa Core'],
            'velocidad' => ['1.5GHz', '2.0GHz', '2.5GHz', '3.0GHz', '3.5GHz'],
            'memoria_grafica' => ['2GB', '4GB', '6GB', '8GB', '12GB', '16GB'],
            'tipo_memoria' => ['DDR3', 'DDR4', 'DDR5', 'LPDDR4', 'LPDDR5'],
            'interfaz' => ['HDMI', 'USB-C', 'Lightning', 'Micro USB', 'Jack 3.5mm'],
            'red' => ['3G', '4G', '5G', 'WiFi 5', 'WiFi 6', 'WiFi 6E'],
            'sensor' => ['Huella', 'Facial', 'Iris', 'Proximidad', 'Giroscopio'],
            'carga' => ['5W', '10W', '15W', '20W', '25W', '30W', '45W', '65W', '100W'],
            'tipo_carga' => ['Inalámbrica', 'Rápida', 'Super Rápida', 'USB-C', 'Lightning'],
            'resistencia_agua' => ['IPX4', 'IPX5', 'IPX7', 'IP67', 'IP68'],
            'acabado' => ['Mate', 'Brillante', 'Metalizado', 'Texturizado', 'Transparente'],
            'forma' => ['Rectangular', 'Cuadrado', 'Circular', 'Ovalado', 'Irregular'],
            'estilo' => ['Moderno', 'Clásico', 'Vintage', 'Minimalista', 'Elegante'],
            'uso' => ['Casual', 'Formal', 'Deportivo', 'Trabajo', 'Fiesta'],
            'ocasion' => ['Diario', 'Especial', 'Navidad', 'Cumpleaños', 'Aniversario'],
            'publico' => ['Niños', 'Adolescentes', 'Adultos', 'Tercera Edad', 'Familiar'],
            'nivel' => ['Principiante', 'Intermedio', 'Avanzado', 'Profesional', 'Experto'],
            'dificultad' => ['Fácil', 'Medio', 'Difícil', 'Muy Difícil', 'Extremo'],
            'categoria_edad' => ['0+', '3+', '6+', '12+', '16+', '18+'],
            'idioma' => ['Español', 'Inglés', 'Francés', 'Portugués', 'Italiano'],
            'region' => ['América', 'Europa', 'Asia', 'África', 'Oceanía'],
            'clima' => ['Tropical', 'Templado', 'Frío', 'Seco', 'Húmedo'],
            'aplicacion' => ['Interior', 'Exterior', 'Baño', 'Cocina', 'Jardín'],
            'instalacion' => ['Fácil', 'Moderada', 'Compleja', 'Profesional'],
            'mantenimiento' => ['Bajo', 'Medio', 'Alto', 'Sin Mantenimiento'],
            'garantia' => ['6 meses', '1 año', '2 años', '3 años', '5 años', 'De por vida'],
            'eficiencia' => ['A+++', 'A++', 'A+', 'A', 'B', 'C'],
            'consumo' => ['Bajo', 'Medio', 'Alto', 'Muy Alto'],
            'ruido' => ['Silencioso', 'Bajo', 'Moderado', 'Alto'],
            'velocidad_maxima' => ['30 km/h', '50 km/h', '80 km/h', '120 km/h'],
            'autonomia' => ['50km', '100km', '200km', '300km', '500km'],
            'tiempo_carga' => ['30min', '1h', '2h', '4h', '8h'],
            'numero_puertos' => ['1', '2', '4', '6', '8', '12'],
            'longitud' => ['1m', '2m', '3m', '5m', '10m', '15m'],
            'ancho' => ['10cm', '20cm', '30cm', '50cm', '100cm'],
            'grosor' => ['1mm', '2mm', '5mm', '10mm', '20mm'],
            'densidad' => ['Baja', 'Media', 'Alta', 'Muy Alta'],
            'flexibilidad' => ['Rígido', 'Semi-rígido', 'Flexible', 'Muy Flexible'],
            'transparencia' => ['Opaco', 'Translúcido', 'Transparente', 'Cristalino'],
            'aroma' => ['Sin aroma', 'Suave', 'Intenso', 'Cítrico', 'Floral', 'Frutal'],
            'presentacion' => ['Individual', 'Pack 2', 'Pack 3', 'Pack 5', 'Pack 10'],
            'unidades' => ['1 unidad', '2 unidades', '5 unidades', '10 unidades', '25 unidades', '50 unidades'],
            'medida' => ['Pequeño', 'Mediano', 'Grande', 'Extra Grande', 'Jumbo']
        ];
    }

    public function addVariant()
    {
        $this->variants[] = [
            'name' => '',
            'values' => [''],
            'price_adjustment' => 0,
            'precio_bs' => 0,
            'quantity' => 0,
            'sku' => $this->sku ?? '',
            'alt_text' => '',
            'image' => null
        ];
    }

    public function changeTab($tab)
    {
        $this->activeTab = $tab;
    }

    public function removeVariant($index)
    {
        unset($this->variants[$index]);
        $this->variants = array_values($this->variants);
    }

    public function addVariantValue($variantIndex)
    {
        $this->variants[$variantIndex]['values'][] = '';
    }

    public function removeVariantValue($variantIndex, $valueIndex)
    {
        unset($this->variants[$variantIndex]['values'][$valueIndex]);
        $this->variants[$variantIndex]['values'] = array_values($this->variants[$variantIndex]['values']);
    }

    public function loadVariantOptions($variantIndex, $variantType)
    {
        if (!empty($variantType) && isset($this->variantOptions[$variantType])) {
            // Solo cargar si no hay valores o solo hay un valor vacío
            $currentValues = $this->variants[$variantIndex]['values'] ?? [''];
            $hasOnlyEmptyValue = count($currentValues) === 1 && empty($currentValues[0]);
            
            if (empty($currentValues) || $hasOnlyEmptyValue) {
                $this->variants[$variantIndex]['values'] = [];
            }
        }
    }

    public function updatedImages()
    {
        $this->validate([
            'images.*' => 'image|max:2048'
        ]);
    }

    public function removeImage($index)
    {
        unset($this->images[$index]);
        $this->images = array_values($this->images);
    }

    public function removeExistingImage($imageId)
    {
        $this->removedImages[] = $imageId;
        $this->existingImages = array_filter($this->existingImages, function($img) use ($imageId) {
            return $img['id'] != $imageId;
        });
    }

    public function save()
    {
        // Validar sin las imágenes de variantes para evitar errores con archivos temporales
        $rules = $this->rules();
        unset($rules['variantImages.*']);
        
        // Si no tiene variantes, remover validaciones de variantes
        if (!$this->has_variants) {
            $rules = array_filter($rules, function($key) {
                return !str_starts_with($key, 'variants.');
            }, ARRAY_FILTER_USE_KEY);
        }
        
        $this->validate($rules);

        try {
            // Crear el producto
            $producto = Producto::create([
                'code' => $this->code,
                'name' => $this->name,
                'slug' => Str::slug($this->name),
                'price' => $this->price,
                'precio_bs' => $this->precio_bs,
                'description' => $this->description,
                'quantity' => $this->quantity,
                'quantity_alert' => $this->quantity_alert,
                'status' => $this->status,
                'categoria_id' => $this->categoria_id,
                'marca_id' => $this->marca_id,
                'weight' => $this->weight,
                'dimensions' => $this->dimensions,
                'batch_number' => $this->batch_number,
                'expiry_date' => $this->expiry_date,
                'requires_batch_tracking' => $this->requires_batch_tracking,
                'has_variants' => $this->has_variants,
                'track_inventory' => $this->track_inventory,
                'allow_backorder' => $this->allow_backorder,
                'inventory_policy' => $this->inventory_policy,
                'empresa_id' => auth()->user()->empresa_id,
                'sucursal_id' => auth()->user()->sucursal_id,
            ]);

            // Guardar imágenes principales
            if ($this->images) {
                foreach ($this->images as $image) {
                    $path = $image->store('productos', 'public');

                    $producto->images()->create([
                        'path' => $path,
                        'alt_text' => $this->name,
                        'is_primary' => false,
                        'empresa_id' => auth()->user()->empresa_id,
                    ]);
                }
            }

            // Guardar imágenes de variantes
            if ($this->has_variants && !empty($this->variants)) {
                foreach ($this->variants as $variant) {
                    if ($variant['image']) {
                        $path = $variant['image']->store('productos', 'public');

                        $producto->images()->create([
                            'path' => $path,
                            'alt_text' => $variant['alt_text'] ?? $this->name,
                            'is_primary' => false,
                            'empresa_id' => auth()->user()->empresa_id,
                        ]);
                    }
                }
            }

            // Crear SEO
            \App\Models\ProductoSeo::create([
                'producto_id' => $producto->id,
                'sku' => $this->sku,
                'barcode' => $this->barcode,
                'tags' => $this->tags,
                'meta_title' => $this->meta_title,
                'meta_description' => $this->meta_description,
                'featured' => $this->featured,
                'empresa_id' => auth()->user()->empresa_id,
            ]);

            // Crear configuración de inventario
            $producto->inventorySetting()->create([
                'track_inventory' => $this->track_inventory,
                'allow_backorder' => $this->allow_backorder,
                'inventory_policy' => $this->inventory_policy,
                'empresa_id' => auth()->user()->empresa_id,
            ]);

            // Procesar imágenes
            if (!empty($this->images)) {
                $this->processImages($producto);
                $this->cleanupTemporaryFiles();
            }

            // Procesar variaciones SOLO si están habilitadas
            if ($this->has_variants) {
                \Log::info('Procesando variantes porque has_variants es true', [
                    'has_variants' => $this->has_variants,
                    'variants_count' => count($this->variants)
                ]);
                $this->processVariants($producto);
            } else {
                \Log::info('NO procesando variantes porque has_variants es false', [
                    'has_variants' => $this->has_variants
                ]);
            }

            // Registrar movimiento de inventario inicial si hay cantidad
            if ($this->quantity > 0) {
                $producto->inventoryMovements()->create([
                    'type' => 'ingreso',
                    'quantity' => $this->quantity,
                    'quantity_before' => 0,
                    'quantity_after' => $this->quantity,
                    'reference' => 'Creación producto: ' . $this->code,
                    'notes' => 'Stock inicial del producto',
                    'user_id' => auth()->id(),
                    'empresa_id' => auth()->user()->empresa_id,
                    'sucursal_id' => auth()->user()->sucursal_id,
                ]);
            }

            // Mostrar notificación de éxito
            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => 'Producto creado exitosamente',
                'timeout' => 5000
            ]);

            // Redirigir después de un breve retraso para que se vea la notificación
            return redirect()->route('admin.productos.index')->with([
                'toast' => [
                    'type' => 'success',
                    'message' => 'Producto creado exitosamente'
                ]
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error de validación: ' . collect($e->errors())->flatten()->first(),
                'timeout' => 8000
            ]);
            throw $e;
        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al crear el producto: ' . $e->getMessage(),
                'timeout' => 8000
            ]);
            report($e);
        }
    }

    protected function processImages($producto)
    {
        foreach ($this->images as $index => $image) {
            if ($image) {
                try {
                    // Procesar imagen principal
                    $filename = time() . '_' . $index . '.' . $image->getClientOriginalExtension();
                    $path = $image->storeAs('productos', $filename);

                    // Crear miniatura
                    $thumbnailPath = 'productos/thumb_' . $filename;
                    // Usar la nueva API de Intervention Image v3
                    $imageManager = new \Intervention\Image\ImageManager(
                        new \Intervention\Image\Drivers\Gd\Driver()
                    );
                    $thumbnail = $imageManager->read($image->getRealPath());
                    $thumbnail->cover(300, 300);
                    $thumbnail->save(public_path('app/' . $thumbnailPath));

                    // Guardar en la base de datos - solo usar campos existentes
                    $producto->images()->create([
                        'path' => $path,
                        'featured' => $index === 0, // Solo marcar la primera imagen como destacada
                        'empresa_id' => auth()->user()->empresa_id,
                    ]);

                } catch (\Exception $e) {
                    \Log::error('Error al procesar imagen: ' . $e->getMessage());
                }
            }
        }
    }

    protected function processVariants($producto, $variants = null)
    {
        $variantsToProcess = $variants ?? $this->variants;
        
        \Log::info('Iniciando processVariants', [
            'total_variants_received' => count($variantsToProcess),
            'variants_data' => $variantsToProcess
        ]);
        
        // Crear todas las variantes que tengan nombre
        foreach ($variantsToProcess as $originalIndex => $variant) {
            // Solo verificar que tenga nombre
            if (empty(trim($variant['name'] ?? ''))) {
                \Log::info('Saltando variante sin nombre', ['index' => $originalIndex]);
                continue;
            }
            
            // Formatear valores
            $values = array_filter($variant['values'] ?? [], function($value) {
                return !empty(trim($value));
            });
            
            $formattedValues = [];
            if (!empty($values)) {
                $variantName = ucfirst($variant['name']);
                $formattedValues[$variantName] = implode(', ', $values);
            }

            $variantData = [
                'name' => $variant['name'],
                'values' => json_encode($formattedValues),
                'sku' => $variant['sku'] ?? $this->sku,
                'price_adjustment' => $variant['price_adjustment'] ?? 0,
                'precio_bs' => $variant['precio_bs'] ?? 0,
                'quantity' => $variant['quantity'] ?? 0,
                'empresa_id' => $producto->empresa_id,
                'alt_text' => $variant['alt_text'] ?? ''
            ];

            // Procesar imagen de variante si existe
            if (isset($this->variantImages[$originalIndex]) && $this->variantImages[$originalIndex] && is_object($this->variantImages[$originalIndex])) {
                try {
                    $image = $this->variantImages[$originalIndex];

                    if ($image->isValid() && file_exists($image->getRealPath())) {
                        $filename = time() . '_variant_' . $originalIndex . '.' . $image->getClientOriginalExtension();
                        $path = $image->storeAs('productos', $filename);

                        $thumbnailPath = 'productos/thumb_' . $filename;
                        $imageManager = new \Intervention\Image\ImageManager(
                            new \Intervention\Image\Drivers\Gd\Driver()
                        );
                        $thumbnail = $imageManager->read($image->getRealPath());
                        $thumbnail->resize(300, 300, function ($constraint) {
                            $constraint->aspectRatio();
                            $constraint->upsize();
                        });
                        $thumbnail->save(public_path('app/' . $thumbnailPath));

                        $variantData['image_path'] = $path;
                        $variantData['image_thumbnail_path'] = $thumbnailPath;
                    }
                } catch (\Exception $e) {
                    \Log::error('Error al procesar imagen de variante: ' . $e->getMessage());
                }
            }

            // Crear la variante
            \Log::info('Creando variante', [
                'index' => $originalIndex,
                'name' => $variant['name'],
                'values' => $formattedValues
            ]);
            
            $producto->variants()->create($variantData);
        }
    }

    protected function cleanupTemporaryFiles()
    {
        try {
            // Limpiar imágenes principales
            foreach ($this->images as $image) {
                if ($image && method_exists($image, 'getRealPath')) {
                    $tempPath = $image->getRealPath();
                    if (file_exists($tempPath)) {
                        unlink($tempPath);
                    }
                }
            }

            // Limpiar imágenes de variantes
            foreach ($this->variantImages as $image) {
                if ($image && method_exists($image, 'getRealPath')) {
                    $tempPath = $image->getRealPath();
                    if (file_exists($tempPath)) {
                        unlink($tempPath);
                    }
                }
            }

            // Limpiar directorio temporal de Livewire si está vacío
            $tempDir = public_path('app/livewire-tmp');
            if (is_dir($tempDir) && count(scandir($tempDir)) <= 2) {
                rmdir($tempDir);
            }
        } catch (\Exception $e) {
            \Log::warning('Error limpiando archivos temporales: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $categorias = Categoria::filterByEmpresaSucursal()
            ->where('activo', true)
            ->orderBy('nombre')
            ->get();

        $marcas = Marca::filterByEmpresaSucursal()
            ->where('activo', true)
            ->orderBy('nombre')
            ->get();

        return $this->renderWithLayout('livewire.admin.productos.create-mejorado', [
            'categorias' => $categorias,
            'marcas' => $marcas
        ], [
            'title' => 'Crear Producto',
            'description' => 'Formulario mejorado de creación de productos con variantes e imágenes',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Productos', 'route' => 'admin.productos.index'],
                ['name' => 'Crear', 'active' => true]
            ]
        ]);
    }
}
