<?php

namespace App\Livewire\Admin\Pos;

use App\Models\Caja;
use App\Models\Cliente;
use App\Models\DetalleVenta;
use App\Models\InventoryMovement;
use App\Models\Producto;
use App\Models\ProductoVariant;
use App\Models\Serie;
use App\Models\Venta;
use App\Models\VentaPago;
use App\Services\CajaMovimientoService;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;
use Illuminate\Support\Facades\DB;
use Livewire\Component;

class Index extends Component
{
    use HasDynamicLayout, HasRegionalFormatting;

    // Caja
    public $cajaAbierta = null;
    public $showModalAbrirCaja = false;
    public $showModalDetalleCaja = false;
    public $showModalSalidaDinero = false;
    public $montoInicial = 0;
    public $observaciones = '';
    public $montoSalida = 0;
    public $motivoSalida = '';
    public $tipoSalida = 'vuelto';
    public $salidasDinero = [];

    // Búsqueda
    public $search = '';
    public $barcode = '';
    public $searchResults = [];
    public $categoriaFilter = '';
    public $marcaFilter = '';

    // Carrito
    public $cart = [];
    public $showVariantsModal = false;
    public $selectedProductForVariants = null;
    public $selectedVariant = null;
    public $variantQuantity = 1;
    public $cartItemNotes = [];
    public $globalDiscount = 0;
    public $editingQuantity = [];
    public $favoriteProducts = [];

    // Pago
    public $showPaymentModal = false;
    public $showCustomerModal = false;
    public $cliente_id = null;
    public $newCustomer = [
        'name' => '',
        'email' => '',
        'telefono' => '',
        'direccion' => ''
    ];
    public $metodoPago = 'efectivo';
    public $tipoDocumento = 'boleta';
    public $serie_id = null;
    public $montoPagado = 0;
    public $montoCambio = 0;
    public $observacionesVenta = '';

    // Hardware
    public $printerEnabled = true;
    public $cashDrawerEnabled = true;
    public $customerDisplayEnabled = false;

    public $clientes = [];
    public $series = [];
    public $categorias = [];
    public $marcas = [];
    public $recentSales = [];

    protected $listeners = ['focusBarcodeInput'];

    public function mount()
    {
        $this->bootHasRegionalFormatting();
        $this->verificarCajaAbierta();
        $this->loadData();
        $this->loadFavoriteProducts();
        $this->loadRecentSales();
    }

    public function loadRecentSales()
    {
        $this->recentSales = \App\Models\Venta::where('user_id', auth()->id())
            ->where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->with(['cliente'])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get()
            ->map(function($venta) {
                return [
                    'id' => $venta->id,
                    'codigo' => $venta->codigo,
                    'total' => $venta->total,
                    'cliente' => $venta->cliente ? $venta->cliente->nombres : 'Cliente General',
                    'fecha' => $venta->created_at->format('d/m H:i'),
                    'metodo_pago' => ucfirst($venta->metodo_pago),
                ];
            })->toArray();
    }

    public function loadFavoriteProducts()
    {
        // Cargar productos más vendidos como favoritos
        $this->favoriteProducts = Producto::where('status', true)
            ->where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->with(['featuredImage'])
            ->orderBy('quantity', 'desc')
            ->limit(6)
            ->get()
            ->map(function($producto) {
                return [
                    'id' => $producto->id,
                    'name' => $producto->name,
                    'code' => $producto->code,
                    'price' => (float) $producto->precio,
                    'quantity' => $producto->quantity,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'has_variants' => $producto->variants && $producto->variants->count() > 0,
                ];
            })->toArray();
    }

    public function loadData()
    {
        $this->clientes = Cliente::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->orderBy('nombres')
            ->get();

        $this->categorias = \App\Models\Categoria::where('activo', true)
            ->where('empresa_id', auth()->user()->empresa_id)
            ->get();

        $this->marcas = \App\Models\Marca::where('activo', true)
            ->where('empresa_id', auth()->user()->empresa_id)
            ->get();

        $this->loadSeries();
    }

    public function loadSeries()
    {
        $this->series = Serie::where('activo', true)
            ->where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->where('tipo_documento', $this->tipoDocumento)
            ->get();

        if ($this->series->isNotEmpty()) {
            $this->serie_id = $this->series->first()->id;
        }
    }

    public function updatedTipoDocumento()
    {
        $this->loadSeries();
    }

    public function verificarCajaAbierta()
    {
        $this->cajaAbierta = Caja::obtenerCajaAbierta(
            auth()->user()->empresa_id,
            auth()->user()->sucursal_id
        );

        if ($this->cajaAbierta) {
            $this->cargarSalidasDinero();
        }
    }

    public function cargarSalidasDinero()
    {
        if ($this->cajaAbierta) {
            $this->salidasDinero = $this->cajaAbierta->movimientos()
                ->orderBy('created_at', 'desc')
                ->get()
                ->map(function ($movimiento) {
                    return [
                        'id' => $movimiento->id,
                        'tipo' => $movimiento->tipo,
                        'tipo_salida' => $movimiento->tipo_salida,
                        'monto' => $movimiento->monto,
                        'motivo' => $movimiento->motivo,
                        'fecha' => $movimiento->created_at->format('d/m/Y H:i'),
                        'usuario' => $movimiento->usuario->name ?? 'Sistema',
                    ];
                })->toArray();
        }
    }

    public function abrirModalCaja()
    {
        if ($this->cajaAbierta) {
            $this->dispatch('show-toast', ['type' => 'warning', 'message' => 'La caja ya ha sido aperturada']);
            return;
        }
        
        $this->showModalAbrirCaja = true;
    }

    public function cerrarModalCaja()
    {
        $this->showModalAbrirCaja = false;
        $this->reset(['montoInicial', 'observaciones']);
    }

    public function abrirCaja()
    {
        $this->validate([
            'montoInicial' => 'required|numeric|min:0',
        ]);

        $this->cajaAbierta = Caja::crearCajaDiaria(
            auth()->user()->empresa_id,
            auth()->user()->sucursal_id,
            $this->montoInicial,
            $this->observaciones,
            auth()->id()
        );

        $this->cerrarModalCaja();
        $this->dispatch('show-toast', ['type' => 'success', 'message' => 'Caja abierta exitosamente']);
    }

    public function abrirModalDetalleCaja()
    {
        if ($this->cajaAbierta) {
            $this->cajaAbierta->calcularTotales();
        }
        $this->showModalDetalleCaja = true;
    }

    public function cerrarModalDetalleCaja()
    {
        $this->showModalDetalleCaja = false;
    }

    public function abrirModalSalidaDinero()
    {
        $this->showModalSalidaDinero = true;
        $this->reset(['montoSalida', 'motivoSalida']);
        $this->tipoSalida = 'vuelto';
    }

    public function cerrarModalSalidaDinero()
    {
        $this->showModalSalidaDinero = false;
    }

    public function registrarSalidaDinero()
    {
        $this->validate([
            'montoSalida' => 'required|numeric|min:0.01',
            'motivoSalida' => 'required|string|max:500',
            'tipoSalida' => 'required|in:vuelto,compra,retiro,otro',
        ]);

        $service = new CajaMovimientoService();
        $resultado = $service->registrarSalida(
            $this->cajaAbierta,
            [
                'monto' => $this->montoSalida,
                'motivo' => $this->motivoSalida,
                'tipo' => $this->tipoSalida,
            ],
            request()->ip(),
            request()->userAgent()
        );

        if ($resultado['success']) {
            $this->cargarSalidasDinero();
            $this->cerrarModalSalidaDinero();
            $this->dispatch('show-toast', ['type' => 'success', 'message' => 'Salida registrada correctamente']);
        } else {
            $this->dispatch('show-toast', ['type' => 'error', 'message' => $resultado['message']]);
        }
    }

    public function updatedSearch()
    {
        $this->searchProducts();
    }

    public function updatedBarcode()
    {
        if (strlen($this->barcode) > 3) {
            $this->searchProducts();
        }
    }

    public function searchProducts()
    {
        $term = $this->search ?: $this->barcode;

        if (strlen($term) < 2) {
            $this->searchResults = [];
            return;
        }

        $query = Producto::where('status', true)
            ->where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->with(['variants', 'featuredImage']);

        if ($this->categoriaFilter) {
            $query->where('categoria_id', $this->categoriaFilter);
        }

        if ($this->marcaFilter) {
            $query->where('marca_id', $this->marcaFilter);
        }

        // Búsqueda exacta por código
        $exactMatch = clone $query;
        $productos = $exactMatch->where('code', $term)->limit(5)->get();

        if ($productos->isEmpty()) {
            $productos = $query->where(function($q) use ($term) {
                $q->where('name', 'like', '%' . $term . '%')
                  ->orWhere('description', 'like', '%' . $term . '%')
                  ->orWhere('code', 'like', '%' . $term . '%');
            })->limit(20)->get();
        }

        $this->searchResults = $productos->map(function($producto) {
            return [
                'id' => $producto->id,
                'name' => $producto->name,
                'code' => $producto->code,
                'description' => $producto->description,
                'price' => (float) $producto->precio,
                'quantity' => $producto->quantity,
                'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                'has_variants' => $producto->variants && $producto->variants->count() > 0,
            ];
        })->toArray();

        // Si es código de barras y hay resultado exacto, agregar automáticamente
        if ($this->barcode && count($this->searchResults) === 1) {
            $this->addToCart($this->searchResults[0]['id']);
            $this->barcode = '';
            $this->searchResults = [];
        }
    }

    public function addToCart($productoId)
    {
        $producto = Producto::with('variants')->find($productoId);
        if (!$producto) return;

        if ($producto->variants && $producto->variants->count() > 0) {
            $this->openVariantsModal($productoId);
        } else {
            $this->addProductToCart($producto, 1);
        }
    }

    public function openVariantsModal($productoId)
    {
        $this->selectedProductForVariants = Producto::with('variants')->find($productoId);
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
        $this->showVariantsModal = true;
    }

    public function closeVariantsModal()
    {
        $this->showVariantsModal = false;
        $this->selectedProductForVariants = null;
        $this->selectedVariant = null;
        $this->variantQuantity = 1;
    }

    public function selectVariant($variantId)
    {
        $this->selectedVariant = $variantId;
    }

    public function addVariantToCart()
    {
        if (!$this->selectedProductForVariants || !$this->selectedVariant) {
            $this->dispatch('show-toast', ['type' => 'error', 'message' => 'Seleccione una opción del producto']);
            return;
        }

        $producto = $this->selectedProductForVariants;

        if ($this->selectedVariant === 'main') {
            $this->addProductToCart($producto, $this->variantQuantity);
        } else {
            $variante = $producto->variants()->find($this->selectedVariant);
            if (!$variante) return;

            if ($variante->quantity < $this->variantQuantity) {
                $this->dispatch('show-toast', ['type' => 'error', 'message' => 'Stock insuficiente. Disponible: ' . $variante->quantity]);
                return;
            }

            $precioVariante = $producto->precio + $variante->price_adjustment;
            $values = $variante->formatted_values;

            $existingIndex = collect($this->cart)->search(function ($item) use ($variante) {
                return isset($item['variante_id']) && $item['variante_id'] == $variante->id;
            });

            if ($existingIndex !== false) {
                $newQty = $this->cart[$existingIndex]['cantidad'] + $this->variantQuantity;
                if ($newQty > $variante->quantity) {
                    $this->dispatch('show-toast', ['type' => 'error', 'message' => 'Stock insuficiente']);
                    return;
                }
                $this->cart[$existingIndex]['cantidad'] = $newQty;
                $this->cart[$existingIndex]['subtotal'] = $newQty * $this->cart[$existingIndex]['precio_unitario'];
            } else {
                $this->cart[] = [
                    'producto_id' => $producto->id,
                    'variante_id' => $variante->id,
                    'nombre' => $producto->name . ' - ' . $variante->name . ': ' . $values,
                    'codigo' => $variante->sku ?? $producto->code,
                    'cantidad' => $this->variantQuantity,
                    'precio_unitario' => (float) $precioVariante,
                    'subtotal' => $this->variantQuantity * (float) $precioVariante,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'stock_disponible' => $variante->quantity,
                    'es_variante' => true,
                    'notas' => '',
                ];
            }
        }

        $this->closeVariantsModal();
        $this->dispatch('show-toast', ['type' => 'success', 'message' => 'Producto agregado al carrito']);
    }

    private function addProductToCart($producto, $cantidad = 1)
    {
        if ($producto->quantity < $cantidad) {
            $this->dispatch('show-toast', ['type' => 'error', 'message' => 'Stock insuficiente. Disponible: ' . $producto->quantity]);
            return;
        }

        $existingIndex = collect($this->cart)->search(function ($item) use ($producto) {
            return $item['producto_id'] == $producto->id && !isset($item['variante_id']);
        });

        if ($existingIndex !== false) {
            $newQty = $this->cart[$existingIndex]['cantidad'] + $cantidad;
            if ($newQty > $producto->quantity) {
                $this->dispatch('show-toast', ['type' => 'error', 'message' => 'Stock insuficiente']);
                return;
            }
            $this->cart[$existingIndex]['cantidad'] = $newQty;
            $this->cart[$existingIndex]['subtotal'] = $newQty * $this->cart[$existingIndex]['precio_unitario'];
        } else {
            $this->cart[] = [
                'producto_id' => $producto->id,
                'nombre' => $producto->name,
                'codigo' => $producto->code,
                'cantidad' => $cantidad,
                'precio_unitario' => (float) $producto->precio,
                'subtotal' => $cantidad * (float) $producto->precio,
                'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                'stock_disponible' => $producto->quantity,
                'es_variante' => false,
                'notas' => '',
            ];
        }

        $this->search = '';
        $this->barcode = '';
        $this->searchResults = [];
        // Don't reset cliente_id when adding products
    }

    public function removeFromCart($index)
    {
        unset($this->cart[$index]);
        $this->cart = array_values($this->cart);
    }

    public function updateQuantity($index, $quantity)
    {
        if ($quantity < 1) {
            $this->removeFromCart($index);
            return;
        }

        $this->cart[$index]['cantidad'] = $quantity;
        $this->cart[$index]['subtotal'] = $quantity * $this->cart[$index]['precio_unitario'];
    }

    public function incrementCartItem($index)
    {
        if (isset($this->cart[$index])) {
            $this->cart[$index]['cantidad']++;
            $this->cart[$index]['subtotal'] = $this->cart[$index]['cantidad'] * $this->cart[$index]['precio_unitario'];
        }
    }

    public function decrementCartItem($index)
    {
        if (isset($this->cart[$index]) && $this->cart[$index]['cantidad'] > 1) {
            $this->cart[$index]['cantidad']--;
            $this->cart[$index]['subtotal'] = $this->cart[$index]['cantidad'] * $this->cart[$index]['precio_unitario'];
        }
    }

    public function clearCart()
    {
        $this->cart = [];
        $this->cartItemNotes = [];
        $this->globalDiscount = 0;
        $this->editingQuantity = [];
        $this->dispatch('show-toast', ['type' => 'success', 'message' => 'Carrito limpiado']);
    }

    public function duplicateCartItem($index)
    {
        if (isset($this->cart[$index])) {
            $item = $this->cart[$index];
            $item['cantidad'] = 1;
            $item['subtotal'] = $item['precio_unitario'];
            $this->cart[] = $item;
            $this->dispatch('show-toast', ['type' => 'success', 'message' => 'Producto duplicado']);
        }
    }

    public function addMoreSameProduct($index)
    {
        if (isset($this->cart[$index])) {
            $item = $this->cart[$index];
            if (isset($item['variante_id'])) {
                $this->addToCart($item['producto_id']);
            } else {
                $producto = Producto::find($item['producto_id']);
                if ($producto) {
                    $this->addProductToCart($producto, 1);
                }
            }
        }
    }

    public function updateCartItemNotes($index, $notes)
    {
        if (isset($this->cart[$index])) {
            $this->cart[$index]['notas'] = $notes;
        }
    }

    public function updateGlobalDiscount($discount)
    {
        $this->globalDiscount = max(0, min(100, $discount));
    }

    public function toggleQuantityEdit($index)
    {
        $this->editingQuantity[$index] = !($this->editingQuantity[$index] ?? false);
    }

    public function updateCartItemQuantity($index, $quantity)
    {
        if (isset($this->cart[$index])) {
            $quantity = max(1, $quantity);
            $this->cart[$index]['cantidad'] = $quantity;
            $this->cart[$index]['subtotal'] = $quantity * $this->cart[$index]['precio_unitario'];
            $this->editingQuantity[$index] = false;
        }
    }

    public function getTotal()
    {
        $subtotal = collect($this->cart)->sum('subtotal');
        return $subtotal * (1 - $this->globalDiscount / 100);
    }

    public function getSubtotal()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function getDiscountAmount()
    {
        return $this->getSubtotal() * ($this->globalDiscount / 100);
    }

    public function openPaymentModal()
    {
        if (empty($this->cart)) {
            $this->dispatch('show-toast', ['type' => 'error', 'message' => 'Agregue productos al carrito']);
            return;
        }

        $this->montoPagado = $this->getTotal();
        $this->calcularCambio();
        $this->showPaymentModal = true;
    }

    public function closePaymentModal()
    {
        $this->showPaymentModal = false;
        $this->reset(['metodoPago', 'montoPagado', 'montoCambio', 'observacionesVenta']);
        $this->metodoPago = 'efectivo';
        $this->tipoDocumento = 'boleta';
        $this->loadSeries();
        // Keep cliente_id selected
    }

    public function updatedMontoPagado()
    {
        $this->calcularCambio();
    }

    public function updatedMetodoPago()
    {
        $this->calcularCambio();
    }

    public function calcularCambio()
    {
        if ($this->metodoPago === 'efectivo') {
            $this->montoCambio = max(0, $this->montoPagado - $this->getTotal());
        } else {
            $this->montoCambio = 0;
            $this->montoPagado = $this->getTotal();
        }
    }

    public function procesarVenta()
    {
        $this->validate([
            'metodoPago' => 'required|in:efectivo,transferencia,tarjeta,pago_movil,punto_de_venta',
            'tipoDocumento' => 'required|in:boleta,factura',
            'serie_id' => 'required|exists:series,id',
            'montoPagado' => 'required|numeric|min:' . $this->getTotal(),
        ]);

        if (empty($this->cart)) {
            $this->dispatch('show-toast', ['type' => 'error', 'message' => 'El carrito está vacío']);
            return;
        }

        try {
            DB::transaction(function () {
                $serie = Serie::find($this->serie_id);

                // Buscar próximo número disponible
                $nextNumber = $serie->correlativo_actual + 1;
                while (Venta::where('serie_id', $this->serie_id)
                    ->where('numero_documento', $nextNumber)
                    ->where('empresa_id', auth()->user()->empresa_id)
                    ->exists()) {
                    $nextNumber++;
                }

                $venta = Venta::create([
                    'codigo' => 'POS-' . time(),
                    'total' => $this->getTotal(),
                    'total_usd' => $this->getTotal(),
                    'total_bs' => $this->getTotal(),
                    'metodo_pago' => $this->metodoPago,
                    'estado' => 'pagado',
                    'tipo_documento' => $this->tipoDocumento,
                    'numero_documento' => $nextNumber,
                    'serie_id' => $this->serie_id,
                    'caja_id' => $this->cajaAbierta->id,
                    'empresa_id' => auth()->user()->empresa_id,
                    'sucursal_id' => auth()->user()->sucursal_id,
                    'user_id' => auth()->id(),
                    'empleado_id' => auth()->id(),
                    'cliente_id' => $this->cliente_id,
                    'observaciones' => $this->observacionesVenta,
                ]);

                foreach ($this->cart as $item) {
                    DetalleVenta::create([
                        'venta_id' => $venta->id,
                        'producto_id' => $item['producto_id'],
                        'variante_id' => $item['variante_id'] ?? null,
                        'cantidad' => $item['cantidad'],
                        'precio_unitario' => $item['precio_unitario'],
                        'subtotal' => $item['subtotal'],
                        'nombre_producto' => $item['nombre'],
                        'codigo_producto' => $item['codigo'],
                    ]);

                    // Actualizar stock
                    if (isset($item['variante_id'])) {
                        $variante = ProductoVariant::find($item['variante_id']);
                        if ($variante) {
                            $stockAnterior = $variante->quantity;
                            $variante->decrement('quantity', $item['cantidad']);

                            InventoryMovement::create([
                                'producto_id' => $item['producto_id'],
                                'type' => 'salida',
                                'quantity' => $item['cantidad'],
                                'quantity_before' => $stockAnterior,
                                'quantity_after' => $variante->quantity,
                                'reference' => 'Venta POS #' . $venta->codigo,
                                'notes' => 'Venta ID: ' . $venta->id . ' - Variante: ' . $item['variante_id'],
                                'user_id' => auth()->id(),
                                'empresa_id' => auth()->user()->empresa_id,
                                'sucursal_id' => auth()->user()->sucursal_id
                            ]);
                        }
                    } else {
                        $producto = Producto::find($item['producto_id']);
                        if ($producto) {
                            $stockAnterior = $producto->quantity;
                            $producto->decrement('quantity', $item['cantidad']);

                            InventoryMovement::create([
                                'producto_id' => $item['producto_id'],
                                'type' => 'salida',
                                'quantity' => $item['cantidad'],
                                'quantity_before' => $stockAnterior,
                                'quantity_after' => $producto->quantity,
                                'reference' => 'Venta POS #' . $venta->codigo,
                                'notes' => 'Venta ID: ' . $venta->id,
                                'user_id' => auth()->id(),
                                'empresa_id' => auth()->user()->empresa_id,
                                'sucursal_id' => auth()->user()->sucursal_id
                            ]);
                        }
                    }
                }

                VentaPago::create([
                    'venta_id' => $venta->id,
                    'metodo_pago' => $this->metodoPago,
                    'monto' => $this->getTotal(),
                    'monto_recibido' => $this->montoPagado,
                    'cambio' => $this->montoCambio,
                    'caja_id' => $this->cajaAbierta->id,
                    'confirmado' => true,
                ]);

                if ($nextNumber > $serie->correlativo_actual) {
                    $serie->update(['correlativo_actual' => $nextNumber]);
                }

                // Hardware
                if ($this->cashDrawerEnabled && $this->metodoPago === 'efectivo') {
                    $this->dispatch('open-cash-drawer');
                }

                if ($this->printerEnabled) {
                    $this->dispatch('print-ticket', ['ventaId' => $venta->id]);
                }
            });

            $this->cart = [];
            $this->closePaymentModal();
            $this->loadRecentSales();
            $this->dispatch('show-toast', ['type' => 'success', 'message' => 'Venta procesada exitosamente']);

        } catch (\Exception $e) {
            $this->dispatch('show-toast', ['type' => 'error', 'message' => 'Error al procesar venta: ' . $e->getMessage()]);
        }
    }

    public function openCustomerModal()
    {
        $this->showCustomerModal = true;
        $this->reset('newCustomer');
    }

    public function closeCustomerModal()
    {
        $this->showCustomerModal = false;
        $this->reset('newCustomer');
    }

    public function saveCustomer()
    {
        $this->validate([
            'newCustomer.name' => 'required|string|max:255',
            'newCustomer.email' => 'nullable|email|unique:users,email',
            'newCustomer.telefono' => 'nullable|string|max:20',
            'newCustomer.direccion' => 'nullable|string|max:500'
        ]);

        // Generate username from name initials
        $nameParts = explode(' ', trim($this->newCustomer['name']));
        $username = '';
        foreach ($nameParts as $part) {
            if (!empty($part)) {
                $username .= strtolower(substr($part, 0, 1));
            }
        }
        $username = $username ?: 'user';
        
        // Ensure unique username
        $originalUsername = $username;
        $counter = 1;
        while (\App\Models\User::where('username', $username)->exists()) {
            $username = $originalUsername . $counter;
            $counter++;
        }

        $customer = \App\Models\User::create([
            'name' => $this->newCustomer['name'],
            'username' => $username,
            'email' => $this->newCustomer['email'] ?: null,
            'telefono' => $this->newCustomer['telefono'] ?: null,
            'direccion' => $this->newCustomer['direccion'] ?: null,
            'empresa_id' => auth()->user()->empresa_id,
            'sucursal_id' => auth()->user()->sucursal_id,
            'password' => bcrypt('123456'),
            'email_verified_at' => now()
        ]);

        // Create record in clientes table
        $nameParts = explode(' ', trim($this->newCustomer['name']));
        $nombres = $nameParts[0] ?? '';
        $apellidos = isset($nameParts[1]) ? implode(' ', array_slice($nameParts, 1)) : '';
        
        \App\Models\Cliente::create([
            'nombres' => $nombres,
            'apellidos' => $apellidos,
            'email' => $this->newCustomer['email'] ?: null,
            'phone' => $this->newCustomer['telefono'] ?: null,
            'direccion' => $this->newCustomer['direccion'] ?: null,
            'empresa_id' => auth()->user()->empresa_id,
            'sucursal_id' => auth()->user()->sucursal_id,
            'user_id' => $customer->id
        ]);

        $this->cliente_id = $customer->id;
        $this->loadData(); // Refresh customers list
        $this->closeCustomerModal();
        $this->dispatch('show-toast', ['type' => 'success', 'message' => 'Cliente registrado y seleccionado']);
    }

    public function render()
    {
        return view('livewire.admin.pos.index')->layout('components.layouts.pos');
    }
}
