<?php

namespace App\Livewire\Admin\Pedidos;

use App\Models\Pedido;
use App\Models\Empleado;
use App\Services\PedidoService;
use App\Traits\HasDynamicLayout;
use Livewire\Component;
use Livewire\WithPagination;

class Index extends Component
{
    use WithPagination, HasDynamicLayout;

    public $search = '';
    public $estadoFilter = '';
    public $metodoPagoFilter = '';
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';
    public $perPage = 10;

    // Modal properties
    public $showAsignarModal = false;
    public $showCancelarModal = false;
    public $pedidoSeleccionado = null;
    public $empleadoSeleccionado = null;
    public $motivoCancelacion = '';

    protected $listeners = ['pedidoActualizado' => '$refresh'];

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function sortBy($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
    }

    public function confirmarPago($pedidoId)
    {
        $pedido = Pedido::findOrFail($pedidoId);
        $pedidoService = app(PedidoService::class);

        $pedidoService->confirmarPago($pedido);

        session()->flash('message', 'Pago confirmado y notificaci贸n enviada al cliente.');
        $this->dispatch('pedidoActualizado');
    }

    public function abrirModalAsignar($pedidoId)
    {
        $this->pedidoSeleccionado = $pedidoId;
        $this->empleadoSeleccionado = null;
        $this->showAsignarModal = true;
    }

    public function asignarEmpleado()
    {
        $this->validate([
            'empleadoSeleccionado' => 'required|exists:empleados,id'
        ]);

        $pedido = Pedido::findOrFail($this->pedidoSeleccionado);
        $empleado = Empleado::findOrFail($this->empleadoSeleccionado);
        $pedidoService = app(PedidoService::class);

        $pedidoService->asignarEmpleado($pedido, $empleado);

        // Procesar Jobs inmediatamente
        \Artisan::call('queue:work', ['--once' => true, '--quiet' => true]);

        $this->showAsignarModal = false;
        session()->flash('message', 'Pedido asignado y notificación enviada al empleado.');
        $this->dispatch('pedidoActualizado');
    }

    public function abrirModalCancelar($pedidoId)
    {
        $this->pedidoSeleccionado = $pedidoId;
        $this->motivoCancelacion = '';
        $this->showCancelarModal = true;
    }

    public function cancelarPedido()
    {
        $this->validate([
            'motivoCancelacion' => 'required|string|min:10'
        ]);

        $pedido = Pedido::findOrFail($this->pedidoSeleccionado);
        $pedidoService = app(PedidoService::class);

        $pedidoService->cancelarPedido($pedido, $this->motivoCancelacion);

        $this->showCancelarModal = false;
        session()->flash('message', 'Pedido cancelado y notificaci贸n enviada al cliente.');
        $this->dispatch('pedidoActualizado');
    }

    public function finalizarEntrega($pedidoId)
    {
        $pedido = Pedido::findOrFail($pedidoId);
        $pedidoService = app(PedidoService::class);

        $pedidoService->finalizarEntrega($pedido);

        session()->flash('message', 'Entrega finalizada exitosamente.');
        $this->dispatch('pedidoActualizado');
    }

    public function revertirPedido($pedidoId)
    {
        $pedido = Pedido::findOrFail($pedidoId);
        $pedidoService = app(PedidoService::class);

        $pedidoService->revertirPedido($pedido);

        session()->flash('message', 'Pedido revertido exitosamente. Los productos han sido devueltos al almac茅n.');
        $this->dispatch('pedidoActualizado');
    }

    public function render()
    {
        $pedidos = Pedido::with(['user', 'detalles.producto', 'empleados'])
            ->when($this->search, function ($query) {
                $query->where(function ($q) {
                    $q->where('codigo', 'like', '%' . $this->search . '%')
                      ->orWhereHas('user', function ($userQuery) {
                          $userQuery->where('name', 'like', '%' . $this->search . '%');
                      });
                });
            })
            ->when($this->estadoFilter, function ($query) {
                $query->where('estado', $this->estadoFilter);
            })
            ->when($this->metodoPagoFilter, function ($query) {
                $query->where('metodo_pago', $this->metodoPagoFilter);
            })
            ->orderBy($this->sortBy, $this->sortDirection)
            ->paginate($this->perPage);

        $empleados = Empleado::where('status', true)->get();

        $stats = [
            'total' => Pedido::count(),
            'pendientes' => Pedido::where('estado', 'pendiente')->count(),
            'pagados' => Pedido::where('estado', 'pagado')->count(),
            'entregados' => Pedido::where('estado', 'entregado')->count(),
        ];

        return $this->renderWithLayout('livewire.admin.pedidos.index', compact('pedidos', 'empleados', 'stats'));
    }
}
