<?php

namespace App\Livewire\Admin\Pedidos;

use App\Models\Pedido;
use App\Models\Empleado;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;
use Livewire\WithPagination;

class Historico extends Component
{
    use HasDynamicLayout, WithPagination;

    public $search = '';
    public $estado = '';
    public $fechaDesde = '';
    public $fechaHasta = '';
    public $perPage = 15;
    public $sortBy = 'created_at';
    public $sortDirection = 'desc';
    public $userId;
    public $empleado;
    public $stats = [];

    protected $queryString = ['search', 'estado', 'fechaDesde', 'fechaHasta', 'perPage'];

    public function mount($userId = null)
    {
        $this->userId = $userId ?: Auth::id();
        $this->empleado = Empleado::where('user_id', $this->userId)->first();
        $this->fechaDesde = now()->subMonth()->format('Y-m-d');
        $this->fechaHasta = now()->format('Y-m-d');
        $this->calculateStats();
    }

    public function updatingSearch() { $this->resetPage(); }
    public function updatingEstado() { $this->resetPage(); }
    public function updatingFechaDesde() { $this->resetPage(); $this->calculateStats(); }
    public function updatingFechaHasta() { $this->resetPage(); $this->calculateStats(); }

    public function sortBy($field)
    {
        if ($this->sortBy === $field) {
            $this->sortDirection = $this->sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            $this->sortBy = $field;
            $this->sortDirection = 'asc';
        }
        $this->resetPage();
    }

    public function clearFilters()
    {
        $this->search = '';
        $this->estado = '';
        $this->fechaDesde = now()->subMonth()->format('Y-m-d');
        $this->fechaHasta = now()->format('Y-m-d');
        $this->resetPage();
        $this->calculateStats();
    }

    public function calculateStats()
    {
        if (!$this->empleado) return;
        
        $query = $this->getBaseQuery();
        
        $this->stats = [
            'total' => $query->count(),
            'pendientes' => (clone $query)->where('estado', 'Pendiente')->count(),
            'procesando' => (clone $query)->where('estado', 'Procesando')->count(),
            'entregados' => (clone $query)->where('estado', 'entregado')->count(),
            'total_ingresos' => (clone $query)->where('estado', 'entregado')->sum('total_usd')
        ];
    }

    private function getBaseQuery()
    {
        $query = Pedido::where('empresaId', Auth::user()->empresa_id)
            ->whereHas('empleados', fn($q) => $q->where('empleadoId', $this->empleado->id));
            
        if ($this->fechaDesde) $query->whereDate('created_at', '>=', $this->fechaDesde);
        if ($this->fechaHasta) $query->whereDate('created_at', '<=', $this->fechaHasta);
        
        return $query;
    }

    public function updateOrderStatus($orderId, $status)
    {
        $pedido = Pedido::find($orderId);
        if ($pedido && $pedido->empleados->contains('id', $this->empleado->id)) {
            $pedido->update(['estado' => $status]);
            $this->calculateStats();
            
            $this->dispatch('alert', [
                'title' => '¡Actualizado!',
                'text' => "Pedido #{$pedido->codigo} actualizado a {$status}",
                'icon' => 'success'
            ]);
        }
    }

    public function render()
    {
        if (!$this->empleado) {
            return view('livewire.admin.pedidos.historico', ['pedidos' => collect()])
                ->layout($this->getLayout(), [
                    'title' => 'Historial de Pedidos',
                    'breadcrumb' => [
                        ['name' => 'Mi Dashboard', 'url' => route('admin.empleados.dashboard')],
                        ['name' => 'Mis Pedidos', 'active' => true]
                    ]
                ]);
        }

        $query = Pedido::where('empresaId', Auth::user()->empresa_id)
            ->whereHas('empleados', fn($q) => $q->where('empleadoId', $this->empleado->id))
            ->with(['user', 'detalles.producto']);

        if ($this->search) {
            $query->where(function($q) {
                $q->where('codigo', 'like', '%' . $this->search . '%')
                  ->orWhereHas('user', fn($userQuery) => 
                      $userQuery->where('name', 'like', '%' . $this->search . '%')
                  );
            });
        }

        if ($this->estado) $query->where('estado', $this->estado);
        if ($this->fechaDesde) $query->whereDate('created_at', '>=', $this->fechaDesde);
        if ($this->fechaHasta) $query->whereDate('created_at', '<=', $this->fechaHasta);

        $pedidos = $query->orderBy($this->sortBy, $this->sortDirection)->paginate($this->perPage);

        return view('livewire.admin.pedidos.historico', compact('pedidos'))
            ->layout($this->getLayout(), [
                'title' => 'Mis Pedidos - ' . ($this->empleado->nombre_completo ?? 'Empleado'),
                'breadcrumb' => [
                    ['name' => 'Mi Dashboard', 'url' => route('admin.empleados.dashboard')],
                    ['name' => 'Mis Pedidos', 'active' => true]
                ]
            ]);
    }
}