<?php

namespace App\Livewire\Admin\Pedidos;

use Livewire\Component;
use App\Models\Pedido;
use App\Models\Producto;
use App\Models\User;
use App\Traits\HasDynamicLayout;

class Create extends Component
{
    use HasDynamicLayout;

    public $cliente_id = '';
    public $clienteOption = 'existente'; // Por defecto mostrar cliente existente
    public $fecha_entrega = '';
    public $notas = '';

    // Datos del cliente
    public $nombres = '';
    public $apellidos = '';
    public $email = '';
    public $telefono = '';
    public $direccion = '';

    // Productos en el carrito
    public $cart = [];
    public $selectedProduct = '';
    public $quantity = 1;

    // Búsqueda de productos
    public $searchTerm = '';
    public $searchResults = [];
    public $showSearchResults = false;

    // Clientes y productos disponibles
    public $clientes = [];
    public $productos = [];

    protected $rules = [
        'cliente_id' => 'required_without_all:nombres,apellidos|exists:users,id',
        'fecha_entrega' => 'nullable|date|after_or_equal:today',
        'notas' => 'nullable|string|max:1000',
        'nombres' => 'required_without:cliente_id|string|max:255',
        'apellidos' => 'required_without:cliente_id|string|max:255',
        'email' => 'nullable|email|max:255',
        'telefono' => 'nullable|string|max:20',
        'direccion' => 'nullable|string|max:1000',
        'quantity' => 'integer|min:1|max:1000',
    ];

    protected $messages = [
        'cliente_id.required_without_all' => 'Debe seleccionar un cliente existente o crear uno nuevo.',
        'cliente_id.exists' => 'El cliente seleccionado no es válido.',
        'nombres.required_without' => 'El nombre es obligatorio para crear un nuevo cliente.',
        'apellidos.required_without' => 'Los apellidos son obligatorios para crear un nuevo cliente.',
        'fecha_entrega.after_or_equal' => 'La fecha de entrega debe ser hoy o posterior.',
        'quantity.min' => 'La cantidad mínima es 1.',
        'quantity.max' => 'La cantidad máxima es 1000.',
    ];

    public function mount()
    {
        $this->clientes = User::role('Cliente')
            ->where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->get()
            ->mapWithKeys(function ($user) {
                return [$user->id => $user->name];
            })
            ->toArray();
    }

    public function render()
    {
        return view('livewire.admin.pedidos.create')
            ->layout($this->getLayout(), [
                'title' => 'Crear Pedido',
                'breadcrumb' => [
                    ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                    ['name' => 'Pedidos', 'route' => 'admin.pedidos.index'],
                    ['name' => 'Crear Pedido', 'active' => true]
                ]
            ]);
    }

    public function updatedClienteId($value)
    {
        // Si se selecciona un cliente existente, limpiar campos de nuevo cliente
        if ($value) {
            $this->reset(['nombres', 'apellidos', 'email', 'telefono', 'direccion']);
        }
    }

    public function updatedClienteOption($value)
    {
        // Limpiar campos cuando se cambia entre existente/nuevo
        if ($value === 'existente') {
            $this->reset(['nombres', 'apellidos', 'email', 'telefono', 'direccion']);
        } else {
            $this->reset(['cliente_id']);
        }
    }

    public function searchProducts()
    {
        if (strlen($this->searchTerm) >= 2) {
            $query = Producto::where('status', true);
            
            // Buscar por código exacto primero
            $exactMatch = clone $query;
            $productos = $exactMatch->where('code', $this->searchTerm)->limit(5)->get();
            
            // Si no hay coincidencia exacta, buscar por coincidencias parciales
            if ($productos->isEmpty()) {
                $productos = $query->where(function($q) {
                    $q->where('name', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('description', 'like', '%' . $this->searchTerm . '%')
                      ->orWhere('code', 'like', '%' . $this->searchTerm . '%');
                })
                ->limit(8)
                ->get();
            }
            
            $this->searchResults = [];
            foreach ($productos as $producto) {
                $this->searchResults[] = [
                    'id' => $producto->id,
                    'name' => $producto->name,
                    'code' => $producto->code,
                    'description' => $producto->description,
                    'price' => (float) $producto->price,
                    'quantity' => $producto->quantity,
                    'image_path' => $producto->featuredImage ? $producto->featuredImage->path : null,
                    'categoria_name' => $producto->categoria ? $producto->categoria->name : null,
                    'marca_name' => $producto->marca ? $producto->marca->name : null,
                ];
            }
            
            $this->showSearchResults = true;
        } else {
            $this->searchResults = [];
            $this->showSearchResults = false;
        }
    }

    public function updatedSearchTerm()
    {
        $this->searchProducts();
    }

    public function selectProduct($productId)
    {
        $this->selectedProduct = $productId;
        $this->searchTerm = '';
        $this->showSearchResults = false;

        // Obtener el producto seleccionado para mostrar información
        $producto = Producto::find($productId);
        if ($producto) {
            $this->searchTerm = $producto->name;
        }
    }

    public function hideSearchResults()
    {
        // Pequeño delay para permitir clicks en los resultados
        $this->showSearchResults = false;
    }

    public function addProductToCart()
    {
        $this->validate([
            'quantity' => 'required|integer|min:1|max:1000'
        ]);

        if (!$this->selectedProduct || $this->quantity < 1) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Seleccione un producto y una cantidad válida.'
            ]);
            return;
        }

        $producto = Producto::find($this->selectedProduct);

        if (!$producto) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Producto no encontrado.'
            ]);
            return;
        }

        // Verificar stock
        if ($producto->quantity < $this->quantity) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'No hay suficiente stock disponible. Stock actual: ' . $producto->quantity
            ]);
            return;
        }

        // Verificar si el producto ya está en el carrito
        $existingItemIndex = collect($this->cart)->search(function ($item) use ($producto) {
            return $item['producto_id'] == $producto->id;
        });

        if ($existingItemIndex !== false) {
            // Verificar stock total antes de actualizar
            $newQuantity = $this->cart[$existingItemIndex]['cantidad'] + $this->quantity;
            if ($newQuantity > $producto->quantity) {
                $this->dispatch('show-toast', [
                    'type' => 'error',
                    'message' => 'No hay suficiente stock. Ya tienes ' . $this->cart[$existingItemIndex]['cantidad'] . ' en el carrito. Stock disponible: ' . $producto->quantity
                ]);
                return;
            }
            
            // Actualizar cantidad si ya existe
            $this->cart[$existingItemIndex]['cantidad'] = $newQuantity;
            $this->cart[$existingItemIndex]['subtotal'] = $newQuantity * $this->cart[$existingItemIndex]['precio_unitario'];
        } else {
            // Agregar nuevo producto al carrito
            $this->cart[] = [
                'producto_id' => $producto->id,
                'nombre' => $producto->name,
                'codigo' => $producto->code,
                'cantidad' => $this->quantity,
                'precio_unitario' => (float) $producto->price,
                'subtotal' => $this->quantity * (float) $producto->price
            ];
        }

        // Resetear campos
        $this->selectedProduct = '';
        $this->quantity = 1;
        $this->searchTerm = '';
        $this->showSearchResults = false;

        $this->dispatch('product-added');
        $this->dispatch('show-toast', [
            'type' => 'success',
            'message' => 'Producto agregado al carrito correctamente.'
        ]);
    }

    public function removeFromCart($index)
    {
        if (isset($this->cart[$index])) {
            $productName = $this->cart[$index]['nombre'];
            unset($this->cart[$index]);
            $this->cart = array_values($this->cart); // Reindexar array
            
            $this->dispatch('product-removed');
            $this->dispatch('show-toast', [
                'type' => 'info',
                'message' => 'Producto "' . $productName . '" eliminado del carrito.'
            ]);
        }
    }

    public function getTotalProperty()
    {
        return collect($this->cart)->sum('subtotal');
    }

    public function save()
    {
        // Validar que haya productos en el carrito
        if (empty($this->cart)) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Debe agregar al menos un producto al pedido.'
            ]);
            return;
        }

        // Validar datos del pedido
        $this->validate();

        try {
            \DB::beginTransaction();

            // Crear cliente si es necesario
            if (!$this->cliente_id && $this->nombres && $this->apellidos) {
                $cliente = User::create([
                    'name' => trim($this->nombres . ' ' . $this->apellidos),
                    'email' => $this->email ?: null,
                    'telefono' => $this->telefono ?: null,
                    'password' => bcrypt('temporal123'), // Contraseña temporal
                    'empresa_id' => auth()->user()->empresa_id,
                    'sucursal_id' => auth()->user()->sucursal_id,
                    'email_verified_at' => now(),
                ]);

                // Asignar rol de Cliente
                $cliente->assignRole('Cliente');
                $this->cliente_id = $cliente->id;
            }

            // Verificar stock antes de crear el pedido
            foreach ($this->cart as $item) {
                $producto = Producto::find($item['producto_id']);
                if (!$producto || $producto->quantity < $item['cantidad']) {
                    throw new \Exception('Stock insuficiente para el producto: ' . ($producto->name ?? 'Desconocido'));
                }
            }

            // Obtener tasa de cambio actual
            $tasaCambio = \App\Models\ExchangeRate::whereDate('date', today())
                ->orderBy('fetch_time', 'desc')
                ->first();
            $usdRate = $tasaCambio ? $tasaCambio->usd_rate : 36.5;

            // Crear pedido
            $pedido = Pedido::create([
                'userId' => $this->cliente_id,
                'fecha_pedido' => now(),
                'fecha_entrega' => $this->fecha_entrega ?: null,
                'estado' => 'pendiente',
                'total_usd' => $this->total,
                'total_bs' => $this->total * $usdRate,
                'codigo' => 'PED-' . str_pad((Pedido::max('id') ?? 0) + 1, 6, '0', STR_PAD_LEFT),
                'metodo_pago' => 'pendiente',
                'notas' => $this->notas,
                'empresaId' => auth()->user()->empresa_id,
                'sucursalId' => auth()->user()->sucursal_id,
            ]);

            // Crear detalles del pedido y actualizar stock
            foreach ($this->cart as $item) {
                $pedido->detalles()->create([
                    'productoId' => $item['producto_id'],
                    'cantidad' => $item['cantidad'],
                    'precio_unitario' => $item['precio_unitario'],
                    'subtotal' => $item['subtotal'],
                    'empresa_id' => auth()->user()->empresa_id,
                    'sucursal_id' => auth()->user()->sucursal_id,
                ]);

                // Actualizar stock del producto
                $producto = Producto::find($item['producto_id']);
                if ($producto) {
                    $producto->decrement('quantity', $item['cantidad']);
                }
            }

            \DB::commit();

            session()->flash('message', 'Pedido #' . $pedido->codigo . ' creado correctamente.');
            session()->flash('message_type', 'success');

            return redirect()->route('admin.pedidos.show', $pedido);
            
        } catch (\Exception $e) {
            \DB::rollBack();
            
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al crear el pedido: ' . $e->getMessage()
            ]);
        }
    }
}
