<?php

namespace App\Livewire\Admin\NotasVenta;

use App\Models\NotaVenta;
use App\Models\Venta;
use App\Models\DetalleVenta;
use App\Models\Serie;
use App\Models\Caja;
use Livewire\Component;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;
use Illuminate\Support\Facades\DB;

class Show extends Component
{
    use HasDynamicLayout, HasRegionalFormatting;

    public NotaVenta $notaVenta;
    public $showConvertModal = false;
    public $metodo_pago = 'efectivo';
    public $observaciones_venta = '';

    public function mount(NotaVenta $notaVenta)
    {
        // Inicializar configuración regional
        $this->bootHasRegionalFormatting();

        $this->notaVenta = $notaVenta->load(['detalles.producto', 'detalles.productoVariant', 'cliente', 'user', 'empresa', 'sucursal', 'venta']);
    }

    public function abrirModalConvertir()
    {
        // Recargar el modelo para asegurar datos actualizados
        $this->notaVenta->refresh();
        //dd($this->notaVenta->venta_id);
        if ($this->notaVenta->venta_id) {
            session()->flash('error', 'Esta nota de venta ya fue convertida a venta');
            return;
        }

        // Verificar que hay caja abierta
        $cajaAbierta = Caja::obtenerCajaAbierta(
            auth()->user()->empresa_id,
            auth()->user()->sucursal_id
        );

        if (!$cajaAbierta) {
            session()->flash('error', 'Debe tener una caja abierta para crear ventas');
            return;
        }

        $this->showConvertModal = true;
    }

    public function cerrarModalConvertir()
    {
        $this->showConvertModal = false;
        $this->metodo_pago = 'efectivo';
        $this->observaciones_venta = '';
    }

    public function convertirAVenta()
    {
        // Validar que tenemos el modelo cargado

        if (!$this->notaVenta || !$this->notaVenta->id) {
            session()->flash('error', 'Error: Nota de venta no encontrada');
            return;
        }

        $this->validate([
            'metodo_pago' => 'required|in:efectivo,transferencia,tarjeta,pago_movil,punto_de_venta'
        ]);

        try {
            DB::transaction(function () {
                // Obtener caja abierta
                $caja = Caja::obtenerCajaAbierta(
                    auth()->user()->empresa_id,
                    auth()->user()->sucursal_id
                );

                // Obtener serie para ventas
                $serie = Serie::where('tipo_documento', 'factura')
                    ->where('activo', true)
                    ->where('empresa_id', auth()->user()->empresa_id)
                    ->where('sucursal_id', auth()->user()->sucursal_id)
                    ->first();

                if (!$serie) {
                    throw new \Exception('No se encontró una serie activa para facturas');
                }

                // Generar código de venta
                $codigo = 'VEN-' . str_pad($serie->correlativo_actual + 1, 6, '0', STR_PAD_LEFT);

                // Crear venta
                $venta = Venta::create([
                    'codigo' => $codigo,
                    'total' => $this->notaVenta->total,
                    'total_usd' => $this->notaVenta->total,
                    'total_bs' => $this->notaVenta->total,
                    'metodo_pago' => $this->metodo_pago,
                    'estado' => 'pagado',
                    'tipo_documento' => 'factura',
                    'numero_documento' => $serie->serie . '-' . str_pad($serie->correlativo_actual + 1, 8, '0', STR_PAD_LEFT),
                    'serie_id' => $serie->id,
                    'caja_id' => $caja->id,
                    'empresa_id' => $this->notaVenta->empresa_id,
                    'sucursal_id' => $this->notaVenta->sucursal_id,
                    'user_id' => auth()->id(),
                    'empleado_id' => auth()->id(),
                    'cliente_id' => $this->notaVenta->cliente_id,
                    'descuento' => 0,
                    'impuesto' => $this->notaVenta->impuestos,
                    'observaciones' => $this->observaciones_venta ?: 'Generada desde nota de venta ' . $this->notaVenta->numero_completo,
                    'nota_venta_id' => $this->notaVenta->id
                ]);

                // Crear detalles de venta y actualizar inventario
                foreach ($this->notaVenta->detalles as $detalle) {
                    // Verificar stock disponible
                    if ($detalle->producto) {
                        if ($detalle->producto->quantity < $detalle->cantidad) {
                            throw new \Exception("Stock insuficiente para el producto: {$detalle->producto->name}. Stock disponible: {$detalle->producto->quantity}");
                        }
                    }

                    DetalleVenta::create([
                        'venta_id' => $venta->id,
                        'producto_id' => $detalle->producto_id,
                        'variante_id' => $detalle->producto_variant_id,
                        'nombre_producto' => $detalle->descripcion,
                        'codigo_producto' => $detalle->producto ? $detalle->producto->code : 'N/A',
                        'cantidad' => $detalle->cantidad,
                        'precio_unitario' => $detalle->precio_unitario,
                        'subtotal' => $detalle->subtotal
                    ]);

                    // Crear pago
                    $venta->pagos()->create([
                        'metodo_pago' => $this->metodo_pago,
                        'monto' => $this->notaVenta->total,
                        'monto_recibido' => $this->notaVenta->total,
                        'cambio' => 0,
                        'caja_id' => $caja->id,
                        'confirmado' => true,
                    ]);

                    // Actualizar stock del producto principal
                    if ($detalle->producto) {
                        $detalle->producto->decrement('quantity', $detalle->cantidad);

                        // Registrar movimiento de inventario
                        \App\Models\InventoryMovement::create([
                            'producto_id' => $detalle->producto_id,
                            'type' => 'salida',
                            'quantity' => $detalle->cantidad,
                            'quantity_before' => $detalle->producto->quantity + $detalle->cantidad,
                            'quantity_after' => $detalle->producto->quantity,
                            'reference' => 'Venta desde nota de venta #' . $this->notaVenta->numero_completo,
                            'notes' => 'Venta ID: ' . $venta->id,
                            'user_id' => auth()->id(),
                            'empresa_id' => auth()->user()->empresa_id,
                            'sucursal_id' => auth()->user()->sucursal_id
                        ]);
                    }

                    // Actualizar stock de variante si existe
                    if ($detalle->productoVariant) {
                        $variantStock = $detalle->productoVariant->quantity ?? 0;
                        if ($variantStock < $detalle->cantidad) {
                            throw new \Exception("Stock insuficiente para la variante: {$detalle->productoVariant->name}. Stock disponible: {$variantStock}");
                        }
                        $detalle->productoVariant->decrement('quantity', $detalle->cantidad);
                    }
                }

                // Actualizar serie
                $serie->increment('correlativo_actual');

                // Vincular venta a nota de venta
                $this->notaVenta->update([
                    'venta_id' => $venta->id,
                    'estado' => 'procesada'
                ]);

                // Recalcular totales de caja
                $caja->calcularTotales();
            });

            $this->cerrarModalConvertir();

            // Recargar la nota de venta para mostrar el estado actualizado
            $this->notaVenta->refresh();
            $this->notaVenta->load(['detalles.producto', 'detalles.productoVariant', 'cliente', 'user', 'empresa', 'sucursal', 'venta']);

            session()->flash('message', 'Nota de venta convertida a venta exitosamente');

        } catch (\Exception $e) {
            session()->flash('error', 'Error al convertir a venta: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $metodosPago = [
            'efectivo' => 'Efectivo',
            'transferencia' => 'Transferencia',
            'tarjeta' => 'Tarjeta',
            'pago_movil' => 'Pago Móvil',
            'punto_de_venta' => 'Punto de Venta'
        ];

        return view('livewire.admin.notas-venta.show', [
            'metodosPago' => $metodosPago
        ])->layout($this->getLayout());
    }
}
