<?php

namespace App\Livewire\Admin\NotasCredito;

use Livewire\Component;
use App\Models\NotaCredito;
use App\Models\DetalleNotaCredito;
use App\Models\Cliente;
use App\Models\Serie;
use App\Models\Venta;
use App\Models\DetalleVenta;
use App\Traits\HasRegionalFormatting;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\DB;

class Create extends Component
{
    use HasRegionalFormatting, HasDynamicLayout;

    public $serie = 'NC01';
    public $numero;
    public $fecha_emision;
    public $cliente_id;
    public $motivo = 'Anulación de la operación';
    public $tipo_nota_credito = '1';
    public $venta_id;
    public $venta;
    public $detalles = [];
    public $subtotal = 0;
    public $impuestos = 0;
    public $total = 0;

    protected $queryString = ['venta_id'];

    public function mount($venta_id = null)
    {
        $this->fecha_emision = now()->format('Y-m-d');
        $this->generateNumber();

        if ($venta_id) {
            $this->venta_id = $venta_id;
            $this->loadVentaData();
        }
    }

    public function loadVentaData()
    {
        $this->venta = Venta::with(['detalles.producto', 'detalles.variante'])
            ->find($this->venta_id);

        if ($this->venta) {
            // Si cliente_id existe, usarlo, sino buscar el cliente por user_id
            if ($this->venta->cliente_id) {
                $cliente = Cliente::find($this->venta->cliente_id);
                if ($cliente) {
                    $this->cliente_id = $cliente->id;
                }
            }

            $this->subtotal = $this->venta->total - ($this->venta->impuesto ?? 0);
            $this->impuestos = $this->venta->impuesto ?? 0;
            $this->total = $this->venta->total;

            // Cargar detalles de la venta
            foreach ($this->venta->detalles as $detalle) {
                $descripcion = $detalle->nombre_producto ??
                              ($detalle->producto ? $detalle->producto->nombre : 'Producto');

                $this->detalles[] = [
                    'producto_id' => $detalle->producto_id,
                    'producto_variant_id' => $detalle->variante_id,
                    'descripcion' => $descripcion,
                    'cantidad' => $detalle->cantidad,
                    'precio_unitario' => $detalle->precio_unitario,
                    'subtotal' => $detalle->subtotal ?? ($detalle->cantidad * $detalle->precio_unitario)
                ];
            }
        }
    }

    public function generateNumber()
    {
        $serie = Serie::where('tipo_documento', 'nota_credito')
            ->where('serie', $this->serie)
            ->where('activo', true)
            ->first();

        if ($serie) {
            $this->numero = str_pad($serie->correlativo_actual + 1, 8, '0', STR_PAD_LEFT);
        }
    }

    public function save()
    {
        logger('Save method called');

        try {
            $this->validate([
                'cliente_id' => 'nullable|exists:clientes,id',
                'motivo' => 'required|string|max:500',
                'tipo_nota_credito' => 'required|string|max:2'
            ]);

            logger('Validation passed');

            DB::transaction(function () {
                // Crear la nota de crédito
                $notaCredito = NotaCredito::create([
                    'serie' => $this->serie,
                    'numero' => $this->numero,
                    'fecha_emision' => $this->fecha_emision,
                    'cliente_id' => $this->cliente_id,
                    'motivo' => $this->motivo,
                    'tipo_nota_credito' => $this->tipo_nota_credito,
                    'subtotal' => $this->subtotal,
                    'impuestos' => $this->impuestos,
                    'total' => $this->total,
                    'venta_id' => $this->venta_id,
                    'empresa_id' => auth()->user()->empresa_id,
                    'sucursal_id' => auth()->user()->sucursal_id,
                    'user_id' => auth()->id()
                ]);

                logger('NotaCredito created: ' . $notaCredito->id);

                // Crear los detalles
                foreach ($this->detalles as $detalle) {
                    DetalleNotaCredito::create([
                        'nota_credito_id' => $notaCredito->id,
                        'producto_id' => $detalle['producto_id'],
                        'producto_variant_id' => $detalle['producto_variant_id'],
                        'descripcion' => $detalle['descripcion'],
                        'cantidad' => $detalle['cantidad'],
                        'precio_unitario' => $detalle['precio_unitario'],
                        'subtotal' => $detalle['subtotal']
                    ]);
                }

                // Actualizar el número de serie
                $serie = Serie::where('tipo_documento', 'nota_credito')
                    ->where('serie', $this->serie)
                    ->where('activo', true)
                    ->first();

                if ($serie) {
                    $serie->increment('correlativo_actual');
                }

                // Marcar la venta como cancelada si es anulación total
                if ($this->venta_id && $this->tipo_nota_credito === '1') {
                    $this->venta->update(['estado' => 'cancelado']);
                }
            });

            logger('Transaction completed');

            $this->dispatch('show-message', [
                'type' => 'success',
                'message' => 'Nota de crédito creada exitosamente.'
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            logger('Validation failed: ' . json_encode($e->errors()));
            // Los errores de validación se muestran automáticamente en el formulario
            return;
        } catch (\Exception $e) {
            logger('Exception: ' . $e->getMessage());
            $this->dispatch('show-message', [
                'type' => 'error',
                'message' => 'Error al crear la nota de crédito: ' . $e->getMessage()
            ]);
        }
    }

    public function testSave()
    {
        session()->flash('message', 'Test method called successfully!');
        $this->dispatch('show-message', [
            'type' => 'success',
            'message' => 'Test method working!'
        ]);
    }

    public function render()
    {
        $clientes = Cliente::orderBy('nombres')->get();
        $tiposNotaCredito = NotaCredito::getTiposNotaCredito();

        return view('livewire.admin.notas-credito.create', [
            'clientes' => $clientes,
            'tiposNotaCredito' => $tiposNotaCredito
        ])->layout($this->getLayout());
    }
}
