<?php

namespace App\Livewire\Admin\Monitoreo;

use Livewire\Component;
use Livewire\WithFileUploads;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;

class DatabaseImport extends Component
{
    use HasDynamicLayout, WithFileUploads;

    public $sqlFile;
    public $filterFromDate;
    public $filterToDate;
    public $useDataFilter = false;
    public $selectedTables = [];
    public $availableTables = [];
    public $isProcessing = false;
    public $previewData = [];
    public $showPreview = false;
    public $conflictData = [];
    public $handleIdConflicts = true;
    public $conflictResolution = 'update'; // 'auto_increment', 'skip', 'replace', 'update'
    public $tempSqlPath = null;

    public function mount()
    {
        $this->filterFromDate = now()->format('Y-m-d');
        $this->filterToDate = now()->format('Y-m-d');
    }

    public function updatedSqlFile()
    {
        $this->validate([
            'sqlFile' => 'required|file|mimes:sql|max:51200' // 50MB max
        ]);

        $this->analyzeSqlFile();
    }

    public function analyzeSqlFile()
    {
        if (!$this->sqlFile) return;

        try {
            $content = file_get_contents($this->sqlFile->getRealPath());
            $this->availableTables = $this->extractTablesFromSql($content);
            $this->selectedTables = array_keys($this->availableTables);

            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => 'Archivo SQL analizado. ' . count($this->availableTables) . ' tablas encontradas.',
                'timeout' => 5000
            ]);
        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al analizar el archivo: ' . $e->getMessage(),
                'timeout' => 8000
            ]);
        }
    }

    private function extractTablesFromSql($content)
    {
        $tables = [];

        // Buscar CREATE TABLE statements
        preg_match_all('/CREATE TABLE.*?`([^`]+)`/i', $content, $matches);

        foreach ($matches[1] as $tableName) {
            // Contar INSERT statements para esta tabla (sin filtro)
            $insertCount = preg_match_all('/INSERT INTO.*?`?' . preg_quote($tableName) . '`?/i', $content);
            $tables[$tableName] = [
                'name' => $tableName,
                'insert_count' => $insertCount,
                'total_inserts' => $insertCount
            ];
        }

        return $tables;
    }

    public function toggleTable($tableName)
    {
        if (in_array($tableName, $this->selectedTables)) {
            $this->selectedTables = array_filter($this->selectedTables, fn($t) => $t !== $tableName);
        } else {
            $this->selectedTables[] = $tableName;
        }
    }

    public function selectAllTables()
    {
        $this->selectedTables = array_keys($this->availableTables);
    }

    public function deselectAllTables()
    {
        $this->selectedTables = [];
    }

    public function previewImport()
    {
        $this->validate([
            'sqlFile' => 'required',
            'selectedTables' => 'required|array|min:1'
        ]);

        $this->isProcessing = true;

        try {
            $content = file_get_contents($this->sqlFile->getRealPath());
            
            $this->tempSqlPath = $this->createTempFilteredSql($content);
            
            $this->conflictData = $this->analyzeConflicts(file_get_contents($this->tempSqlPath));
            $this->showPreview = true;

            $this->dispatch('show-toast', [
                'type' => 'info',
                'message' => 'Previsualización generada. Revise los cambios antes de importar.',
                'timeout' => 5000
            ]);

        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al generar previsualización: ' . $e->getMessage(),
                'timeout' => 8000
            ]);
        } finally {
            $this->isProcessing = false;
        }
    }

    public function confirmImport()
    {
        $this->isProcessing = true;

        try {
            if (!$this->tempSqlPath || !file_exists($this->tempSqlPath)) {
                throw new \Exception('Archivo temporal no encontrado. Genere la previsualización primero.');
            }
            
            $filteredSql = file_get_contents($this->tempSqlPath);
            
            if (empty($filteredSql)) {
                throw new \Exception('No hay datos para importar con los filtros seleccionados');
            }
            
            DB::unprepared($filteredSql);

            $this->showPreview = false;
            $this->conflictData = [];
            $this->cleanupTempFile();

            $this->dispatch('show-toast', [
                'type' => 'success',
                'message' => 'Base de datos importada exitosamente',
                'timeout' => 5000
            ]);

        } catch (\Exception $e) {
            $this->dispatch('show-toast', [
                'type' => 'error',
                'message' => 'Error al importar: ' . $e->getMessage(),
                'timeout' => 8000
            ]);
        } finally {
            $this->isProcessing = false;
        }
    }

    public function cancelPreview()
    {
        $this->showPreview = false;
        $this->conflictData = [];
        $this->cleanupTempFile();
    }

    private function importComplete($content)
    {
        // Procesar solo las tablas seleccionadas
        foreach ($this->selectedTables as $tableName) {
            $this->importTableSafely($content, $tableName);
        }
    }

    private function importTableSafely($content, $tableName)
    {
        // Solo crear tabla si no existe y está en las tablas seleccionadas
        if (!DB::getSchemaBuilder()->hasTable($tableName)) {
            $createPattern = '/CREATE TABLE.*?`' . preg_quote($tableName) . '`.*?;/is';
            if (preg_match($createPattern, $content, $createMatch)) {
                // Usar CREATE TABLE IF NOT EXISTS para evitar errores
                $createStatement = str_replace('CREATE TABLE', 'CREATE TABLE IF NOT EXISTS', $createMatch[0]);
                DB::unprepared($createStatement);
            }
        }

        // Extraer datos usando método seguro (con filtros aplicados)
        $data = $this->extractFilteredTableDataFromSql($content, $tableName);

        if (!empty($data)) {
            $this->insertDataSafely($tableName, $data);
        }
    }

    private function extractFilteredTableDataFromSql($content, $tableName)
    {
        // Verificar que la tabla esté seleccionada
        if (!in_array($tableName, $this->selectedTables)) {
            return [];
        }

        $pattern = '/INSERT INTO\s+`?' . preg_quote($tableName) . '`?[^;]*;/is';
        preg_match_all($pattern, $content, $matches);

        $allData = [];

        foreach ($matches[0] as $insertStatement) {
            // Aplicar filtro de fecha si está habilitado
            if (!$this->matchesDateFilter($insertStatement)) {
                continue;
            }
            
            $rowData = $this->parseInsertStatement($insertStatement, $tableName);
            if (!empty($rowData)) {
                $allData = array_merge($allData, $rowData);
            }
        }

        return $allData;
    }

    private function extractTableDataFromSql($content, $tableName)
    {
        // Buscar INSERT statements para esta tabla
        $pattern = '/INSERT INTO\s+`?' . preg_quote($tableName) . '`?[^;]*;/is';
        preg_match_all($pattern, $content, $matches);

        $allData = [];

        foreach ($matches[0] as $insertStatement) {
            $rowData = $this->parseInsertStatement($insertStatement, $tableName);
            if (!empty($rowData)) {
                $allData = array_merge($allData, $rowData);
            }
        }

        return $allData;
    }

    private function parseInsertStatement($insertStatement, $tableName)
    {
        // Obtener columnas de la tabla actual
        $tableColumns = $this->getTableColumns($tableName);

        // Extraer columnas del INSERT (si las tiene)
        $insertColumns = [];
        if (preg_match('/INSERT INTO[^(]*\(([^)]+)\)/i', $insertStatement, $colMatches)) {
            $insertColumns = array_map('trim', explode(',', str_replace('`', '', $colMatches[1])));
        } else {
            // Si no especifica columnas, usar todas las de la tabla
            $insertColumns = $tableColumns;
        }

        // Extraer valores
        if (preg_match('/VALUES\s*(.+)$/is', $insertStatement, $valueMatches)) {
            $valuesString = trim($valueMatches[1]);
            $rows = $this->parseValueRows($valuesString);

            $result = [];
            foreach ($rows as $row) {
                if (count($row) === count($insertColumns)) {
                    // Mapear valores a columnas de la tabla
                    $mappedRow = [];
                    foreach ($tableColumns as $tableCol) {
                        $insertIndex = array_search($tableCol, $insertColumns);
                        $mappedRow[$tableCol] = $insertIndex !== false ? $row[$insertIndex] : null;
                    }
                    $result[] = $mappedRow;
                }
            }

            return $result;
        }

        return [];
    }

    private function parseValueRows($valuesString)
    {
        $rows = [];
        $currentRow = [];
        $currentValue = '';
        $inQuotes = false;
        $quoteChar = null;
        $parenLevel = 0;

        for ($i = 0; $i < strlen($valuesString); $i++) {
            $char = $valuesString[$i];
            $prevChar = $i > 0 ? $valuesString[$i-1] : '';

            if (!$inQuotes && ($char === "'" || $char === '"')) {
                $inQuotes = true;
                $quoteChar = $char;
                $currentValue .= $char;
            } elseif ($inQuotes && $char === $quoteChar && $prevChar !== '\\') {
                $inQuotes = false;
                $quoteChar = null;
                $currentValue .= $char;
            } elseif (!$inQuotes) {
                if ($char === '(') {
                    $parenLevel++;
                    if ($parenLevel === 1) {
                        continue; // Omitir el paréntesis de apertura
                    }
                } elseif ($char === ')') {
                    $parenLevel--;
                    if ($parenLevel === 0) {
                        // Fin de fila
                        if ($currentValue !== '') {
                            $currentRow[] = $this->cleanValue($currentValue);
                        }
                        if (!empty($currentRow)) {
                            $rows[] = $currentRow;
                        }
                        $currentRow = [];
                        $currentValue = '';
                        continue;
                    }
                } elseif ($char === ',' && $parenLevel === 1) {
                    // Separador de valores
                    $currentRow[] = $this->cleanValue($currentValue);
                    $currentValue = '';
                    continue;
                }
            }

            if ($parenLevel > 0) {
                $currentValue .= $char;
            }
        }

        return $rows;
    }

    private function insertDataSafely($tableName, $data)
    {
        foreach ($data as $row) {
            try {
                // Usar updateOrInsert por defecto para manejar duplicados automáticamente
                if (isset($row['id']) && !empty($row['id'])) {
                    DB::table($tableName)->updateOrInsert(['id' => $row['id']], $row);
                } else {
                    DB::table($tableName)->insert($row);
                }
            } catch (\Exception $e) {
                \Log::warning("Error inserting row in {$tableName}: " . $e->getMessage());
            }
        }
    }

    private function handleIdConflictForRow($tableName, $row)
    {
        $id = $row['id'];

        switch ($this->conflictResolution) {
            case 'auto_increment':
                unset($row['id']);
                DB::table($tableName)->insert($row);
                break;
            case 'skip':
                if (!DB::table($tableName)->where('id', $id)->exists()) {
                    DB::table($tableName)->insert($row);
                }
                break;
            case 'replace':
                DB::table($tableName)->where('id', $id)->delete();
                DB::table($tableName)->insert($row);
                break;
            case 'update':
            default:
                // Actualizar si existe, insertar si no existe
                DB::table($tableName)->updateOrInsert(['id' => $id], $row);
                break;
        }
    }

    private function importTableWithConflictResolution($content, $tableName)
    {
        // Extraer CREATE TABLE si es necesario
        if (!DB::getSchemaBuilder()->hasTable($tableName)) {
            $createPattern = '/CREATE TABLE.*?`' . preg_quote($tableName) . '`.*?;/is';
            if (preg_match($createPattern, $content, $createMatch)) {
                DB::unprepared($createMatch[0]);
            }
        }

        // Procesar INSERT statements con mejor parsing
        $this->processTableInserts($content, $tableName);
    }

    private function processTableInserts($content, $tableName)
    {
        // Buscar el patrón de INSERT para esta tabla
        $pattern = '/INSERT INTO\s+`?' . preg_quote($tableName) . '`?\s*(?:\([^)]+\))?\s*VALUES\s*([^;]+);/is';

        if (preg_match_all($pattern, $content, $matches)) {
            foreach ($matches[0] as $fullMatch) {
                try {
                    // Procesar cada INSERT completo
                    $this->processSingleInsert($fullMatch, $tableName);
                } catch (\Exception $e) {
                    // Si falla, intentar con método alternativo
                    $this->processInsertAlternative($fullMatch, $tableName);
                }
            }
        }
    }

    private function processSingleInsert($insertStatement, $tableName)
    {
        if ($this->handleIdConflicts) {
            $this->processInsertWithConflictResolution($insertStatement, $tableName);
        } else {
            DB::unprepared($insertStatement);
        }
    }

    private function processInsertAlternative($insertStatement, $tableName)
    {
        // Método alternativo: usar Eloquent para insertar
        $values = $this->extractValuesFromInsert($insertStatement);
        if (!empty($values)) {
            $columns = $this->getTableColumns($tableName);

            foreach ($values as $valueSet) {
                if (count($valueSet) === count($columns)) {
                    $data = array_combine($columns, $valueSet);
                    DB::table($tableName)->insert($data);
                }
            }
        }
    }

    private function extractValuesFromInsert($insertStatement)
    {
        // Extraer la parte VALUES
        if (preg_match('/VALUES\s*(.+)$/is', $insertStatement, $matches)) {
            $valuesString = trim($matches[1]);

            // Dividir por filas de valores
            $rows = [];
            $currentRow = '';
            $parenLevel = 0;
            $inQuotes = false;
            $quoteChar = null;

            for ($i = 0; $i < strlen($valuesString); $i++) {
                $char = $valuesString[$i];

                if (!$inQuotes && ($char === "'" || $char === '"')) {
                    $inQuotes = true;
                    $quoteChar = $char;
                } elseif ($inQuotes && $char === $quoteChar && ($i === 0 || $valuesString[$i-1] !== '\\')) {
                    $inQuotes = false;
                    $quoteChar = null;
                } elseif (!$inQuotes) {
                    if ($char === '(') {
                        $parenLevel++;
                    } elseif ($char === ')') {
                        $parenLevel--;
                        if ($parenLevel === 0) {
                            $currentRow .= $char;
                            $rows[] = $this->parseRowValues(trim($currentRow, '()'));
                            $currentRow = '';
                            continue;
                        }
                    }
                }

                if ($parenLevel > 0) {
                    $currentRow .= $char;
                }
            }

            return $rows;
        }

        return [];
    }

    private function parseRowValues($rowString)
    {
        $values = [];
        $current = '';
        $inQuotes = false;
        $quoteChar = null;

        for ($i = 0; $i < strlen($rowString); $i++) {
            $char = $rowString[$i];

            if (!$inQuotes && ($char === "'" || $char === '"')) {
                $inQuotes = true;
                $quoteChar = $char;
            } elseif ($inQuotes && $char === $quoteChar && ($i === 0 || $rowString[$i-1] !== '\\')) {
                $inQuotes = false;
                $quoteChar = null;
            } elseif (!$inQuotes && $char === ',') {
                $values[] = $this->cleanValue(trim($current));
                $current = '';
                continue;
            }

            $current .= $char;
        }

        if ($current !== '') {
            $values[] = $this->cleanValue(trim($current));
        }

        return $values;
    }

    private function cleanValue($value)
    {
        $value = trim($value);

        if ($value === 'NULL') {
            return null;
        }

        if (($value[0] === "'" && $value[-1] === "'") || ($value[0] === '"' && $value[-1] === '"')) {
            $value = substr($value, 1, -1);
            $value = str_replace(["\\", "\'", '\"'], ['\\', "'", '"'], $value);
        }

        return $value;
    }

    private function getTableColumns($tableName)
    {
        $columns = DB::select("SHOW COLUMNS FROM `{$tableName}`");
        return array_map(function($column) {
            return $column->Field;
        }, $columns);
    }

    private function processInsertWithConflictResolution($insertStatement, $tableName)
    {
        // Verificar si la tabla tiene columna 'id'
        $columns = DB::getSchemaBuilder()->getColumnListing($tableName);
        if (!in_array('id', $columns)) {
            // Si no tiene ID, insertar normalmente
            DB::unprepared($insertStatement);
            return;
        }

        // Extraer valores del INSERT
        preg_match('/VALUES\s*\((.*?)\)/is', $insertStatement, $valuesMatch);
        if (empty($valuesMatch[1])) return;

        $values = str_getcsv($valuesMatch[1], ',', "'");
        $id = !empty($values[0]) && is_numeric(trim($values[0], "' ")) ? (int)trim($values[0], "' ") : null;

        if ($id && DB::table($tableName)->where('id', $id)->exists()) {
            switch ($this->conflictResolution) {
                case 'auto_increment':
                    // Remover el ID y dejar que se auto-incremente
                    $modifiedInsert = $this->removeIdFromInsert($insertStatement, $tableName);
                    if ($modifiedInsert) {
                        DB::unprepared($modifiedInsert);
                    }
                    break;
                case 'skip':
                    // No hacer nada, saltar este registro
                    break;
                case 'replace':
                    // Ejecutar como está (reemplazará el registro existente)
                    DB::unprepared($insertStatement);
                    break;
            }
        } else {
            // No hay conflicto, insertar normalmente
            DB::unprepared($insertStatement);
        }
    }

    private function removeIdFromInsert($insertStatement, $tableName)
    {
        // Obtener columnas de la tabla
        $columns = DB::getSchemaBuilder()->getColumnListing($tableName);

        // Extraer la parte VALUES
        preg_match('/VALUES\s*\((.*?)\)/is', $insertStatement, $valuesMatch);
        if (empty($valuesMatch[1])) return null;

        $values = str_getcsv($valuesMatch[1], ',', "'");

        // Remover el primer valor (ID) si la primera columna es 'id'
        if (!empty($columns) && $columns[0] === 'id') {
            array_shift($values);
            $columnsWithoutId = array_slice($columns, 1);

            // Reconstruir el INSERT sin el ID
            $columnsList = '(`' . implode('`, `', $columnsWithoutId) . '`)';
            $valuesList = "('" . implode("', '", $values) . "')";

            return "INSERT INTO `{$tableName}` {$columnsList} VALUES {$valuesList};";
        }

        return $insertStatement;
    }

    private function importWithDateFilter($content)
    {
        // Procesar solo las tablas seleccionadas con filtro de fecha
        foreach ($this->selectedTables as $tableName) {
            $this->importTableSafely($content, $tableName);
        }
    }



    private function matchesDateFilter($insertStatement)
    {
        if (!$this->useDataFilter) {
            return true;
        }

        // Extraer tabla del INSERT
        if (preg_match('/INSERT INTO\s+`?([^`\s]+)`?/i', $insertStatement, $tableMatch)) {
            $tableName = $tableMatch[1];
            
            // Obtener las columnas de fecha de esta tabla
            try {
                $dateColumns = $this->getDateColumnsForTable($tableName);
                
                if (empty($dateColumns)) {
                    // Si no tiene columnas de fecha, incluir el registro
                    return true;
                }
                
                // Buscar fechas en las posiciones correspondientes a created_at/updated_at
                return $this->checkDateColumnsInInsert($insertStatement, $tableName, $dateColumns);
                
            } catch (\Exception $e) {
                // Si hay error, incluir el registro
                return true;
            }
        }

        return true;
    }
    
    private function getDateColumnsForTable($tableName)
    {
        try {
            if (!DB::getSchemaBuilder()->hasTable($tableName)) {
                return [];
            }
            
            $columns = DB::select("SHOW COLUMNS FROM `{$tableName}`");
            $dateColumns = [];
            
            foreach ($columns as $column) {
                if (in_array($column->Field, ['created_at', 'updated_at']) || 
                    in_array(strtolower($column->Type), ['datetime', 'timestamp', 'date'])) {
                    $dateColumns[] = $column->Field;
                }
            }
            
            return $dateColumns;
        } catch (\Exception $e) {
            return [];
        }
    }
    
    private function checkDateColumnsInInsert($insertStatement, $tableName, $dateColumns)
    {
        // Buscar todas las fechas en el INSERT
        preg_match_all('/\'(\d{4}-\d{2}-\d{2}(?:\s\d{2}:\d{2}:\d{2})?)\'/i', $insertStatement, $dateMatches);

        if (empty($dateMatches[1])) {
            return true;
        }

        foreach ($dateMatches[1] as $dateString) {
            try {
                $date = \Carbon\Carbon::parse($dateString)->format('Y-m-d');
                if ($date >= $this->filterFromDate && $date <= $this->filterToDate) {
                    return true;
                }
            } catch (\Exception $e) {
                continue;
            }
        }

        return false;
    }

    private function analyzeConflicts($content)
    {
        $conflicts = [];

        foreach ($this->selectedTables as $tableName) {
            if (!DB::getSchemaBuilder()->hasTable($tableName)) {
                $importCount = $this->getFilteredImportCount($content, $tableName);
                $conflicts[$tableName] = [
                    'status' => 'new',
                    'current_count' => 0,
                    'import_count' => $importCount,
                    'id_conflicts' => [],
                    'sample_data' => []
                ];
                continue;
            }

            $currentCount = DB::table($tableName)->count();
            $importCount = $this->getFilteredImportCount($content, $tableName);

            // Obtener muestra de datos actuales
            $currentSample = DB::table($tableName)->limit(3)->get()->toArray();

            // Extraer muestra de datos del SQL (filtrada)
            $importSample = $this->extractFilteredSampleFromSql($content, $tableName);

            // Analizar conflictos de ID (filtrados)
            $idConflicts = $this->analyzeFilteredIdConflicts($content, $tableName);

            $status = 'new';
            if ($currentCount > 0) {
                $status = !empty($idConflicts) ? 'id_conflict' : 'replace';
            }

            $conflicts[$tableName] = [
                'status' => $status,
                'current_count' => $currentCount,
                'import_count' => $importCount,
                'current_sample' => $currentSample,
                'import_sample' => $importSample,
                'id_conflicts' => $idConflicts
            ];
        }

        return $conflicts;
    }

    private function analyzeIdConflicts($content, $tableName)
    {
        if (!DB::getSchemaBuilder()->hasTable($tableName)) {
            return [];
        }

        // Verificar si la tabla tiene columna 'id'
        $columns = DB::getSchemaBuilder()->getColumnListing($tableName);
        if (!in_array('id', $columns)) {
            return [];
        }

        // Extraer IDs del SQL de importación
        $importIds = $this->extractIdsFromSql($content, $tableName);

        if (empty($importIds)) {
            return [];
        }

        // Verificar qué IDs ya existen
        $existingIds = DB::table($tableName)
            ->whereIn('id', $importIds)
            ->pluck('id')
            ->toArray();

        return array_intersect($importIds, $existingIds);
    }

    private function extractIdsFromSql($content, $tableName)
    {
        $ids = [];
        $insertPattern = '/INSERT INTO.*?`?' . preg_quote($tableName) . '`?.*?VALUES\s*\((.*?)\);/is';
        preg_match_all($insertPattern, $content, $matches);

        foreach ($matches[1] as $valuesString) {
            // Asumir que el primer valor es el ID
            $values = str_getcsv($valuesString, ',', "'");
            if (!empty($values[0]) && is_numeric(trim($values[0], "' "))) {
                $ids[] = (int)trim($values[0], "' ");
            }
        }

        return array_unique($ids);
    }

    private function extractSampleFromSql($content, $tableName)
    {
        // Buscar INSERT statements con o sin nombres de columnas
        $insertPattern = '/INSERT INTO.*?`?' . preg_quote($tableName) . '`?.*?VALUES\s*\((.*?)\);/is';
        preg_match_all($insertPattern, $content, $matches);

        $samples = [];
        for ($i = 0; $i < min(3, count($matches[1])); $i++) {
            // Manejar valores que pueden contener comas dentro de comillas
            $valuesString = $matches[1][$i];
            $values = $this->parseInsertValues($valuesString);
            $samples[] = array_slice($values, 0, 3); // Solo mostrar primeras 3 columnas
        }

        return $samples;
    }

    private function parseInsertValues($valuesString)
    {
        $values = [];
        $current = '';
        $inQuotes = false;
        $quoteChar = null;

        for ($i = 0; $i < strlen($valuesString); $i++) {
            $char = $valuesString[$i];

            if (!$inQuotes && ($char === "'" || $char === '"')) {
                $inQuotes = true;
                $quoteChar = $char;
                $current .= $char;
            } elseif ($inQuotes && $char === $quoteChar) {
                // Verificar si es escape
                if ($i > 0 && $valuesString[$i-1] === '\\') {
                    $current .= $char;
                } else {
                    $inQuotes = false;
                    $quoteChar = null;
                    $current .= $char;
                }
            } elseif (!$inQuotes && $char === ',') {
                $values[] = trim($current);
                $current = '';
            } else {
                $current .= $char;
            }
        }

        if ($current !== '') {
            $values[] = trim($current);
        }

        return array_map(function($value) {
            // Remover comillas externas y limpiar
            $value = trim($value);
            if (($value[0] === "'" && $value[-1] === "'") || ($value[0] === '"' && $value[-1] === '"')) {
                $value = substr($value, 1, -1);
            }
            return $value === 'NULL' ? null : $value;
        }, $values);
    }

    private function getFilteredImportCount($content, $tableName)
    {
        $insertPattern = '/INSERT INTO.*?`?' . preg_quote($tableName) . '`?.*?;/is';
        preg_match_all($insertPattern, $content, $matches);
        
        if (!$this->useDataFilter) {
            return count($matches[0]);
        }
        
        $filteredCount = 0;
        foreach ($matches[0] as $insertStatement) {
            if ($this->matchesDateFilter($insertStatement)) {
                $filteredCount++;
            }
        }
        
        return $filteredCount;
    }

    private function extractFilteredSampleFromSql($content, $tableName)
    {
        $insertPattern = '/INSERT INTO.*?`?' . preg_quote($tableName) . '`?.*?VALUES\s*\((.*?)\);/is';
        preg_match_all($insertPattern, $content, $matches);

        $samples = [];
        $count = 0;
        
        for ($i = 0; $i < count($matches[0]) && $count < 3; $i++) {
            $insertStatement = $matches[0][$i];
            
            // Aplicar filtro de fecha si está habilitado
            if ($this->useDataFilter && !$this->matchesDateFilter($insertStatement)) {
                continue;
            }
            
            $valuesString = $matches[1][$i];
            $values = $this->parseInsertValues($valuesString);
            $samples[] = array_slice($values, 0, 3);
            $count++;
        }

        return $samples;
    }

    private function analyzeFilteredIdConflicts($content, $tableName)
    {
        if (!DB::getSchemaBuilder()->hasTable($tableName)) {
            return [];
        }

        $columns = DB::getSchemaBuilder()->getColumnListing($tableName);
        if (!in_array('id', $columns)) {
            return [];
        }

        $importIds = $this->extractFilteredIdsFromSql($content, $tableName);

        if (empty($importIds)) {
            return [];
        }

        $existingIds = DB::table($tableName)
            ->whereIn('id', $importIds)
            ->pluck('id')
            ->toArray();

        return array_intersect($importIds, $existingIds);
    }

    private function extractFilteredIdsFromSql($content, $tableName)
    {
        $ids = [];
        $insertPattern = '/INSERT INTO.*?`?' . preg_quote($tableName) . '`?.*?VALUES\s*\((.*?)\);/is';
        preg_match_all($insertPattern, $content, $matches);

        for ($i = 0; $i < count($matches[0]); $i++) {
            $insertStatement = $matches[0][$i];
            
            // Aplicar filtro de fecha si está habilitado
            if ($this->useDataFilter && !$this->matchesDateFilter($insertStatement)) {
                continue;
            }
            
            $valuesString = $matches[1][$i];
            $values = str_getcsv($valuesString, ',', "'");
            if (!empty($values[0]) && is_numeric(trim($values[0], "' "))) {
                $ids[] = (int)trim($values[0], "' ");
            }
        }

        return array_unique($ids);
    }

    private function createTempFilteredSql($content)
    {
        $filteredSql = "";
        
        foreach ($this->selectedTables as $tableName) {
            // Solo crear tabla si no existe
            if (!DB::getSchemaBuilder()->hasTable($tableName)) {
                if (preg_match('/CREATE TABLE[^;]*`' . preg_quote($tableName) . '`[^;]*;/is', $content, $createMatch)) {
                    $filteredSql .= $createMatch[0] . "\n\n";
                }
            }
            
            // Extraer INSERT INTO
            if (preg_match_all('/INSERT INTO[^;]*`' . preg_quote($tableName) . '`[^;]*;/is', $content, $insertMatches)) {
                foreach ($insertMatches[0] as $insertStatement) {
                    if (!$this->useDataFilter || $this->matchesDateFilter($insertStatement)) {
                        $filteredSql .= $insertStatement . "\n";
                    }
                }
            }
        }
        
        $tempPath = storage_path('app/temp_import_' . uniqid() . '.sql');
        file_put_contents($tempPath, $filteredSql);
        
        return $tempPath;
    }
    
    private function filterSqlBySelectedTables($content)
    {
        if (empty($this->selectedTables)) {
            return "-- No hay tablas seleccionadas";
        }
        
        $filteredSql = "";
        
        // Solo procesar las tablas que están en selectedTables
        foreach ($this->selectedTables as $tableName) {
            // Buscar CREATE TABLE para esta tabla específica
            if (preg_match('/CREATE TABLE[^;]*`' . preg_quote($tableName) . '`[^;]*;/is', $content, $createMatch)) {
                $filteredSql .= $createMatch[0] . "\n\n";
            }
            
            // Buscar INSERT para esta tabla específica
            if (preg_match_all('/INSERT INTO[^;]*`' . preg_quote($tableName) . '`[^;]*;/is', $content, $insertMatches)) {
                foreach ($insertMatches[0] as $insertStatement) {
                    if (!$this->useDataFilter || $this->matchesDateFilter($insertStatement)) {
                        $filteredSql .= $insertStatement . "\n";
                    }
                }
            }
            
            $filteredSql .= "\n";
        }
        
        return $filteredSql;
    }
    
    private function cleanupTempFile()
    {
        if ($this->tempSqlPath && file_exists($this->tempSqlPath)) {
            unlink($this->tempSqlPath);
            $this->tempSqlPath = null;
        }
    }
    
    private function applyDateFilterToTableBlock($tableBlock, $tableName)
    {
        // Separar estructura de datos
        $parts = preg_split('/(-- Volcando datos para la tabla.*?)/', $tableBlock, -1, PREG_SPLIT_DELIM_CAPTURE);
        
        if (count($parts) < 3) {
            return $tableBlock; // No hay datos, devolver como está
        }
        
        $structure = $parts[0]; // CREATE TABLE
        $dataHeader = $parts[1]; // Comentario de datos
        $dataContent = $parts[2]; // DELETE + INSERT
        
        // Verificar si los datos coinciden con el filtro de fecha
        if (!$this->matchesDateFilter($dataContent)) {
            return $structure . $dataHeader . "-- Sin datos en el rango de fechas seleccionado\n";
        }
        
        return $tableBlock;
    }

    public function render()
    {
        return $this->renderWithLayout('livewire.admin.monitoreo.database-import', [], [
            'title' => 'Importar Base de Datos',
            'description' => 'Importar archivos SQL con filtros de fecha',
            'breadcrumb' => [
                ['name' => 'Dashboard', 'route' => 'admin.dashboard'],
                ['name' => 'Monitoreo', 'active' => false],
                ['name' => 'Importar BD', 'active' => true]
            ]
        ]);
    }
}
