<?php

namespace App\Livewire\Admin\Marcas;

use App\Models\Marca;
use Livewire\Component;
use App\Traits\HasDynamicLayout;

class Edit extends Component
{
    use HasDynamicLayout;

    public Marca $marca;
    public $nombre;
    public $descripcion;
    public $activo = true;

    protected $rules = [
        'nombre' => 'required|string|max:255|unique:marcas,nombre,{id}',
        'descripcion' => 'nullable|string|max:1000',
        'activo' => 'boolean'
    ];

    protected $messages = [
        'nombre.required' => 'El nombre de la marca es obligatorio.',
        'nombre.unique' => 'Ya existe una marca con este nombre.',
        'nombre.max' => 'El nombre no puede exceder 255 caracteres.',
        'descripcion.max' => 'La descripción no puede exceder 1000 caracteres.'
    ];

    public function mount(Marca $marca)
    {
        $this->marca = $marca;
        $this->nombre = $marca->nombre;
        $this->descripcion = $marca->descripcion;
        $this->activo = $marca->activo;
    }

    public function save()
    {
        $this->rules['nombre'] = str_replace('{id}', $this->marca->id, $this->rules['nombre']);
        $this->validate();

        $this->marca->update([
            'nombre' => $this->nombre,
            'descripcion' => $this->descripcion,
            'activo' => $this->activo
        ]);

        $this->dispatch('notify', ['message' => 'Marca actualizada correctamente.', 'type' => 'success']);
        return redirect()->route('admin.marcas.index');
    }

    public function render()
    {
        return $this->renderWithLayout('livewire.admin.marcas.edit', [
            'marca' => $this->marca
        ], [
            'title' => 'Editar Marca',
            'description' => 'Modificar marca del sistema'
        ]);
    }
}
