<?php

namespace App\Livewire\Admin\Empleados;

use App\Models\Pedido;
use App\Models\Empleado;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\Auth;
use Livewire\Component;

class PedidoDetalle extends Component
{
    use HasDynamicLayout;
    
    public $pedido;
    public $codigo;
    public $empleado;

    public function mount($codigo)
    {
        $this->codigo = $codigo;
        $this->empleado = Empleado::where('user_id', Auth::id())->first();
        
        if (!$this->empleado) {
            abort(403, 'No tienes perfil de empleado');
        }

        $this->pedido = Pedido::with([
            'detalles.producto.images',
            'referenciaBancaria.banco', 
            'pagos',
            'user',
            'empleados'
        ])->where('codigo', $codigo)->first();

        if (!$this->pedido) {
            abort(404, 'Pedido no encontrado');
        }

        if (!$this->pedido->empleados->contains('id', $this->empleado->id)) {
            abort(403, 'No tienes acceso a este pedido');
        }
    }

    public function updateOrderStatus($status)
    {
        $this->pedido->update(['estado' => $status]);
        $this->sendWhatsAppNotification($status);
        
        $this->dispatch('alert', [
            'title' => '¡Actualizado!',
            'text' => "Pedido #{$this->pedido->codigo} actualizado a {$status}",
            'icon' => 'success'
        ]);
    }

    private function sendWhatsAppNotification($status)
    {
        try {
            if (class_exists('\App\Services\WhatsAppService') && $this->pedido->user->telefono) {
                $whatsapp = new \App\Services\WhatsAppService();
                $message = $this->getStatusMessage($status);
                $whatsapp->sendMessage($this->pedido->user->telefono, $message);
            }
        } catch (\Exception $e) {
            // Silently fail WhatsApp notification
        }
    }

    private function getStatusMessage($status)
    {
        $name = $this->pedido->user->name;
        $code = $this->pedido->codigo;
        
        return match($status) {
            'Procesando' => "¡Hola {$name}! Tu pedido #{$code} está siendo preparado. 📦",
            'entregado' => "¡Hola {$name}! Tu pedido #{$code} ha sido entregado exitosamente. ✅",
            'cancelado' => "Hola {$name}, lamentamos informarte que tu pedido #{$code} ha sido cancelado. ❌",
            default => "Tu pedido #{$code} ha cambiado a: {$status}"
        };
    }
    
    public function goBack()
    {
        return redirect()->route('admin.empleados.dashboard');
    }

    public function render()
    {
        return view('livewire.admin.empleados.pedido-detalle')
            ->layout($this->getLayout(), [
                'title' => 'Pedido #' . $this->pedido->codigo,
                'breadcrumb' => [
                    ['name' => 'Mi Dashboard', 'url' => route('admin.empleados.dashboard')],
                    ['name' => 'Pedido #' . $this->pedido->codigo, 'active' => true]
                ]
            ]);
    }
}
