<?php

namespace App\Livewire\Admin\Empleados;

use App\Models\Empleado;
use App\Models\Empresa;
use App\Models\Sucursal;
use App\Models\User;
use App\Services\WhatsAppService;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Livewire\Component;
use Spatie\Permission\Models\Role;

class Create extends Component
{
    use HasDynamicLayout;
    public $nombres = '';
    public $apellidos = '';
    public $documento = '';
    public $tipo_documento = 'cedula';
    public $email = '';
    public $telefono = '';
    public $fecha_nacimiento = '';
    public $genero = '';
    public $direccion = '';
    public $cargo = '';
    public $departamento = '';
    public $salario = '';
    public $fecha_ingreso = '';
    public $estado_civil = '';
    public $contacto_emergencia = '';
    public $telefono_emergencia = '';
    public $observaciones = '';
    public $status = true;
    public $empresa_id = '';
    public $sucursal_id = '';

    protected $rules = [
        'nombres' => 'required|string|max:255',
        'apellidos' => 'required|string|max:255',
        'documento' => 'required|string|unique:empleados,documento',
        'tipo_documento' => 'required|in:cedula,pasaporte,rif',
        'email' => 'required|email|unique:empleados,email|unique:users,email',
        'telefono' => 'required|string|max:20',
        'fecha_nacimiento' => 'nullable|date',
        'genero' => 'nullable|in:masculino,femenino,otro',
        'direccion' => 'nullable|string',
        'cargo' => 'required|string|max:255',
        'departamento' => 'required|string|max:255',
        'salario' => 'nullable|numeric|min:0',
        'fecha_ingreso' => 'required|date',
        'estado_civil' => 'nullable|in:soltero,casado,divorciado,viudo',
        'contacto_emergencia' => 'nullable|string|max:255',
        'telefono_emergencia' => 'nullable|string|max:20',
        'observaciones' => 'nullable|string',
        'empresa_id' => 'required|exists:empresas,id',
        'sucursal_id' => 'required|exists:sucursales,id'
    ];

    public function mount()
    {
        $user = auth()->user();
        if ($user->empresa_id) {
            $this->empresa_id = $user->empresa_id;
        }
        if ($user->sucursal_id) {
            $this->sucursal_id = $user->sucursal_id;
        }
        $this->fecha_ingreso = now()->format('Y-m-d');
    }

    public function save()
    {
        $this->validate();

        $codigo = 'EMP-' . str_pad(Empleado::count() + 1, 4, '0', STR_PAD_LEFT);
        $password = 'password';
        $username = $this->generateUsername();

        // Crear usuario
        $user = User::create([
            'name' => $this->nombres . ' ' . $this->apellidos,
            'email' => $this->email,
            'username' => $username,
            'password' => Hash::make($password),
            'empresa_id' => $this->empresa_id,
            'sucursal_id' => $this->sucursal_id
        ]);

        // Asignar rol Empleado
        $empleadoRole = Role::where('name', 'Empleado')->first();
        if ($empleadoRole) {
            $user->assignRole($empleadoRole);
        }

        // Crear empleado
        $empleado = Empleado::create([
            'codigo' => $codigo,
            'nombres' => $this->nombres,
            'apellidos' => $this->apellidos,
            'documento' => $this->documento,
            'tipo_documento' => $this->tipo_documento,
            'email' => $this->email,
            'telefono' => $this->telefono,
            'fecha_nacimiento' => $this->fecha_nacimiento,
            'genero' => $this->genero,
            'direccion' => $this->direccion,
            'cargo' => $this->cargo,
            'departamento' => $this->departamento,
            'salario' => $this->salario,
            'fecha_ingreso' => $this->fecha_ingreso,
            'estado_civil' => $this->estado_civil,
            'contacto_emergencia' => $this->contacto_emergencia,
            'telefono_emergencia' => $this->telefono_emergencia,
            'observaciones' => $this->observaciones,
            'status' => $this->status,
            'empresa_id' => $this->empresa_id,
            'sucursal_id' => $this->sucursal_id,
            'user_id' => $user->id
        ]);

        // Enviar WhatsApp con credenciales
        $this->sendWhatsAppCredentials($username, $password);

        session()->flash('message', 'Empleado y usuario creados exitosamente. Se han enviado las credenciales por WhatsApp.');
        return redirect()->route('admin.empleados.index');
    }

    private function generateUsername()
    {
        $nombres = explode(' ', trim($this->nombres));
        $apellidos = explode(' ', trim($this->apellidos));
        
        $primerNombre = strtolower($nombres[0] ?? '');
        $primerApellido = strtolower($apellidos[0] ?? '');
        
        $baseUsername = substr($primerNombre, 0, 1) . $primerApellido;
        $username = $baseUsername;
        $counter = 1;
        
        while (User::where('username', $username)->exists()) {
            $username = $baseUsername . $counter;
            $counter++;
        }
        
        return $username;
    }

    private function sendWhatsAppCredentials($username, $password)
    {
        if (!$this->telefono) return;

        $whatsappService = app(WhatsAppService::class);
        $systemUrl = config('app.url');
        
        $message = "¡Hola {$this->nombres}! 👋\n\n";
        $message .= "Te damos la bienvenida al sistema. Aquí tienes tus credenciales de acceso:\n\n";
        $message .= "🔐 **Credenciales:**\n";
        $message .= "Usuario: {$username}\n";
        $message .= "Contraseña: {$password}\n\n";
        $message .= "🌐 **Acceso al sistema:**\n";
        $message .= "{$systemUrl}\n\n";
        $message .= "Por favor, cambia tu contraseña después del primer acceso.\n\n";
        $message .= "¡Que tengas un excelente día! 🚀";

        try {
            $whatsappService->sendMessage($this->telefono, $message);
        } catch (\Exception $e) {
            // Log error but don't fail the employee creation
            \Log::error('Error enviando WhatsApp: ' . $e->getMessage());
        }
    }

    public function render()
    {
        $empresas = Empresa::all();
        $sucursales = $this->empresa_id ? Sucursal::where('empresa_id', $this->empresa_id)->get() : collect();

        return $this->renderWithLayout('livewire.admin.empleados.create', compact('empresas', 'sucursales'));
    }
}