<?php

namespace App\Livewire\Admin\Cotizaciones;

use Livewire\Component;
use App\Models\Cotizacion;
use App\Models\NotaVenta;
use App\Models\DetalleNotaVenta;
use App\Models\Serie;
use App\Traits\HasRegionalFormatting;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\DB;

class Show extends Component
{
    use HasRegionalFormatting, HasDynamicLayout;

    public Cotizacion $cotizacion;
    public $showConvertModal = false;
    public $observaciones_nota = '';

    public function mount(Cotizacion $cotizacion)
    {
        $this->bootHasRegionalFormatting();
        $this->cotizacion = $cotizacion->load([
            'cliente',
            'detalles.producto',
            'detalles.productoVariant',
            'user',
            'empresa',
            'sucursal'
        ]);
    }

    public function cambiarEstado($nuevoEstado)
    {
        if ($this->cotizacion->estado === 'aprobada' && $nuevoEstado !== 'aprobada') {
            session()->flash('error', 'No se puede cambiar el estado de una cotización aprobada');
            return;
        }

        $this->cotizacion->update(['estado' => $nuevoEstado]);
        session()->flash('message', 'Estado actualizado correctamente');
    }

    public function abrirModalConvertir()
    {
        if ($this->cotizacion->estado !== 'aprobada') {
            session()->flash('error', 'Solo se pueden convertir cotizaciones aprobadas');
            return;
        }

        if ($this->cotizacion->estado === 'convertida') {
            session()->flash('error', 'Esta cotización ya fue convertida');
            return;
        }

        $this->showConvertModal = true;
    }

    public function cerrarModalConvertir()
    {
        $this->showConvertModal = false;
        $this->observaciones_nota = '';
    }

    public function convertirANotaVenta()
    {
        try {
            DB::transaction(function () {
                // Obtener serie para notas de venta
                $serie = Serie::where('tipo_documento', 'nota_venta')
                    ->where('activo', true)
                    ->where('empresa_id', auth()->user()->empresa_id)
                    ->where('sucursal_id', auth()->user()->sucursal_id)
                    ->first();

                if (!$serie) {
                    throw new \Exception('No se encontró una serie activa para notas de venta');
                }

                // Crear nota de venta
                $notaVenta = NotaVenta::create([
                    'serie' => $serie->serie,
                    'numero' => $serie->correlativo_actual + 1,
                    'fecha_emision' => now(),
                    'fecha_vencimiento' => now()->addDays(30),
                    'cliente_id' => $this->cotizacion->cliente_id,
                    'observaciones' => $this->observaciones_nota ?: 'Generada desde cotización ' . $this->cotizacion->numero_completo,
                    'subtotal' => $this->cotizacion->subtotal,
                    'impuestos' => $this->cotizacion->impuestos,
                    'total' => $this->cotizacion->total,
                    'estado' => 'pendiente',
                    'empresa_id' => $this->cotizacion->empresa_id,
                    'sucursal_id' => $this->cotizacion->sucursal_id,
                    'user_id' => auth()->id(),
                    'serie_id' => $serie->id
                ]);

                // Crear detalles de nota de venta
                foreach ($this->cotizacion->detalles as $detalle) {
                    DetalleNotaVenta::create([
                        'nota_venta_id' => $notaVenta->id,
                        'producto_id' => $detalle->producto_id,
                        'producto_variant_id' => $detalle->producto_variant_id,
                        'descripcion' => $detalle->descripcion,
                        'cantidad' => $detalle->cantidad,
                        'precio_unitario' => $detalle->precio_unitario,
                        'subtotal' => $detalle->subtotal
                    ]);
                }

                // Actualizar serie
                $serie->increment('correlativo_actual');

                // Marcar cotización como convertida y vincular nota de venta
                $this->cotizacion->update([
                    'estado' => 'convertida',
                    'nota_venta_id' => $notaVenta->id
                ]);
            });

            $this->cerrarModalConvertir();
            session()->flash('message', 'Cotización convertida a nota de venta exitosamente');

        } catch (\Exception $e) {
            session()->flash('error', 'Error al convertir a nota de venta: ' . $e->getMessage());
        }
    }

    public function render()
    {
        return view('livewire.admin.cotizaciones.show')->layout($this->getLayout());
    }
}
