<?php

namespace App\Livewire\Admin\Cotizaciones;

use Livewire\Component;
use Livewire\WithPagination;
use App\Models\Cotizacion;
use App\Models\Cliente;
use App\Traits\HasRegionalFormatting;
use App\Traits\HasDynamicLayout;
use Illuminate\Support\Facades\DB;

class Index extends Component
{
    use WithPagination, HasRegionalFormatting, HasDynamicLayout;

    public $search = '';
    public $fechaInicio = '';
    public $fechaFin = '';
    public $estado = '';
    public $clienteId = '';
    public $perPage = 15;

    protected $paginationTheme = 'bootstrap';

    protected $queryString = [
        'search' => ['except' => ''],
        'fechaInicio' => ['except' => ''],
        'fechaFin' => ['except' => ''],
        'estado' => ['except' => ''],
        'clienteId' => ['except' => '']
    ];

    public function mount()
    {
        $this->bootHasRegionalFormatting();
        $this->fechaInicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechaFin = now()->endOfMonth()->format('Y-m-d');
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function limpiarFiltros()
    {
        $this->search = '';
        $this->fechaInicio = now()->startOfMonth()->format('Y-m-d');
        $this->fechaFin = now()->endOfMonth()->format('Y-m-d');
        $this->estado = '';
        $this->clienteId = '';
        $this->resetPage();
    }

    public function cambiarEstado($cotizacionId, $nuevoEstado)
    {
        $cotizacion = Cotizacion::find($cotizacionId);
        if ($cotizacion) {
            $cotizacion->update(['estado' => $nuevoEstado]);
            $this->dispatch('show-message', [
                'type' => 'success',
                'message' => 'Estado actualizado correctamente'
            ]);
        }
    }

    public function getTotalCotizacionesProperty()
    {
        return Cotizacion::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->whereDate('fecha_emision', '>=', $this->fechaInicio)
            ->whereDate('fecha_emision', '<=', $this->fechaFin)
            ->count();
    }

    public function getTotalMontoProperty()
    {
        return Cotizacion::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->whereDate('fecha_emision', '>=', $this->fechaInicio)
            ->whereDate('fecha_emision', '<=', $this->fechaFin)
            ->sum('total');
    }

    public function getCotizacionesPorEstadoProperty()
    {
        $estados = ['pendiente','rechazada', 'vencida', 'convertida'];
        $result = [];

        foreach ($estados as $estado) {
            $cotizaciones = Cotizacion::where('empresa_id', auth()->user()->empresa_id)
                ->where('sucursal_id', auth()->user()->sucursal_id)
                ->where('estado', $estado)
                ->whereDate('fecha_emision', '>=', $this->fechaInicio)
                ->whereDate('fecha_emision', '<=', $this->fechaFin);

            $result[] = [
                'nombre' => ucfirst($estado),
                'cantidad' => $cotizaciones->count(),
                'monto' => $this->formatMoney($cotizaciones->sum('total'))
            ];
        }

        return $result;
    }

    public function render()
    {
        $query = Cotizacion::with(['cliente', 'user'])
            ->where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id);

        if ($this->search) {
            $query->where(function($q) {
                $q->where('numero', 'like', '%' . $this->search . '%')
                  ->orWhere('serie', 'like', '%' . $this->search . '%')
                  ->orWhereHas('cliente', function ($subQ) {
                      $subQ->where('nombres', 'like', '%' . $this->search . '%')
                           ->orWhere('apellidos', 'like', '%' . $this->search . '%');
                  });
            });
        }

        if ($this->fechaInicio) {
            $query->whereDate('created_at', '>=', $this->fechaInicio);
        }
        if ($this->fechaFin) {
            $query->whereDate('created_at', '<=', $this->fechaFin);
        }
        if ($this->estado) {
            $query->where('estado', $this->estado);
        }
        if ($this->clienteId) {
            $query->where('cliente_id', $this->clienteId);
        }

        $query->orderBy('created_at', 'desc');
        $cotizaciones = $query->paginate($this->perPage);

        $clientes = Cliente::where('empresa_id', auth()->user()->empresa_id)
            ->where('sucursal_id', auth()->user()->sucursal_id)
            ->orderBy('nombres')
            ->get();

        return view('livewire.admin.cotizaciones.index', [
            'cotizaciones' => $cotizaciones,
            'clientes' => $clientes,
            'estados' => Cotizacion::getEstados()
        ])->layout($this->getLayout());
    }
}
