<?php

namespace App\Livewire\Admin\Cajas;

use App\Models\Caja;
use App\Models\ExchangeRate;
use App\Traits\HasDynamicLayout;
use App\Traits\HasRegionalFormatting;
use App\Traits\HasDualCurrency;
use Livewire\Component;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Alignment;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

class Show extends Component
{
    use HasDynamicLayout, HasRegionalFormatting, HasDualCurrency;


    public Caja $caja;
    public $observaciones_cierre = '';
    public $showCerrarModal = false;

    protected $rules = [
        'observaciones_cierre' => 'nullable|string|max:500',
    ];

    public function mount(Caja $caja)
    {
        $this->caja = $caja->load(['usuario', 'sucursal']);
    }

    public function abrirModalCerrar()
    {
        if ($this->caja->estado === 'cerrada') {
            session()->flash('error', 'La caja ya está cerrada.');
            return;
        }

        $this->caja->calcularTotales();
        $this->showCerrarModal = true;
    }

    public function cerrarCaja()
    {
        $this->validate();

        if ($this->caja->cerrar($this->observaciones_cierre)) {
            $this->showCerrarModal = false;
            session()->flash('message', 'Caja cerrada exitosamente.');
            $this->caja->refresh();
        } else {
            session()->flash('error', 'No se pudo cerrar la caja.');
        }
    }

    public function getResumenPorMetodoProperty()
    {
        // Usar PedidoPago para el sistema de comercio
        // Incluir pagos de pedidos confirmados (estado = 'pagado' o 'entregado')
        return $this->caja->pedidoPagos()
            ->whereHas('pedido', function($query) {
                $query->whereIn('estado', ['pagado', 'entregado']);
            })
            ->selectRaw('metodo_pago, COUNT(*) as cantidad, SUM(total_usd) as total')
            ->groupBy('metodo_pago')
            ->get()
            ->map(function ($item) {
                // Normalizar el nombre del método para agrupación
                $metodo = strtolower(trim($item->metodo_pago));
                switch ($metodo) {
                    case 'efectivo bs':
                    case 'efectivo $':
                        $item->metodo_pago = 'efectivo';
                        break;
                    case 'pago móvil':
                        $item->metodo_pago = 'pago_movil';
                        break;
                }
                return $item;
            })
            ->groupBy('metodo_pago')
            ->map(function ($group) {
                $first = $group->first();
                return (object) [
                    'metodo_pago' => $first->metodo_pago,
                    'cantidad' => $group->sum('cantidad'),
                    'total' => $group->sum('total'),
                ];
            })
            ->values();
    }

    public function getResumenPorConceptoProperty()
    {
        // Para el sistema de comercio, usar métodos de pago como conceptos
        // Incluir pagos de pedidos confirmados (estado = 'pagado' o 'entregado')
        return $this->caja->pedidoPagos()
            ->whereHas('pedido', function($query) {
                $query->whereIn('estado', ['pagado', 'entregado']);
            })
            ->selectRaw('metodo_pago as concepto, COUNT(*) as cantidad, SUM(total_usd) as total')
            ->groupBy('metodo_pago')
            ->get()
            ->map(function ($item) {
                return [
                    'concepto' => ucfirst(str_replace('_', ' ', $item->concepto)),
                    'cantidad' => $item->cantidad,
                    'total' => $item->total,
                ];
            });
    }

    public function exportarExcel()
    {
        return redirect()->route('admin.cajas.export', $this->caja->id);
    }

    public function render()
    {
        return view('livewire.admin.cajas.show')->layout($this->getLayout());
    }
}
