<?php

namespace App\Livewire\Admin\Cajas;

use App\Services\CajaTasaConfigService;
use App\Models\ExchangeRate;
use App\Models\Caja;
use Carbon\Carbon;
use Livewire\Component;

class CajaTasaAutomationStatus extends Component
{
    public $config;
    public $tasaActual;
    public $proximosCortes;
    public $ultimoCorte;
    public $logsRecientes;

    protected $listeners = ['refreshStatus' => '$refresh'];

    public function mount()
    {
        $this->cargarDatos();
    }

    public function cargarDatos()
    {
        $this->config = CajaTasaConfigService::getConfiguracionActual();
        $this->tasaActual = ExchangeRate::getTodayRate();
        $this->proximosCortes = $this->obtenerProximosCortes();
        $this->ultimoCorte = $this->obtenerUltimoCorte();
        $this->logsRecientes = $this->obtenerLogsRecientes();
    }

    public function render()
    {
        return view('livewire.admin.cajas.caja-tasa-automation-status', [
            'config' => $this->config,
            'tasaActual' => $this->tasaActual,
            'proximosCortes' => $this->proximosCortes,
            'ultimoCorte' => $this->ultimoCorte,
            'logsRecientes' => $this->logsRecientes,
        ]);
    }

    public function actualizarTasaManual()
    {
        try {
            $service = new \App\Services\CajaTasaAutomationService();
            $resultado = $service->actualizarTasaAutomaticamente();

            if ($resultado['exitoso']) {
                session()->flash('success', $resultado['mensaje']);
            } else {
                session()->flash('error', $resultado['mensaje']);
            }

            $this->cargarDatos();
        } catch (\Exception $e) {
            session()->flash('error', 'Error al actualizar tasa: ' . $e->getMessage());
        }
    }

    public function realizarCorteManual()
    {
        try {
            $service = new \App\Services\CajaTasaAutomationService();
            $resultado = $service->realizarCorteCajaAutomatico('manual');

            if ($resultado['exitoso']) {
                session()->flash('success', $resultado['mensaje']);
            } else {
                session()->flash('error', $resultado['mensaje']);
            }

            $this->cargarDatos();
        } catch (\Exception $e) {
            session()->flash('error', 'Error al realizar corte: ' . $e->getMessage());
        }
    }

    private function obtenerProximosCortes()
    {
        $horarios = \App\Services\CajaTasaConfigService::CORTE_CAJA_HORARIOS;
        $hoy = Carbon::today();
        $ahora = Carbon::now();
        $proximos = [];

        foreach ($horarios as $hora) {
            $fechaHora = $hoy->copy()->setTimeFromTimeString($hora);
            if ($fechaHora->isFuture()) {
                $proximos[] = [
                    'hora' => $hora,
                    'faltan' => $ahora->diffForHumans($fechaHora, true),
                ];
            }
        }

        return $proximos;
    }

    private function obtenerUltimoCorte()
    {
        return Caja::where('estado', 'cerrada')
            ->whereDate('fecha_cierre', today())
            ->orderBy('fecha_cierre', 'desc')
            ->first();
    }

    private function obtenerLogsRecientes()
    {
        $logs = [];
        $archivos = [
            'cajatasa-automation.log',
            'tasa-programada.log',
            'corte-caja-1pm.log',
            'corte-caja-6pm.log',
            'tasa-verificacion.log',
        ];

        foreach ($archivos as $archivo) {
            $ruta = storage_path("logs/{$archivo}");
            if (file_exists($ruta)) {
                $contenido = file_get_contents($ruta);
                $lineas = array_slice(array_filter(explode("\n", $contenido)), -5);
                
                foreach ($lineas as $linea) {
                    if (trim($linea)) {
                        $logs[] = [
                            'archivo' => $archivo,
                            'linea' => $linea,
                            'fecha' => date('Y-m-d H:i:s', filemtime($ruta)),
                        ];
                    }
                }
            }
        }

        // Ordenar por fecha descendente
        usort($logs, function($a, $b) {
            return strtotime($b['fecha']) - strtotime($a['fecha']);
        });

        return array_slice($logs, 0, 10);
    }
}