<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\Pedido;
use App\Services\WhatsAppService;
use Illuminate\Support\Facades\Log;

class SendOrderWhatsAppNotification implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;
    public $backoff = [30, 60, 120]; // Reintentos en 30s, 1m, 2m

    protected $pedido;

    public function __construct(Pedido $pedido)
    {
        $this->pedido = $pedido;
    }

    public function handle()
    {
        try {
            Log::info('🚀 Enviando notificación WhatsApp para pedido', [
                'pedido_id' => $this->pedido->id,
                'codigo' => $this->pedido->codigo,
                'user_id' => $this->pedido->userId
            ]);

            $whatsappService = new WhatsAppService();
            $result = $whatsappService->sendOrderNotification($this->pedido);

            if ($result['success']) {
                Log::info('✅ Notificación WhatsApp enviada exitosamente', [
                    'pedido_id' => $this->pedido->id,
                    'codigo' => $this->pedido->codigo
                ]);
            } else {
                Log::warning('⚠️ Fallo al enviar notificación WhatsApp', [
                    'pedido_id' => $this->pedido->id,
                    'error' => $result['error'] ?? 'Error desconocido'
                ]);
                
                // Lanzar excepción para activar reintentos
                throw new \Exception($result['error'] ?? 'Error al enviar WhatsApp');
            }
        } catch (\Exception $e) {
            Log::error('❌ Error en Job de WhatsApp', [
                'pedido_id' => $this->pedido->id,
                'error' => $e->getMessage(),
                'attempt' => $this->attempts()
            ]);
            
            throw $e; // Re-lanzar para activar reintentos
        }
    }

    public function failed(\Throwable $exception)
    {
        Log::error('💥 Job de WhatsApp falló definitivamente', [
            'pedido_id' => $this->pedido->id,
            'error' => $exception->getMessage(),
            'attempts' => $this->attempts()
        ]);
    }
}