<?php

namespace App\Jobs;

use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use App\Models\Pedido;
use App\Models\User;
use App\Services\WhatsAppService;
use Illuminate\Support\Facades\Log;

class SendOrderAssignmentNotification implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;
    public $backoff = [30, 60, 120];

    protected $pedido;
    protected $empleado;

    public function __construct(Pedido $pedido, $empleado)
    {
        $this->pedido = $pedido;
        $this->empleado = $empleado;
    }

    public function handle()
    {
        try {
            Log::info('🚀 Enviando notificaciones de asignación de pedido', [
                'pedido_id' => $this->pedido->id,
                'empleado_id' => $this->empleado->id
            ]);

            $whatsappService = new WhatsAppService();

            // Notificar al empleado
            $this->notifyEmployee($whatsappService);
            
            // Notificar al cliente
            $this->notifyCustomer($whatsappService);

        } catch (\Exception $e) {
            Log::error('❌ Error en Job de asignación', [
                'pedido_id' => $this->pedido->id,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    private function notifyEmployee($whatsappService)
    {
        // Obtener teléfono del empleado (puede ser del modelo Empleado o User)
        $telefono = $this->empleado->telefono ?? $this->empleado->user->telefono ?? null;
        
        if (!$telefono) {
            Log::warning('Empleado sin teléfono', ['empleado_id' => $this->empleado->id]);
            return;
        }

        // Formatear teléfono del empleado
        $telefonoFormateado = $whatsappService->cleanPhoneNumber($telefono);

        $message = $this->formatEmployeeMessage();
        $result = $whatsappService->sendMessage($telefonoFormateado, $message);

        if ($result['success']) {
            Log::info('✅ Notificación enviada al empleado', [
                'empleado_id' => $this->empleado->id,
                'pedido_id' => $this->pedido->id
            ]);
        } else {
            Log::warning('⚠️ Fallo notificación empleado', [
                'empleado_id' => $this->empleado->id,
                'error' => $result['error'] ?? 'Error desconocido'
            ]);
        }
    }

    private function notifyCustomer($whatsappService)
    {
        if (!$this->pedido->user->telefono) {
            Log::warning('Cliente sin teléfono', ['user_id' => $this->pedido->user->id]);
            return;
        }

        $message = $this->formatCustomerMessage();
        $result = $whatsappService->sendMessage($this->pedido->user->telefono, $message);

        if ($result['success']) {
            Log::info('✅ Notificación enviada al cliente', [
                'user_id' => $this->pedido->user->id,
                'pedido_id' => $this->pedido->id
            ]);
        } else {
            Log::warning('⚠️ Fallo notificación cliente', [
                'user_id' => $this->pedido->user->id,
                'error' => $result['error'] ?? 'Error desconocido'
            ]);
        }
    }

    private function formatEmployeeMessage()
    {
        // Obtener nombre del empleado
        $nombreEmpleado = $this->empleado->nombres ?? $this->empleado->name ?? 'Empleado';
        
        $message = "🚚 *¡Nuevo Pedido Asignado!* 🚚\n\n";
        $message .= "*Hola {$nombreEmpleado}!*\n\n";
        $message .= "Se te ha asignado el pedido *#{$this->pedido->codigo}*\n\n";
        
        $message .= "👤 *Cliente:* {$this->pedido->user->name}\n";
        $message .= "📱 *Teléfono:* {$this->pedido->user->telefono}\n";
        $message .= "💰 *Total:* " . format_money($this->pedido->total_usd) . "\n";
        $message .= "📍 *Dirección:* {$this->pedido->ubicacion}\n\n";
        
        if ($this->pedido->detalles->count() > 0) {
            $message .= "📦 *Productos:*\n";
            foreach ($this->pedido->detalles as $detalle) {
                $productoNombre = $detalle->producto ? $detalle->producto->name : 'Producto';
                $message .= "• {$detalle->quantity}x {$productoNombre}\n";
            }
            $message .= "\n";
        }
        
        $message .= "⏰ *Tiempo estimado:* 15-30 minutos\n\n";
        $message .= "¡Por favor, procede con la entrega!";
        
        return $message;
    }

    private function formatCustomerMessage()
    {
        // Obtener datos del empleado
        $nombreEmpleado = $this->empleado->nombres ?? $this->empleado->name ?? 'Nuestro repartidor';
        $telefonoEmpleado = $this->empleado->telefono ?? $this->empleado->user->telefono ?? 'Se contactará contigo';
        
        $message = "🚚 *¡Pedido en Camino!* 🚚\n\n";
        $message .= "*Hola {$this->pedido->user->name}!*\n\n";
        $message .= "Tu pedido *#{$this->pedido->codigo}* ha sido asignado para entrega.\n\n";
        
        $message .= "👨‍🚚 *Repartidor:* {$nombreEmpleado}\n";
        $message .= "📱 *Contacto:* {$telefonoEmpleado}\n";
        $message .= "⏰ *Tiempo estimado:* 15-30 minutos\n\n";
        
        $message .= "¡Tu pedido está en camino! El repartidor se pondrá en contacto contigo pronto.";
        
        return $message;
    }
}