<?php

namespace App\Http\Middleware;

use App\Services\CajaTasaAutomationService;
use App\Services\CajaTasaConfigService;
use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;

class CheckCajaTasa
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next): Response
    {
      
        try {
            // Verificar si existe tasa del día (con cache para mejorar rendimiento)
            $tasaExiste = Cache::remember('tasa_hoy_existe', 300, function () {
                return \App\Models\ExchangeRate::whereDate('date', today())->exists();
            });

            if ($tasaExiste) {
                // Cerrar cajas antiguas abiertas (solo una vez por día)
                $this->cerrarCajasAntiguas();
                
                // Verificar y crear caja si es necesario (solo para el usuario actual)
                $this->verificarCajaUsuarioActual();
                
            } else {
                // Intentar actualizar tasa si es el período de verificación
                
                if (CajaTasaConfigService::enPeriodoVerificacionTasa()) {
                    $this->actualizarTasaSiNecesario();
                }
            }

        } catch (\Exception $e) {
            // Log del error pero no interrumpir el flujo del usuario
            Log::error('Error en CheckCajaTasa middleware', [
                'error' => $e->getMessage(),
                'user_id' => auth()->id(),
                'url' => $request->url()
            ]);
        }

        return $next($request);
    }

    /**
     * Cerrar cajas antiguas que estén abiertas
     */
    private function cerrarCajasAntiguas(): void
    {
        $cacheKey = 'cajas_antiguas_cerradas_' . today()->format('Y-m-d');
        
        if (!Cache::has($cacheKey)) {
            $cajas = \App\Models\Caja::where('estado', 'abierta')
                ->whereDate('created_at', '<', today())
                ->get();

            foreach ($cajas as $caja) {
                try {
                    $caja->cerrar('Cierre automático por antigüedad');
                } catch (\Exception $e) {
                    Log::error('Error al cerrar caja antigua', ['caja_id' => $caja->id]);
                }
            }
            
            // Marcar que ya se ejecutó hoy
            Cache::put($cacheKey, true, 86400); // 24 horas
        }
    }

    /**
     * Verificar y crear caja para el usuario actual
     */
    private function verificarCajaUsuarioActual(): void
    {
        $user = auth()->user();
        $empresaId = $user->empresa_id ?? 1;
        $sucursalId = $user->sucursal_id ?? 1;

        // Cache por usuario para evitar múltiples verificaciones
        $cacheKey = "caja_verificada_usuario_{$user->id}_" . today()->format('Y-m-d');
        
        if (!Cache::has($cacheKey)) {
            // Verificar si ya existe caja abierta
            $cajaAbierta = \App\Models\Caja::where('empresa_id', $empresaId)
                ->where('sucursal_id', $sucursalId)
                ->where('estado', 'abierta')
                ->first();

            if (!$cajaAbierta) {
                // Verificar si existe caja del día
                $cajaHoy = \App\Models\Caja::where('empresa_id', $empresaId)
                    ->where('sucursal_id', $sucursalId)
                    ->whereDate('fecha', today())
                    ->first();

                if ($cajaHoy) {
                    // Crear corte de caja
                    \App\Models\Caja::crearCorte(
                        $empresaId,
                        $sucursalId,
                        $cajaHoy->monto_final,
                        'Corte automático',
                        $user->id
                    );
                } else {
                    // Crear primera caja del día
                    \App\Models\Caja::crearCajaDiaria(
                        $empresaId,
                        $sucursalId,
                        0,
                        'Apertura automática',
                        $user->id
                    );
                }
            }
            
            // Marcar que ya se verificó para este usuario hoy
            Cache::put($cacheKey, true, 3600); // 1 hora
        }
    }

    /**
     * Actualizar tasa si es necesario
     */
    private function actualizarTasaSiNecesario(): void
    {
        try {
            $service = new \App\Services\ExchangeRateService();
            $success = $service->fetchAndStoreRates();

            if ($success) {
                $todayRate = \App\Models\ExchangeRate::getTodayRate();
                if ($todayRate) {
                    session()->flash('success', "Tasa actualizada: USD = {$todayRate->usd_rate} Bs. (Fuente: {$todayRate->source})");
                }
            } else {
                session()->flash('error', 'No se pudo obtener la tasa. Verifique la conexión a internet.');
            }
        } catch (\Exception $e) {
            session()->flash('error', 'Error al actualizar tasa: ' . $e->getMessage());
        }
    }
}