<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\MensajePendiente;

class VerEstadoMensajesPendientes extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'whatsapp:estado-pendientes 
                            {--detalle : Mostrar detalles de cada mensaje}
                            {--tipo= : Filtrar por tipo de mensaje}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Ver el estado de los mensajes de WhatsApp pendientes';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('📊 Estado de Mensajes de WhatsApp Pendientes');
        $this->info('============================================');
        
        $detalle = $this->option('detalle');
        $tipo = $this->option('tipo');
        
        // Estadísticas generales
        $totalPendientes = MensajePendiente::pendientes()->count();
        $totalEnviados = MensajePendiente::where('estado', 'enviado')->count();
        $totalFallidos = MensajePendiente::where('estado', 'fallido')->count();
        
        $this->info("📱 Total pendientes: {$totalPendientes}");
        $this->info("✅ Total enviados: {$totalEnviados}");
        $this->info("❌ Total fallidos: {$totalFallidos}");
        $this->newLine();
        
        // Por tipo
        $this->info('📋 Por tipo de mensaje:');
        $tipos = MensajePendiente::selectRaw('tipo, COUNT(*) as total')
            ->groupBy('tipo')
            ->orderBy('total', 'desc')
            ->get();
            
        if ($tipos->isEmpty()) {
            $this->info('No hay mensajes registrados.');
            return Command::SUCCESS;
        }
        
        foreach ($tipos as $tipoEstadistica) {
            $pendientes = MensajePendiente::pendientes()->porTipo($tipoEstadistica->tipo)->count();
            $this->info("  {$tipoEstadistica->tipo}: {$tipoEstadistica->total} ({$pendientes} pendientes)");
        }
        $this->newLine();
        
        // Mensajes pendientes actuales
        $query = MensajePendiente::pendientes();
        
        if ($tipo) {
            $query->porTipo($tipo);
        }
        
        $mensajesPendientes = $query->orderBy('created_at', 'asc')->get();
        
        if ($mensajesPendientes->isEmpty()) {
            $this->info('✅ No hay mensajes pendientes.');
            return Command::SUCCESS;
        }
        
        $this->info("📨 Mensajes pendientes actuales: {$mensajesPendientes->count()}");
        $this->newLine();
        
        if ($detalle) {
            $this->info('📄 Detalles de mensajes pendientes:');
            $this->info('=====================================');
            
            foreach ($mensajesPendientes as $mensaje) {
                $this->info("ID: {$mensaje->id}");
                $this->info("Tipo: {$mensaje->tipo}");
                $this->info("Teléfono: {$mensaje->telefono}");
                $this->info("Intentos: {$mensaje->intentos}");
                $this->info("Creado: {$mensaje->created_at->format('d/m/Y H:i:s')}");
                $this->info("Último intento: " . ($mensaje->ultimo_intento ? $mensaje->ultimo_intento->format('d/m/Y H:i:s') : 'Nunca'));
                
                if ($mensaje->error) {
                    $this->error("Error: {$mensaje->error}");
                }
                
                // Mostrar primeras 100 caracteres del mensaje
                $mensajeCorto = substr($mensaje->mensaje, 0, 100);
                if (strlen($mensaje->mensaje) > 100) {
                    $mensajeCorto .= '...';
                }
                $this->info("Mensaje: {$mensajeCorto}");
                $this->info('----------------------------------------');
            }
        } else {
            // Vista resumida
            $headers = ['ID', 'Tipo', 'Teléfono', 'Intentos', 'Creado'];
            $rows = [];
            
            foreach ($mensajesPendientes as $mensaje) {
                $rows[] = [
                    $mensaje->id,
                    $mensaje->tipo,
                    $mensaje->telefono,
                    $mensaje->intentos,
                    $mensaje->created_at->format('d/m/Y H:i')
                ];
            }
            
            $this->table($headers, $rows);
        }
        
        // Sugerencias
        $this->newLine();
        $this->info('💡 Sugerencias:');
        $this->info('  • Para procesar mensajes pendientes: php artisan whatsapp:procesar-pendientes');
        $this->info('  • Para ver detalles: php artisan whatsapp:estado-pendientes --detalle');
        $this->info('  • Para filtrar por tipo: php artisan whatsapp:estado-pendientes --tipo=asignacion_empleado');
        
        return Command::SUCCESS;
    }
}