<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use Illuminate\Support\Str;

class SetupApiCommand extends Command
{
    protected $signature = 'setup:api {--port=8001 : Puerto para la API}';
    protected $description = 'Configura e inicia la API con PM2, genera tokens y actualiza .env';

    public function handle()
    {
        $port = $this->option('port');
        
        $this->info('🚀 Configurando API del Sistema de Pedidos...');
        
        // 1. Verificar e instalar PM2
        $this->checkAndInstallPM2();
        
        // 2. Generar tokens para usuarios admin
        $this->generateApiTokens();
        
        // 3. Crear archivo ecosystem.config.js para PM2
        $this->createEcosystemConfig($port);
        
        // 4. Iniciar API con PM2
        $this->startApiWithPM2();
        
        $this->info('✅ API configurada y ejecutándose en puerto ' . $port);
        $this->info('📋 Tokens API agregados al archivo .env');
        $this->info('🔧 Usa "pm2 status" para ver el estado de la API');
    }

    private function checkAndInstallPM2()
    {
        $this->info('📦 Verificando PM2...');
        
        // Verificar si PM2 está instalado
        $pm2Check = shell_exec('pm2 --version 2>/dev/null');
        
        if (!$pm2Check) {
            $this->warn('PM2 no está instalado. Instalando...');
            
            // Instalar PM2 globalmente
            $installResult = shell_exec('npm install -g pm2 2>&1');
            
            if (strpos($installResult, 'error') !== false) {
                $this->error('Error instalando PM2: ' . $installResult);
                return false;
            }
            
            $this->info('✅ PM2 instalado correctamente');
        } else {
            $this->info('✅ PM2 ya está instalado (v' . trim($pm2Check) . ')');
        }
        
        return true;
    }

    private function generateApiTokens()
    {
        $this->info('🔑 Generando tokens API...');
        
        // Buscar usuarios admin y super admin
        $adminUsers = User::whereHas('roles', function($query) {
            $query->whereIn('name', ['Administrador', 'Super Administrador']);
        })->get();
        
        if ($adminUsers->isEmpty()) {
            $this->warn('No se encontraron usuarios con roles de administrador');
            return;
        }
        
        $tokens = [];
        
        foreach ($adminUsers as $user) {
            // Eliminar tokens existentes
            $user->tokens()->delete();
            
            // Crear nuevo token
            $token = $user->createToken('API Token - ' . $user->name)->plainTextToken;
            $tokens[$user->email] = $token;
            
            $this->info("Token generado para: {$user->name} ({$user->email})");
        }
        
        // Actualizar .env con tokens y URL de API
        $this->updateEnvWithTokens($tokens, $port);
    }

    private function updateEnvWithTokens($tokens, $port)
    {
        $envPath = base_path('.env');
        $envContent = file_get_contents($envPath);
        
        // Configurar URL de API
        $apiUrl = "http://localhost:{$port}";
        
        // Agregar configuración de API al .env
        $apiSection = "\n# API Configuration\n";
        $apiSection .= "API_URL={$apiUrl}\n";
        $apiSection .= "API_PORT={$port}\n";
        
        // Agregar tokens al .env
        $apiSection .= "\n# API Tokens\n";
        foreach ($tokens as $email => $token) {
            $key = 'API_TOKEN_' . strtoupper(str_replace(['@', '.', '-'], '_', $email));
            $apiSection .= "{$key}={$token}\n";
        }
        
        // Remover sección anterior si existe
        $envContent = preg_replace('/\n# API Configuration\n.*?(?=\n[A-Z_]+=|\n$)/s', '', $envContent);
        
        // Agregar nueva sección
        $envContent .= $apiSection;
        
        file_put_contents($envPath, $envContent);
        
        $this->info('✅ Configuración de API y tokens agregados al archivo .env');
        $this->info("📡 API URL: {$apiUrl}");
    }

    private function createEcosystemConfig($port)
    {
        $this->info('📝 Creando configuración PM2...');
        
        $config = [
            'apps' => [
                [
                    'name' => 'pedidos-api',
                    'script' => 'artisan',
                    'args' => 'serve --host=0.0.0.0 --port=' . $port,
                    'interpreter' => 'php',
                    'cwd' => base_path(),
                    'instances' => 1,
                    'autorestart' => true,
                    'watch' => false,
                    'max_memory_restart' => '1G',
                    'env' => [
                        'APP_ENV' => 'production',
                        'NODE_ENV' => 'production'
                    ],
                    'error_file' => base_path('storage/logs/pm2-error.log'),
                    'out_file' => base_path('storage/logs/pm2-out.log'),
                    'log_file' => base_path('storage/logs/pm2-combined.log'),
                    'time' => true
                ]
            ]
        ];
        
        $configPath = base_path('ecosystem.config.js');
        $jsConfig = 'module.exports = ' . json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES) . ';';
        
        file_put_contents($configPath, $jsConfig);
        
        $this->info('✅ Archivo ecosystem.config.js creado');
    }

    private function startApiWithPM2()
    {
        $this->info('🚀 Iniciando API con PM2...');
        
        // Detener proceso anterior si existe
        shell_exec('pm2 delete pedidos-api 2>/dev/null');
        
        // Iniciar con PM2
        $result = shell_exec('pm2 start ecosystem.config.js 2>&1');
        
        if (strpos($result, 'error') !== false) {
            $this->error('Error iniciando API: ' . $result);
            return false;
        }
        
        // Guardar configuración PM2
        shell_exec('pm2 save');
        
        // Configurar startup (opcional)
        $this->info('💡 Para auto-inicio en boot del sistema, ejecuta: pm2 startup');
        
        $this->info('✅ API iniciada correctamente');
        
        // Mostrar status
        $status = shell_exec('pm2 status');
        $this->line($status);
        
        return true;
    }
}