<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Database\Seeders\ProductosSeeder;
use Illuminate\Support\Facades\Artisan;

class SeedProductosCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'db:seed:productos
                            {--fresh : Limpiar la tabla de productos antes de sembrar}
                            {--count=10 : Número de productos a crear}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Crear productos de ejemplo en la base de datos';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Iniciando creación de productos de ejemplo...');

        // Verificar si se debe limpiar la tabla
        if ($this->option('fresh')) {
            $this->warn('🧹 Limpiando tabla de productos...');
            \App\Models\Producto::truncate();
            $this->info('✅ Tabla de productos limpiada');
        }

        try {
            // Ejecutar el seeder
            $seeder = new ProductosSeeder();
            $seeder->run();

            // Mostrar estadísticas
            $totalProductos = \App\Models\Producto::count();
            $productosVenezuela = \App\Models\Producto::whereNotNull('precio_bs')->count();
            $empresa = \App\Models\Empresa::first();
            $esVenezuela = $empresa && $empresa->pais && $empresa->pais->codigo_iso2 === 'VE';

            $this->newLine();
            $this->info('📊 Estadísticas de productos:');
            $this->table(
                ['Métrica', 'Valor'],
                [
                    ['Total de productos', $totalProductos],
                    ['Productos con precio en Bs.', $productosVenezuela],
                    ['País de la empresa', $esVenezuela ? 'Venezuela (VE)' : 'Otro país'],
                    ['Precios en Bs. activos', $esVenezuela ? 'Sí' : 'No'],
                ]
            );

            $this->newLine();
            $this->info('✅ Productos creados exitosamente!');

            if ($esVenezuela) {
                $this->info('📍 Se detectó empresa venezolana - Se incluyeron precios en bolívares.');
            }

            // Mostrar algunos productos de ejemplo
            $this->newLine();
            $this->info('📝 Ejemplos de productos creados:');
            $productosEjemplo = \App\Models\Producto::with(['categoria', 'marca'])
                ->limit(5)
                ->get();

            foreach ($productosEjemplo as $producto) {
                $precioInfo = $producto->precio_bs && $esVenezuela
                    ? "\${$producto->price} / Bs.{$producto->precio_bs}"
                    : "\${$producto->price}";

                $this->line("  • {$producto->name} - {$precioInfo} - Stock: {$producto->quantity}");
            }

        } catch (\Exception $e) {
            $this->error('❌ Error al crear productos: ' . $e->getMessage());
            $this->error('Asegúrese de tener al menos una empresa creada y configurada.');
            return 1;
        }

        return 0;
    }
}
