<?php

namespace App\Console\Commands;

use App\Models\Caja;
use App\Models\ExchangeRate;
use App\Services\CajaTasaAutomationService;
use App\Services\CajaTasaConfigService;
use Illuminate\Console\Command;
use Carbon\Carbon;

class ProbarCajaTasaSistema extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cajatasa:probar-sistema 
                            {--modo=simulacion : Modo de ejecución (simulacion|real)}
                            {--empresa= : ID de empresa específica}
                            {--sucursal= : ID de sucursal específica}
                            {--hora= : Hora para simular (formato H:i)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Probar el sistema completo de cajas y tasas en modo simulación o real';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('🧪 INICIANDO PRUEBA DEL SISTEMA DE CAJAS Y TASAS');
        $this->newLine();

        $modo = $this->option('modo');
        $horaSimulada = $this->option('hora');
        $empresaId = $this->option('empresa');
        $sucursalId = $this->option('sucursal');

        if ($horaSimulada) {
            Carbon::setTestNow(Carbon::createFromFormat('H:i', $horaSimulada));
            $this->warn("⏰ Simulando hora: " . Carbon::now()->format('H:i'));
        }

        if ($modo === 'real') {
            $this->warn('⚠️  Ejecutando en modo REAL - Se realizarán cambios en la base de datos');
            if (!$this->confirm('¿Estás seguro de ejecutar en modo real?')) {
                return Command::SUCCESS;
            }
        } else {
            $this->info('🔍 Ejecutando en modo SIMULACIÓN - Solo se mostrarán los resultados esperados');
        }

        $this->newLine();

        // Prueba 1: Configuración de horarios
        $this->probarConfiguracionHorarios();

        // Prueba 2: Estado de tasas
        $this->probarEstadoTasas($modo);

        // Prueba 3: Estado de cajas
        $this->probarEstadoCajas($empresaId, $sucursalId, $modo);

        // Prueba 4: Automatización
        if ($modo === 'real') {
            $this->probarAutomatizacionReal($empresaId, $sucursalId);
        } else {
            $this->probarAutomatizacionSimulacion($empresaId, $sucursalId);
        }

        // Prueba 5: Integridad de datos
        $this->probarIntegridadDatos();

        $this->newLine();
        $this->info('✅ Prueba del sistema completada');

        if ($horaSimulada) {
            Carbon::setTestNow(); // Restaurar hora real
        }

        return Command::SUCCESS;
    }

    private function probarConfiguracionHorarios(): void
    {
        $this->info('📅 PROBANDO CONFIGURACIÓN DE HORARIOS:');
        
        $config = CajaTasaConfigService::getConfiguracionActual();
        
        $this->line('Horarios de corte de caja: ' . implode(', ', CajaTasaConfigService::CORTE_CAJA_HORARIOS));
        $this->line('Período verificación tasa: ' . CajaTasaConfigService::TASA_VERIFICACION_INICIO . ' - ' . CajaTasaConfigService::TASA_VERIFICACION_FIN);
        $this->line('Actualización automática tasa: ' . CajaTasaConfigService::TASA_ACTUALIZACION_AUTO);
        
        $this->line('Estado actual:');
        $this->line('  - En verificación tasa: ' . ($config['en_verificacion_tasa'] ? '✅ SÍ' : '❌ NO'));
        $this->line('  - Es hora actualización tasa: ' . ($config['es_actualizacion_tasa'] ? '✅ SÍ' : '❌ NO'));
        $this->line('  - Es hora corte caja: ' . ($config['hora_corte_caja'] ? '✅ ' . $config['hora_corte_caja'] : '❌ NO'));
        
        $proximoCorte = CajaTasaConfigService::getProximoCorteCaja();
        if ($proximoCorte) {
            $this->line('  - Próximo corte: ' . $proximoCorte->format('H:i') . ' (en ' . now()->diffForHumans($proximoCorte, true) . ')');
        }
        
        $this->newLine();
    }

    private function probarEstadoTasas(string $modo): void
    {
        $this->info('💰 PROBANDO ESTADO DE TASAS:');
        
        $tasaHoy = ExchangeRate::getTodayRate();
        
        if ($tasaHoy) {
            $this->line('✅ Tasa del día registrada:');
            $this->line("  - USD: {$tasaHoy->usd_rate} Bs.");
            $this->line("  - EUR: {$tasaHoy->eur_rate} Bs.");
            $this->line("  - Fuente: {$tasaHoy->source}");
            $this->line("  - Actualizada: {$tasaHoy->fetch_time}");
            
            $horasActualizacion = Carbon::parse($tasaHoy->fetch_time)->diffInHours(now());
            if ($horasActualizacion > 6) {
                $this->warn("  ⚠️  Tasa desactualizada (hace {$horasActualizacion} horas)");
            }
        } else {
            $this->warn('❌ No hay tasa registrada para hoy');
            
            if ($modo === 'real' && CajaTasaConfigService::estaEnPeriodoVerificacionTasa()) {
                $this->info('  🔄 Intentando obtener tasa automáticamente...');
                $automationService = app(CajaTasaAutomationService::class);
                $resultado = $automationService->actualizarTasaAutomatica(true);
                
                if ($resultado['success']) {
                    $this->info('  ✅ Tasa obtenida exitosamente');
                } else {
                    $this->error('  ❌ Error: ' . $resultado['message']);
                }
            }
        }
        
        $this->newLine();
    }

    private function probarEstadoCajas(?int $empresaId, ?int $sucursalId, string $modo): void
    {
        $this->info('💼 PROBANDO ESTADO DE CAJAS:');
        
        $query = Caja::with(['empresa', 'sucursal']);
        
        if ($empresaId) {
            $query->where('empresa_id', $empresaId);
        }
        
        if ($sucursalId) {
            $query->where('sucursal_id', $sucursalId);
        }
        
        $cajasAbiertas = $query->where('estado', 'abierta')->get();
        $cajasHoy = (clone $query)->whereDate('fecha', today())->get();
        
        $this->line("Cajas abiertas: {$cajasAbiertas->count()}");
        $this->line("Cajas del día: {$cajasHoy->count()}");
        $this->line("Cajas cerradas hoy: {$cajasHoy->where('estado', 'cerrada')->count()}");
        
        if ($cajasAbiertas->isEmpty() && $modo === 'real') {
            $this->warn('⚠️  No hay cajas abiertas');
            
            if ($this->confirm('¿Desea crear cajas automáticamente?')) {
                $automationService = app(CajaTasaAutomationService::class);
                $resultado = $automationService->verificarCrearCajaDiaria(true);
                
                if ($resultado['success']) {
                    $this->info('✅ Cajas creadas exitosamente');
                } else {
                    $this->error('❌ Error: ' . $resultado['message']);
                }
            }
        }
        
        foreach ($cajasAbiertas as $caja) {
            $this->line("  - {$caja->empresa->nombre} / {$caja->sucursal->nombre}: {$caja->estado}");
        }
        
        $this->newLine();
    }

    private function probarAutomatizacionSimulacion(?int $empresaId, ?int $sucursalId): void
    {
        $this->info('🤖 SIMULACIÓN DE AUTOMATIZACIÓN:');
        
        $automationService = app(CajaTasaAutomationService::class);
        
        // Simular verificación de cajas
        $this->line('📋 Simulando verificación de cajas diarias...');
        $resultado = $automationService->verificarCrearCajaDiaria(false);
        $this->line("  Resultado: {$resultado['message']}");
        
        // Simular actualización de tasa
        if (CajaTasaConfigService::estaEnPeriodoVerificacionTasa()) {
            $this->line('💰 Simulando actualización de tasa...');
            $resultado = $automationService->actualizarTasaAutomatica(false);
            $this->line("  Resultado: {$resultado['message']}");
        }
        
        // Simular corte de caja
        if (CajaTasaConfigService::esHoraCorteCaja()) {
            $this->line('✂️ Simulando corte de caja automático...');
            $resultado = $automationService->ejecutarCorteCajaAutomatico($empresaId, $sucursalId, false);
            $this->line("  Resultado: {$resultado['message']}");
        }
        
        $this->newLine();
    }

    private function probarAutomatizacionReal(?int $empresaId, ?int $sucursalId): void
    {
        $this->info('🤖 EJECUTANDO AUTOMATIZACIÓN REAL:');
        
        $automationService = app(CajaTasaAutomationService::class);
        
        // Verificar y crear cajas
        $this->line('📋 Verificando cajas diarias...');
        $resultado = $automationService->verificarCrearCajaDiaria(true);
        $this->line("  Resultado: {$resultado['message']}");
        
        // Actualizar tasa si es necesario
        if (CajaTasaConfigService::estaEnPeriodoVerificacionTasa()) {
            $this->line('💰 Actualizando tasa automáticamente...');
            $resultado = $automationService->actualizarTasaAutomatica(true);
            $this->line("  Resultado: {$resultado['message']}");
        }
        
        // Corte de caja si es hora
        if (CajaTasaConfigService::esHoraCorteCaja()) {
            $this->line('✂️ Ejecutando corte de caja automático...');
            $resultado = $automationService->ejecutarCorteCajaAutomatico($empresaId, $sucursalId, true);
            $this->line("  Resultado: {$resultado['message']}");
        }
        
        $this->newLine();
    }

    private function probarIntegridadDatos(): void
    {
        $this->info('🔍 PROBANDO INTEGRIDAD DE DATOS:');
        
        // Verificar cajas sin empresa o sucursal
        $cajasInvalidas = Caja::whereNull('empresa_id')
            ->orWhereNull('sucursal_id')
            ->orWhereNull('user_id')
            ->count();
        
        if ($cajasInvalidas > 0) {
            $this->warn("⚠️  Hay {$cajasInvalidas} cajas con datos incompletos");
        } else {
            $this->line('✅ Todas las cajas tienen datos completos');
        }
        
        // Verificar tasas duplicadas
        $tasasDuplicadas = ExchangeRate::select('date')
            ->groupBy('date')
            ->havingRaw('COUNT(*) > 1')
            ->count();
        
        if ($tasasDuplicadas > 0) {
            $this->warn("⚠️  Hay {$tasasDuplicadas} fechas con tasas duplicadas");
        } else {
            $this->line('✅ No hay tasas duplicadas');
        }
        
        // Verificar cajas abiertas antiguas
        $cajasAntiguas = Caja::where('estado', 'abierta')
            ->where('fecha', '<', now()->subDays(2)->format('Y-m-d'))
            ->count();
        
        if ($cajasAntiguas > 0) {
            $this->warn("⚠️  Hay {$cajasAntiguas} cajas abiertas de hace más de 2 días");
        } else {
            $this->line('✅ No hay cajas abiertas antiguas');
        }
        
        $this->newLine();
    }
}