<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\WhatsAppService;

class DebugTestConnection extends Command
{
    protected $signature = 'whatsapp:debug-test-connection';
    protected $description = 'Depurar el método testConnection del WhatsAppService';

    public function handle()
    {
        $this->info('=== Depurando testConnection() ===');
        
        $whatsappService = new WhatsAppService();
        
        // Verificar configuración
        $this->info('1. Configuración del servicio:');
        $reflection = new \ReflectionClass($whatsappService);
        $apiUrlProperty = $reflection->getProperty('apiUrl');
        $apiUrlProperty->setAccessible(true);
        $this->info('   API URL: ' . $apiUrlProperty->getValue($whatsappService));
        
        // Verificar JWT
        $jwtTokenProperty = $reflection->getProperty('jwtToken');
        $jwtTokenProperty->setAccessible(true);
        $jwtToken = $jwtTokenProperty->getValue($whatsappService);
        $this->info('   JWT Token: ' . ($jwtToken ? 'Presente' : 'No configurado'));
        
        // Ejecutar testConnection con detalles
        $this->info('2. Ejecutando testConnection()...');
        
        // Crear una versión debug del testConnection
        $result = $this->debugTestConnection($whatsappService);
        
        $this->info('3. Resultado:');
        $this->info('   Success: ' . ($result['success'] ? 'true' : 'false'));
        $this->info('   Message: ' . ($result['message'] ?? 'N/A'));
        $this->info('   Error: ' . ($result['error'] ?? 'N/A'));
        
        if (isset($result['http_code'])) {
            $this->info('   HTTP Code: ' . $result['http_code']);
        }
        
        if (isset($result['response'])) {
            $this->info('   Response: ' . substr($result['response'], 0, 200));
        }
        
        $this->info('=== Fin de la depuración ===');
    }
    
    private function debugTestConnection($whatsappService)
    {
        try {
            $curl = curl_init();
            
            // Obtener configuración del servicio
            $reflection = new \ReflectionClass($whatsappService);
            $apiUrlProperty = $reflection->getProperty('apiUrl');
            $apiUrlProperty->setAccessible(true);
            $apiUrl = $apiUrlProperty->getValue($whatsappService);
            
            $jwtTokenProperty = $reflection->getProperty('jwtToken');
            $jwtTokenProperty->setAccessible(true);
            $jwtToken = $jwtTokenProperty->getValue($whatsappService);
            
            // Construir headers
            $headers = [
                'Content-Type: application/json',
                'Accept: application/json'
            ];
            if ($jwtToken) {
                $headers[] = 'Authorization: Bearer ' . $jwtToken;
            }
            
            $url = $apiUrl . '/api/whatsapp/send-message';
            
            $this->info('   URL probada: ' . $url);
            $this->info('   Headers: ' . json_encode($headers));
            
            curl_setopt_array($curl, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_CUSTOMREQUEST => 'GET',
                CURLOPT_TIMEOUT => 5,
                CURLOPT_HTTPHEADER => $headers,
                CURLOPT_HEADER => true, // Incluir headers en la respuesta
                CURLOPT_VERBOSE => true, // Para debugging
            ]);
            
            $response = curl_exec($curl);
            $err = curl_error($curl);
            $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
            $headerSize = curl_getinfo($curl, CURLINFO_HEADER_SIZE);
            
            curl_close($curl);
            
            $result = [
                'http_code' => $httpCode,
                'error' => $err,
                'response' => $response,
                'header_size' => $headerSize
            ];
            
            if ($err) {
                $result['success'] = false;
                $result['error'] = 'API no disponible';
                return $result;
            }
            
            // Separar headers del body
            $headers = substr($response, 0, $headerSize);
            $body = substr($response, $headerSize);
            
            $result['headers'] = $headers;
            $result['body'] = $body;
            
            // Si el endpoint no existe (404) o hay error de autenticación (401)
            if ($httpCode === 404 || $httpCode === 401) {
                $result['success'] = false;
                $result['error'] = 'API no disponible o requiere autenticación';
                return $result;
            }
            
            if ($httpCode >= 400) {
                $result['success'] = false;
                $result['error'] = 'API no disponible';
                return $result;
            }
            
            $result['success'] = true;
            $result['message'] = 'API disponible';
            return $result;
            
        } catch (\Exception $e) {
            return [
                'success' => false,
                'error' => 'API no disponible',
                'exception' => $e->getMessage()
            ];
        }
    }
}