<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use RecursiveDirectoryIterator;
use RecursiveIteratorIterator;
use RegexIterator;

class ConvertIconsToRemixCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'icons:convert-to-remix {--dry-run : Ejecutar sin realizar cambios reales} {--check-only : Solo verificar iconos} {--fix-common : Corregir iconos comunes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Convertir iconos al formato Remix Icon (ri ri-*) en todas las vistas';

    /**
     * Iconos comunes que deberían usarse en lugar de ri-menu-line y ri-circle-line
     */
    protected $commonIcons = [
        'dashboard' => 'dashboard-line',
        'pedidos' => 'shopping-cart-line',
        'productos' => 'box-line',
        'clientes' => 'group-line',
        'usuarios' => 'user-line',
        'configuración' => 'settings-line',
        'reportes' => 'bar-chart-line',
        'pagos' => 'money-dollar-circle-line',
        'inventario' => 'archive-line',
        'estadísticas' => 'pie-chart-line',
        'notificaciones' => 'notification-line',
        'mensajes' => 'message-line',
        'calendario' => 'calendar-line',
        'archivos' => 'file-line',
        'descargas' => 'download-line',
        'subir' => 'upload-line',
        'buscar' => 'search-line',
        'editar' => 'edit-line',
        'eliminar' => 'delete-bin-line',
        'ver' => 'eye-line',
        'imprimir' => 'printer-line',
        'exportar' => 'share-forward-line',
        'importar' => 'login-box-line',
        'ayuda' => 'question-line',
        'información' => 'information-line',
        'alerta' => 'alert-line',
        'éxito' => 'checkbox-circle-line',
        'error' => 'close-circle-line',
        'cerrar' => 'close-line',
        'añadir' => 'add-line',
        'guardar' => 'save-line',
        'cancelar' => 'close-line',
        'actualizar' => 'refresh-line',
        'filtrar' => 'filter-line',
        'ordenar' => 'arrow-up-down-line',
    ];

    /**
     * Execute the console command.
     *
     * @return int
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');
        $checkOnly = $this->option('check-only');
        $fixCommon = $this->option('fix-common');
        
        if ($checkOnly) {
            $this->info('=== Verificación de Iconos Remix ===');
            $this->line('Analizando vistas para identificar iconos...');
        } else if ($fixCommon) {
            $this->info('=== Corrección de Iconos Comunes ===');
            if ($dryRun) {
                $this->line('[MODO SIMULACIÓN] No se realizarán cambios reales');
            }
            $this->line('Corrigiendo iconos comunes...');
        } else {
            $this->info('=== Conversión de Iconos a Remix Icon ===');
            if ($dryRun) {
                $this->line('[MODO SIMULACIÓN] No se realizarán cambios reales');
            }
            $this->line('Analizando vistas y actualizando iconos...');
        }
        $this->newLine();

        // Directorios a escanear recursivamente
        $directories = [
            'resources/views',
            'app/View/Components'
        ];

        $totalFilesProcessed = 0;
        $totalReplacements = 0;
        $filesModified = [];
        $iconIssues = [];
        $errors = [];

        foreach ($directories as $directory) {
            $fullPath = base_path($directory);
            if (!is_dir($fullPath)) {
                $this->warn("Directorio no encontrado: $directory");
                continue;
            }
            
            try {
                $iterator = new RecursiveIteratorIterator(new RecursiveDirectoryIterator($fullPath));
                $bladeFiles = new RegexIterator($iterator, '/\.blade\.php$/i', RegexIterator::GET_MATCH);
                
                foreach ($bladeFiles as $file) {
                    $filePath = $file[0];
                    
                    // Verificar que el archivo sea un archivo real (no directorio) y exista
                    if (!is_file($filePath) || !file_exists($filePath) || !is_readable($filePath)) {
                        continue; // Saltar archivos no válidos
                    }
                    
                    try {
                        $content = file_get_contents($filePath);
                        if ($content === false) {
                            $this->warn("No se pudo leer el contenido de: " . str_replace(base_path() . '/', '', $filePath));
                            continue;
                        }
                        
                        $originalContent = $content;
                        
                        // Si estamos corrigiendo iconos comunes
                        if ($fixCommon) {
                            // Buscar iconos genéricos y reemplazarlos con iconos más específicos
                            foreach ($this->commonIcons as $keyword => $iconName) {
                                // Buscar elementos <div> con texto relacionado y el icono genérico después
                                $content = preg_replace_callback(
                                    '/(<div[^>]*>\s*' . preg_quote(ucfirst($keyword), '/') . '\s*<\/div>\s*<\s*i\s+[^>]*class=)(["\'])([^"\']*ri-(menu|circle)-line[^"\']*)(\2[^>]*>)/i',
                                    function($matches) use ($iconName, &$totalReplacements) {
                                        $prefix = $matches[1];
                                        $quote = $matches[2];
                                        $classes = $matches[3];
                                        $suffix = $matches[5];
                                        
                                        // Reemplazar el icono genérico con el específico
                                        $newClasses = preg_replace('/ri-(menu|circle)-line/', 'ri-' . $iconName, $classes);
                                        $totalReplacements++;
                                        
                                        return "{$prefix}{$quote}{$newClasses}{$quote}{$suffix}";
                                    },
                                    $content
                                );
                                
                                // Buscar en atributos href o texto relacionado
                                $content = preg_replace_callback(
                                    '/(<a[^>]*(href|title)=["\'][^"\']*' . preg_quote($keyword, '/') . '[^"\']*["\'][^>]*>\s*<\s*i\s+[^>]*class=)(["\'])([^"\']*ri-(menu|circle)-line[^"\']*)(\3[^>]*>)/i',
                                    function($matches) use ($iconName, &$totalReplacements) {
                                        $prefix = $matches[1];
                                        $quote = $matches[3];
                                        $classes = $matches[4];
                                        $suffix = $matches[6];
                                        
                                        // Reemplazar el icono genérico con el específico
                                        $newClasses = preg_replace('/ri-(menu|circle)-line/', 'ri-' . $iconName, $classes);
                                        $totalReplacements++;
                                        
                                        return "{$prefix}{$quote}{$newClasses}{$quote}{$suffix}";
                                    },
                                    $content
                                );
                            }
                        }
                        
                        // Buscar todas las etiquetas <i> con clases
                        preg_match_all('/<i\s+([^>]*class=)(["\'])([^"\']*)(\2[^>]*)>/i', $content, $iconMatches, PREG_SET_ORDER);
                        
                        foreach ($iconMatches as $match) {
                            $fullTag = $match[0];
                            $beforeClass = $match[1];
                            $quote = $match[2];
                            $classes = $match[3];
                            $afterClass = $match[4];
                            
                            // Verificar si es un icono Remix
                            if (preg_match('/\bri-[a-zA-Z0-9\-]+/', $classes)) {
                                // Verificar si sigue el formato correcto (ri ri-icon-name)
                                if (!preg_match('/^ri\s+ri-[a-zA-Z0-9\-]+/', $classes) && 
                                    !preg_match('/\sri\s+ri-[a-zA-Z0-9\-]+/', $classes)) {
                                    
                                    // Registrar el problema
                                    $relativePath = str_replace(base_path() . '/', '', $filePath);
                                    if (!isset($iconIssues[$relativePath])) {
                                        $iconIssues[$relativePath] = [];
                                    }
                                    $iconIssues[$relativePath][] = [
                                        'tag' => $fullTag,
                                        'classes' => $classes
                                    ];
                                    
                                    if (!$checkOnly && !$fixCommon) {
                                        // Corregir el formato
                                        // Si tiene una clase ri-* pero no comienza con "ri ", añadirlo
                                        if (preg_match('/^ri-[a-zA-Z0-9\-]+/', $classes)) {
                                            $newClasses = 'ri ' . $classes;
                                            $totalReplacements++;
                                            $content = str_replace($fullTag, "<i {$beforeClass}{$quote}{$newClasses}{$quote}{$afterClass}>", $content);
                                        } else if (preg_match('/ri-[a-zA-Z0-9\-]+/', $classes) && !preg_match('/^ri\s/', $classes)) {
                                            // Añadir 'ri' al principio si no está
                                            $newClasses = 'ri ' . $classes;
                                            $totalReplacements++;
                                            $content = str_replace($fullTag, "<i {$beforeClass}{$quote}{$newClasses}{$quote}{$afterClass}>", $content);
                                        }
                                    }
                                }
                            }
                        }
                        
                        // Guardar cambios si hubo modificaciones
                        if (($fixCommon || !$checkOnly) && $content !== $originalContent) {
                            if (!$dryRun) {
                                file_put_contents($filePath, $content);
                            }
                            $this->line("Procesado: " . str_replace(base_path() . '/', '', $filePath) . " (Correcciones: $totalReplacements)");
                            $totalFilesProcessed++;
                            $filesModified[] = $filePath;
                        }
                    } catch (\Exception $e) {
                        $this->error("Error procesando " . str_replace(base_path() . '/', '', $filePath) . ": " . $e->getMessage());
                    }
                }
            } catch (\Exception $e) {
                $this->error("Error accediendo al directorio {$directory}: " . $e->getMessage());
            }
        }

        if ($checkOnly) {
            if (count($iconIssues) > 0) {
                $this->info('=== Iconos con problemas encontrados ===');
                foreach ($iconIssues as $file => $issues) {
                    $this->line("Archivo: {$file}");
                    foreach ($issues as $issue) {
                        $this->line("  - Icono: {$issue['tag']}");
                        $this->line("    Clases: {$issue['classes']}");
                    }
                    $this->newLine();
                }
                $this->line('Total de archivos con problemas: ' . count($iconIssues));
                $this->line('Total de iconos con problemas: ' . array_sum(array_map('count', $iconIssues)));
                $this->newLine();
                $this->comment('Para corregir estos iconos, ejecuta: php artisan icons:convert-to-remix');
            } else {
                $this->info('No se encontraron iconos con problemas.');
                $this->line('Todos los iconos revisados tienen el formato correcto.');
            }
        } else if ($fixCommon) {
            $this->newLine();
            $this->info('=== Resumen de Corrección ===');
            $this->line("Archivos procesados: $totalFilesProcessed");
            $this->line("Correcciones realizadas: $totalReplacements");

            if ($dryRun && $totalReplacements > 0) {
                $this->newLine();
                $this->line('Archivos que serían modificados:');
                foreach ($filesModified as $file) {
                    $this->line("- " . str_replace(base_path() . '/', '', $file));
                }
                $this->newLine();
                $this->comment('Para aplicar estos cambios, ejecuta el comando sin la opción --dry-run');
            } else if (!$dryRun && $totalReplacements > 0) {
                $this->newLine();
                $this->info('¡Cambios aplicados exitosamente!');
            } else if ($totalReplacements == 0) {
                $this->newLine();
                $this->comment('No se encontraron iconos que necesiten corrección.');
            }
        } else {
            $this->newLine();
            $this->info('=== Resumen de Conversión ===');
            $this->line("Archivos procesados: $totalFilesProcessed");
            $this->line("Correcciones realizadas: $totalReplacements");

            if ($dryRun && $totalReplacements > 0) {
                $this->newLine();
                $this->line('Archivos que serían modificados:');
                foreach ($filesModified as $file) {
                    $this->line("- " . str_replace(base_path() . '/', '', $file));
                }
                $this->newLine();
                $this->comment('Para aplicar estos cambios, ejecuta el comando sin la opción --dry-run');
            } else if (!$dryRun && $totalReplacements > 0) {
                $this->newLine();
                $this->info('¡Cambios aplicados exitosamente!');
            } else if ($totalReplacements == 0) {
                $this->newLine();
                $this->comment('No se encontraron iconos que necesiten corrección.');
            }
        }

        $this->info('Conversión completada.');
        return Command::SUCCESS;
    }
}