<?php

namespace App\Console\Commands;

use App\Services\CajaTasaAutomationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class AutomatizarCajaTasa extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cajatasa:automatizar {--force : Forzar ejecución sin verificar horarios}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Ejecutar automatizaciones de cajas y tasas según horario configurado';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Iniciando automatización de cajas y tasas...');
        Log::info('Comando automatizar cajatasa iniciado');

        try {
            $service = new CajaTasaAutomationService();
            $force = $this->option('force');

            if ($force) {
                $this->warn('Ejecutando en modo FORZADO - sin verificar horarios');
            }

            $resultados = $service->ejecutarAutomatizaciones();

            // Mostrar resultados
            $this->mostrarResultados($resultados);

            // Registrar en log
            Log::info('Automatización completada', $resultados);

            $this->info('Automatización finalizada exitosamente.');
            return Command::SUCCESS;

        } catch (\Exception $e) {
            $this->error('Error en la automatización: ' . $e->getMessage());
            Log::error('Error en comando automatizar cajatasa', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Command::FAILURE;
        }
    }

    /**
     * Mostrar resultados en consola
     */
    private function mostrarResultados(array $resultados): void
    {
        if (isset($resultados['tasa'])) {
            $tasa = $resultados['tasa'];
            if ($tasa['exitoso']) {
                $this->info("✓ Tasa actualizada: USD = {$tasa['tasa']} Bs.");
            } else {
                $this->warn("⚠ Tasa: {$tasa['mensaje']}");
            }
        }

        if (isset($resultados['corte_caja'])) {
            $corte = $resultados['corte_caja'];
            if ($corte['exitoso']) {
                $this->info("✓ Corte de caja realizado: {$corte['mensaje']}");
                foreach ($corte['cortes_realizados'] as $corteRealizado) {
                    $this->line("  - {$corteRealizado['empresa']} / {$corteRealizado['sucursal']}: {$corteRealizado['monto_final']}$");
                }
            } else {
                $this->warn("⚠ Corte caja: {$corte['mensaje']}");
            }
        }

        if (isset($resultados['verificacion_cajas'])) {
            $verif = $resultados['verificacion_cajas'];
            if ($verif['exitoso'] && !empty($verif['cajas_creadas'])) {
                $this->info("✓ Cajas diarias creadas: {$verif['mensaje']}");
                foreach ($verif['cajas_creadas'] as $caja) {
                    $this->line("  - {$caja['empresa']} / {$caja['sucursal']}");
                }
            }
        }
    }
}